<?php

class Settings extends Cf_Controller
{

    function __construct()
    {
        parent::__construct();

        setCurrentApplication('settings');

        $this->load->library('form_validation');
        $this->load->library('ion_auth');
        $this->load->library('ldap_api');
        $this->load->helper('form');
        $this->load->helper(array(
            'url',
            'cf_directory_helper'
        ));
        $this->load->model('settings_rest_model');
        $this->settings_rest_model->setRestClient($this->getRestClient());
    }

    function index()
    {
        redirect('settings/preferences'); //default action
    }

    /**
     * @rbacName Preferences
     * @rbacGroup Settings
     * @rbacAlias settings.manage
     */
    function manage()
    {
        $requiredjs = [['widgets/notes.js']];
        $this->carabiner->js($requiredjs);

        try {
            $settingsValue = $this->settings_rest_model->get_app_settings();
            unset($settingsValue['ldapPasswordIsSet']);
        } catch (Exception $e) {
            // for now set the setting value as empty array whn error
            show_error($e->getMessage());
        }

        $this->form_validation->set_rules('mode', 'Authentication mode', 'xss_clean|trim');
        $this->form_validation->set_rules('rbac', 'Role based access control', 'xss_clean|trim|required');
        $this->form_validation->set_rules('bluehost_threshold_global', 'Blue host horizon',
            'callback_validate_bluehost_threshold');
        $this->form_validation->set_rules('hostsCollisionsThreshold', 'hostsCollisionsThreshold',
            'xss_clean|trim|required|greater_than[1]');
        $this->form_validation->set_rules('passwordComplexity', 'passwordComplexity',
            'xss_clean|trim|required|greater_than_equal_to[0]|less_than[5]');
        $this->form_validation->set_rules('minPasswordLength', 'minPasswordLength',
            'xss_clean|trim|required|greater_than_equal_to[8]');
        $this->form_validation->set_rules('passwordExpirationAfterResetHours', 'passwordExpirationAfterResetHours',
            'xss_clean|trim');
        $this->form_validation->set_error_delimiters('<span>', '</span><br/>');

        if ($this->form_validation->run($this) == true) {
            log_message('debug', "manage(): Mission Portal validation - OK");

            $returnData = $this->_updateData($settingsValue); //pass OLD data to compare new and old pass
            if (is_array($returnData) && $returnData['status']) {
                $this->session->set_flashdata('message', array(
                    'content' => $returnData['message'],
                    'type'    => 'alert-success'
                ));
                redirect('settings/preferences');
            } else {
                $feedback = array(
                    'type'    => 'alert-error',
                    'content' => $returnData['message']
                );
            }


        } else {
            log_message('debug', "manage(): Mission Portal validation - Failed");
            $feedback = validation_errors();
        }


        $bluehost_threshold_min = set_value('bluehost_threshold_global', $settingsValue['blueHostHorizon'] / 60);
        $mode                   = $settingsValue['ldapEnabled'] ? 'ldap' : 'internal';

        $form_data = array(
            'appemail'                  => set_value('appemail', $settingsValue['appemail']),
            'mode'                      => set_value('mode', $mode),
            'logLevel'                  => $settingsValue['logLevel'],
            'rbac'                      => set_value('rbac', $settingsValue['rbacEnabled']),
            'bluehost_threshold_global' => set_value('bluehost_threshold_global', $bluehost_threshold_min),
            'hostsCollisionsThreshold' => $settingsValue['hostsCollisionsThreshold'],
            'passwordComplexity' => $settingsValue['passwordComplexity'],
            'minPasswordLength' => $settingsValue['minPasswordLength'],
            'passwordExpirationAfterResetHours' => $settingsValue['passwordExpirationAfterResetHours']
        );

        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Preferences",
            'message'        => $feedback
        );

        $data = array_merge($form_data, $data);
        $this->template->load('template', 'appsetting/missionportalpref', $data);
    }

    /**
     * updates the setting data
     */
    function _updateData($currentSettings = array())
    {
        $returnData            = array();
        $returnData['status']  = false;
        $returnData['message'] = 'Data update failed';

        $blueHostHorizon = set_value('bluehost_threshold_global') * 60;

        $data                = array();
        $data['rbacEnabled'] = (bool)set_value('rbac');
        $data['logLevel']    = $this->input->post('api_log_level');
        $mode                = set_value('mode');

        $data['blueHostHorizon']       = $blueHostHorizon; // store it in seconds
        $data['hostsCollisionsThreshold'] = (int) $this->input->post('hostsCollisionsThreshold');
        $data['passwordComplexity'] = (int) $this->input->post('passwordComplexity');
        $data['minPasswordLength'] = (int) $this->input->post('minPasswordLength');
        $data['passwordExpirationAfterResetHours'] = (int) $this->input->post('passwordExpirationAfterResetHours');

        try {
            log_message('debug', "_updateData(): Trying to update data via REST ");
            $this->settings_rest_model->updateData($data);
            $returnData['status']  = true;
            $returnData['message'] = 'Data saved successfully.';
        } catch (Exception $e) {
            log_message('debug', "_updateData(): REST - Could not save your settings");
            $returnData['status']  = false;
            $returnData['message'] = "Could not save your settings. " . $e->getMessage();
        }

        return $returnData;
    }


    function required_valid_role($value)
    {
        if ($value == "select") {
            $this->form_validation->set_message('required_valid_role', 'A valid role is required.');

            return false;
        } else {
            return true;
        }
    }

    function validate_ldap_search($value)
    {
        $pattern = "/^\((.*)\)$/"; //  regex looks for start and end with parentheses

        if ($value &&!preg_match($pattern, $value)) {
            $this->form_validation->set_message('validate_ldap_search',
                'The search term should be enclosed in parentheses e.g \'(objectClass=inetOrgPerson)\'');

            return false;
        }

        return true;
    }

    function validate_bluehost_threshold($value)
    {
        $value = trim($value);
        if (empty($value)) {
            $this->form_validation->set_message('validate_bluehost_threshold',
                'Unreachable host threshold field is required.');

            return false;
        }

        if (!ctype_digit($value)) {
            $this->form_validation->set_message('validate_bluehost_threshold',
                'Unreachable host threshold field must contain a valid number of minutes.');

            return false;
        }

        if ($value < 5 || $value > 50400) {
            $this->form_validation->set_message('validate_bluehost_threshold',
                'Unreachable host threshold field must be in range of 5 minutes - 5 weeks (50400 minutes).');

            return false;
        }

        return true;
    }

    function __db_check()
    {
        if ($this->input->post('mode') && strtolower($this->input->post('mode')) == 'internal') {
            return '|required';
        } else {
            return '';
        }
    }

    function get_native_roles()
    {
        $roles    = $this->ion_auth->get_roles();
        $rolesarr = array();

        foreach ($roles as $role) {
            $rolesarr[$role['name']] = $role['name'];
        }
        $formData = form_dropdown('admin_role', $rolesarr);
        respond_ok($formData);
    }

    //for setting up usertime zone on fly.
    function setUserTimeZone()
    {
        $timezone = $this->input->post('userTimeZone');
        $timezone = $this->input->post('userTimeZone');
        $this->session->set_userdata('user_timezone', $timezone);
    }

    /**
     * @rbacName Host identifier
     * @rbacGroup Settings
     * @rbacAlias settings.systemIdentifier
     */
    function systemIdentifier($filter = 'sys')
    {
        $username = $this->session->userdata('username');
        $this->load->model('advancedreports_model');
        $this->load->config('hostidentifier');
        $this->advancedreports_model->setRestClient($this->getRestClient());

        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Host identifier"
        );
        // these variables will be execluded from the listing
        $includeVariables = config_item('host-identifier');
        if (empty($includeVariables)) {
            $includeVariables = array(
                'fqhost' => 'Fully qualified domain name'
            ); // Add one by default if no items are configured

        }

        $includeVariablesKeys = array_keys($includeVariables);
        $variableNames        = array();
        foreach ($includeVariablesKeys as $keys) {
            $keysArray        = explode('.', $keys);
            $variablesNames[] = $keysArray[1];
        }

        $in                   = "'" . implode("','", $variablesNames) . "'";
        $data['defaultList']  = $includeVariables;
        $data['exampleValue'] = array();

        $data['updated_host_identifier'] = $this->settings_rest_model->app_settings_get_item('hostIdentifier');
        // variable has prefix default. which we should delete, because updateHostIdentifier supports only scope and name
        if (!empty($data['updated_host_identifier'])) {
            $data['updated_host_identifier'] = str_replace('default.', '', $data['updated_host_identifier']);
        } else {
            $data['updated_host_identifier'] = 'sys.fqhost';
        }

        $query = sprintf('SELECT DISTINCT Bundle, VariableName, VariableValue FROM Variables');
        $query = $query . " WHERE VariableName IN ($in) AND hostkey in (SELECT hostkey from contexts where  ContextName = 'policy_server') ORDER BY VariableName";
        try {
            $reported     = $this->advancedreports_model->runQuery($username, $query);
            $data['vars'] = $reported['rows'];
            foreach ((array)$data['vars'] as $row) {
                $key                        = $row[0] . '.' . $row[1];
                $value                      = $row[2];
                $data['exampleValue'][$key] = $value;
            }
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), "Error getting variables for system identifier:: " . $e->getMessage());
            $data['vars'] = array();
        }
        $this->template->load('template', 'appsetting/systemidentifier', $data);
    }


    function about()
    {
        $this->load->model('license/license_model');
        $this->license_model->setRestClient($this->ion_auth->getRestClient());
        $data = array();

        try {
            $licenseDetails = $this->license_model->getLicenseDetails()->toArray();
            $data           = array(
                'title'          => $this->lang->line('mission_portal_title') . " - About CFEngine ",
                'details'        => $licenseDetails,
                'licenseGranted' => isset($licenseDetails['granted']) ? $licenseDetails['granted'] : '',

                'licenseOwner' => isset($licenseDetails['owner']) ? $licenseDetails['owner'] : '',
                'licenseType'  => isset($licenseDetails['licenseType']) ? $licenseDetails['licenseType'] : '',
                'expiry'       => isset($licenseDetails['expires']) ? $licenseDetails['expires'] : ''
            );
        } catch (Exception $e) {
            log_message('license error:' . $e->getMessage());
        }

        $this->template->load('template', 'appsetting/about', $data);
    }

    /**
     * @rbacName Host identifier
     * @rbacGroup Settings
     * @rbacAlias settings.systemIdentifier
     */
    function updateSystemIdentifierSetting()
    {
        if ($this->input->post('update')) {
            $username = $this->session->userdata('username');
            $scope    = $this->input->post('scope', true);
            $name     = $this->input->post('name', true);
            try {

                if (trim($scope) == '' && trim($name) == '') {
                    // reset the identifier
                    $data   = array('hostIdentifier' => '');
                    $result = $this->settings_rest_model->updateData($data);
                } else {
                    $hostIdentifier = 'default.' . $scope . '.' . $name;
                    $data           = array('hostIdentifier' => $hostIdentifier);
                    $result         = $this->settings_rest_model->updateData($data);
                }
                $this->output->set_status_header(200, 'Update successful');
                $array = array(
                    'message' => 'Sucessfully updated',
                    'type'    => 'success'
                );
                $this->output->set_output(json_encode($array));

            } catch (Exception $e) {
                $msg = $e->getMessage();
                $this->output->set_status_header(406, 'Update unsuccessful');
                $array = array(
                    'message' => $msg,
                    'type'    => 'error'
                );
                $this->output->set_output(json_encode($array));
            }
        }
    }

    /**
     * @rbacName Version control repository
     * @rbacGroup Settings
     * @rbacAlias settings.vcs
     */
    public function vcs()
    {
        $this->load->model('settings_model');
        $this->load->model('vcsSettingsModel');

        $feedback = array();

        if ($this->input->post('update')) {
            try {
                $feedback = $this->handleFormUploadForGitSettings();
            } catch (Exception $e) {
                $output   = $e->getMessage();
                $feedback = array(
                    'type'    => 'alert-error',
                    'content' => $output
                );
            }

            if ($feedback === true) {
                redirect('settings/vcs');
            }
        }

        try {
            $userSettings = $this->vcsSettingsModel->getSettings();
        } catch (Exception $e) {
            $userSettings = [];
        }


        $data      = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Version control repository",
            'message'        => $feedback,
            "hasSettings"    => element_object('GIT_URL', $userSettings, null)
        );

        $data = array_merge($userSettings, $data);

        $data['gitRefspec'] = !$data['gitRefspec'] ? 'master' : $data['gitRefspec'];

        $this->template->load('template', 'appsetting/vcs', $data);
    }

    /**
     * @rbacName Reset version control repository
     * @rbacGroup Settings
     * @rbacAlias settings.vcsReset
     */
    public function vcsReset()
    {

        $this->load->model('settings_model');
        $this->load->model('vcsSettingsModel');


        try {
            $this->vcsSettingsModel->deleteSettings(); // its admin setting
        } catch (Exception $e) {
            $this->session->set_flashdata('message', array(
                'content' => "There was some error deleting git configuration",
                'type'    => 'alert-error'
            ));
        }

        redirect('settings/vcs');
    }

    /**
     * @return array|bool|string
     */
    private function handleFormUploadForGitSettings()
    {
        $this->form_validation->set_rules('vcsType', 'VCS type', 'xss_clean|trim|required');
        $this->form_validation->set_rules('gitServer', 'git server url', 'xss_clean|trim|required');
        $this->form_validation->set_rules('gitRefspec', 'Git refspec', 'xss_clean|required');
        $this->form_validation->set_rules('projectSubdirectory', 'Project subdirectory', 'xss_clean|trim');
        $this->form_validation->set_rules('gitUsername', 'git username', 'xss_clean|trim');
        $this->form_validation->set_rules('gitPassword', 'git password', 'xss_clean|trim');


        if($this->vcsSettingsModel->isSshUrl($this->input->post('gitServer')) && empty($_FILES['gitPrivateKey']['name'])){
            $this->form_validation->set_rules('gitPrivateKey', 'Git private key', 'xss_clean|trim|required');
        }


        if ($this->form_validation->run() === true) {

            $gitPrivateKey = (isset($_FILES['gitPrivateKey']) && ! empty($_FILES['gitPrivateKey']['tmp_name'])) ?  $this->vcsSettingsModel->readSshKey($_FILES['gitPrivateKey']['tmp_name']) : '';

            try {
                $postData = array(
                    'vcsType'         => $this->input->post('vcsType', true),
                    'gitServer'       => $this->input->post('gitServer', true),
                    'gitRefspec'      => $this->input->post('gitRefspec', true),
                    'projectSubdirectory' => $this->input->post('projectSubdirectory', true),
                    'gitUsername'     => $this->input->post('gitUsername', true),
                    'gitPassword'     => $this->input->post('gitPassword', true),
                    'gitPrivateKey'   => $gitPrivateKey
                );

                $this->vcsSettingsModel->createSettings($postData);

                $this->session->set_flashdata('message', array(
                    'content' => "The git repository has been successfully configured.",
                    'type'    => 'alert-success'
                ));
                $feedback = true;
            } catch (Exception $e) {
                $message = json_decode($e->getMessage(), true);
                log_message('error', $message);
                $feedback = array(
                    'type'    => 'alert-error',
                    'content' => $message
                );
            }
        } else {
            $feedback = validation_errors();
        }

        return $feedback;
    }

    /**
     * @rbacName Active tokens
     * @rbacGroup Settings
     * @rbacAlias settings.online_users
     */
    function online_users()
    {

        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Logged in users"
        );

        $this->load->model('Authentication_model');

        $data['activeSessions'] = $this->Authentication_model->getActiveSession();

        $this->template->load('template', 'appsetting/users_online', $data);
    }

    /**
     * @rbacName Revoke tokens
     * @rbacGroup Settings
     * @rbacAlias settings.revoke_token
     */
    function revoke_token($tokens)
    {
        $tokens = urldecode($tokens);
        $tokens = explode('|', $tokens);
        $this->load->model('Authentication_model');
        try {
            $this->Authentication_model->unsetToken($tokens);
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), 'ERROR revoking auth tokens. Error:: ' . $e->getMessage());
            show_error('Unexpected error while performing token revoke.');
        }
        redirect('settings/active-tokens');
    }

    /**
     * @rbacName View
     * @rbacGroup Notification scripts
     * @rbacAlias settings.notification_scripts
     */
    function notification_scripts($action = null, $id = null)
    {

        if (!empty($action) && ($action == "edit" || $action == "add")) {
            redirect('settings/notification-scripts-update/' . $id);
        } elseif (!empty($action) && ($action == "delete")) {
            redirect('settings/notification-scripts-delete/' . $id);
        }


        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Custom notification scripts"
        );

        $this->load->model('dashboard/alerts_model');

        try {
            $data['scripts'] = $this->alerts_model->getAllNotificationScripts();
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), 'Error getting scripts:: ' . $e->getMessage());
            show_error('Unexpected error while retrieving list of custom scripts.');
        }

        $this->template->load('template', 'appsetting/notification_scripts', $data);
    }


    /**
     * @rbacName Delete
     * @rbacGroup Notification scripts
     * @rbacAlias settings.notification_scripts_delete
     */
    function notification_scripts_delete($id)
    {
        $this->load->model('dashboard/alerts_model');
        $this->alerts_model->deleteAlertScript($id);
        redirect('settings/notification-scripts');

    }

    /**
     * @rbacName Create
     * @rbacGroup Notification scripts
     * @rbacAlias settings.notification_scripts_update
     */
    function notification_scripts_update($id = null)
    {

        $this->load->model('dashboard/alerts_model');

        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Custom notification scripts"
        );

        $feedback = false;
        $script   = null;

        try {
            if (!empty($id)) {
                $script = $this->alerts_model->getNotificationScript($id);
            }
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), 'Error getting scripts:: ' . $e->getMessage());
            show_error('Unexpected error while retrieving details.');
        }

        $hasObjectValue =!empty($script);
        $data['script'] = $script;

        if ($this->input->post('update')) {
            try {
                $feedback = $this->handleFormUploadForNotificationScripts($script);
            } catch (Exception $e) {
                $output   = $e->getMessage();
                $feedback = array(
                    'type'    => 'alert-error',
                    'content' => $output
                );
            }

            if ($feedback === true) {
                redirect('settings/notification-scripts');
            }
        }

        $formData = array(
            'scriptname'        => set_value('scriptname', $hasObjectValue ? $script->getName() : null),
            'scriptdescription' => set_value('scriptdescription', $hasObjectValue ? $script->getDescription() : null),
            'scriptfilename'    => set_value('scriptfilename', $hasObjectValue ? $script->getScriptFileName() : null),
            'id'                => set_value('id', $hasObjectValue ? $script->getId() : null)
        );

        $data['message']   = $feedback;
        $data['form_data'] = $formData;

        $this->template->load('template', 'appsetting/notification_scripts_update', $data);

    }

    private function handleFormUploadForNotificationScripts($script)
    {
        $feedback = true;
        $this->form_validation->set_rules('scriptname', 'name', 'xss_clean|trim|required');
        $this->form_validation->set_rules('scriptdescription', 'description', 'xss_clean|trim');
        if (empty($script) && empty($_FILES['userfile']['name'])) {
            $this->form_validation->set_rules('userfile', 'file', 'required');
        }

        if ($this->form_validation->run() === true) {
            try {
                $data['name']        = $_POST['scriptname'];
                $data['description'] = $_POST['scriptdescription'];
                $uploadOk            = true;
                if (!empty($_FILES['userfile']['name'])) {
                    //try upload the files

                    $config['upload_path'] = $this->config->item('notification_script_dir');;
                    $config['allowed_types'] = '*';

                    if (!file_exists($config['upload_path'])) {
                        if (!mkdir($config['upload_path'], 0755)) {
                            throw new Exception("Cannot make directory on path: " . $config['upload_path']);
                        }

                    }

                    $this->load->library('upload', $config);


                    if (!$this->upload->do_upload()) {
                        $feedback = array(
                            'type'    => 'alert-error',
                            'content' => $this->upload->display_errors()
                        );
                        $uploadOk = false;
                    } else {
                        $fileData               = $this->upload->data();
                        $data['scriptFileName'] = $fileData['file_name'];
                        chmod($fileData['full_path'], 0750); // make it executable
                    }

                }
                if ($uploadOk) {
                    if ($script) {
                        $this->alerts_model->updateAlertScripts($script, $data);
                    } else {
                        $this->alerts_model->insertAlertScripts($data);
                    }
                }

            } catch (Exception $e) {
                $message = $e->getMessage();
                log_message('error', $message);
                $feedback = array(
                    'type'    => 'alert-error',
                    'content' => $message
                );
            }
        } else {
            $feedback = validation_errors();
        }

        return $feedback;
    }


    /**
     * @rbacName Update mail settings
     * @rbacGroup Settings
     * @rbacAlias settings.mail
     */
    public function mail()
    {
        $this->load->model('MailSettingsModel');
        $settings = $this->MailSettingsModel->getSettings();
        $data     = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Mail settings CFEngine "
        );
        $data     = array_merge($settings, $data);


        if ($this->input->method() == 'post') {
            $data = array_merge($data, $this->input->post());

            $this->form_validation->set_rules('protocol', 'Mail protocol', 'required');
            $this->form_validation->set_rules('max_email_attach_size', 'Max email attachment size (MB)',
                'required|integer|less_than_equal_to[100]');
            $this->form_validation->set_rules('default_from_email', 'Default from email', 'required|valid_email');

            if ($this->input->post('protocol') == 'smtp') {
                $this->form_validation->set_rules('smtp_host', 'SMTP host', 'required');
                $this->form_validation->set_rules('smtp_user', 'SMTP user', 'required');
                $this->form_validation->set_rules('smtp_port', 'SMTP port', 'required');
            }

            if ($this->form_validation->run($this) == true) {
                $this->MailSettingsModel->updateSettings($this->input->post());
                if ($this->input->is_ajax_request()) {
                    return true;
                }
            } else {
                $errors = validation_errors();
                if ($this->input->is_ajax_request()) {
                    $this->output->set_status_header(419, 'validation error');
                    echo $errors['content'];

                    return false;
                }
                $this->session->set_flashdata('message', validation_errors());
            }
        }


        $this->template->load('template', 'appsetting/mail', $data);
    }


    public function profile()
    {

        $this->load->model('ProfileModel');
        $userName = $this->session->userdata('username');

        if ($this->input->method() == 'post') {
            $postData = $this->input->post();
            $this->ProfileModel->updateProfile($postData, $userName);

            $neverAsk =  (isset($postData['neverAskAgain']))  ? (int) $postData['neverAskAgain'] : 1;
            $this->ProfileModel->setNeverAskTimeZoneChange($neverAsk, $userName);

            $useBrowserTime =  (isset($postData['useBrowserTime']))  ? (int) $postData['useBrowserTime'] : 0;
            $this->ProfileModel->setUseBrowserTime($useBrowserTime, $userName);

            redirect('settings/profile');
        }

        $data = [
            'title' => $this->lang->line('mission_portal_title') . " - User profile CFEngine",
            'timeZonesList' => $this->ProfileModel->getTimeZoneList(),
            'profile' => $this->ProfileModel->getProfileData($userName),
            'neverAskAgain' => $this->ProfileModel->getNeverAskTimeZoneChange($userName),
            'useBrowserTime' => $this->ProfileModel->getUseBrowserTime($userName)
        ];

        $this->template->load('template', 'appsetting/profile', $data);
    }

    /**
     * @rbacName Send test mail
     * @rbacGroup Settings
     * @rbacAlias settings.sendTestEmail
     */
    public function sendTestEmail()
    {
        $this->load->model('MailSettingsModel');
        $defaultEmail = $this->MailSettingsModel->getSetting('default_from_email');

        $data = array(
            'title'          => $this->lang->line('mission_portal_title') . " - Send test email ",
            'defaultEmail'   => $defaultEmail,
        );

        $this->form_validation->set_rules('from', 'From', 'required|valid_email');
        $this->form_validation->set_rules('to', 'To', 'required|valid_email');

        if ($this->input->post() && $this->form_validation->run($this) == true) {
            $this->load->library('email');
            $this->email->from($this->input->post('from'));
            $this->email->to($this->input->post('to'));
            $this->email->subject('CFEngine Enterprise hub test email');
            $this->email->message('Test email sent from ' . gethostname() . ' by user ' . $this->session->userdata('username') . '.');
            $result = $this->email->send();
            if (!$result) {
                $data['debugInfo'] = $this->email->print_debugger();
                log_message('debug', 'send test email error:' . $data['debugInfo']);
            } else {
                $data['debugInfo'] = 'Email successfully sent';
            }
        } else {
            $this->session->set_flashdata('message', validation_errors());
        }
        $data = array_merge($data, $this->input->post());
        $this->template->load('template', 'appsetting/sendTestEmail', $data);
    }

    /**
     * @rbacName Update LDAP auth settings
     * @rbacGroup Settings
     * @rbacAlias settings.ldap_api_settings
     */
    public function ldap_api_settings()
    {
        $ldapApiUrl            = $this->config->item('ldapApiUrl');
        $pestClient            = new Http_client(['base_url' => $ldapApiUrl]);
        $apiServerLDAPSettings = $this->ldap_api->getSettings($pestClient);
        $is_ldap_available     = $this->ldap_api->testLdapUrl($pestClient);

        $settingsValue = $this->settings_rest_model->get_app_settings();

        if (!$is_ldap_available) {
            log_message('error',
                'Could not communicate with LDAP API at ldapApiUrl=' . $ldapApiUrl . ' check configuration in /var/cfengine/share/GUI/application/config/appsettings.php');
        }

        $settingsValue['ldap_roles_list_to_sync'] = isset($settingsValue['ldap_roles_list_to_sync']) ?
            implode("\r\n", json_decode($settingsValue['ldap_roles_list_to_sync'])) :
            '';

        $data = array(
            'title'              => $this->lang->line('mission_portal_title') . ' - Authentication settings',
            'is_correct_api_url' => (bool)$apiServerLDAPSettings,
            'is_ldap_available'  => $this->ldap_api->testLdapUrl($pestClient),
            'mode'               => $settingsValue['ldapEnabled'] ? 'ldap' : 'internal',
            'settings'           => $settingsValue
        );


        $post = $this->input->post();
        if ($post) {

            $this->form_validation->set_rules('domain_controller', 'Ldap Host', 'required');
            $this->form_validation->set_rules('port', 'Port', 'required|integer');
            $this->form_validation->set_rules('base_dn', 'Base DN', 'required');
            $this->form_validation->set_rules('login_attribute', 'Login attribute', 'required');
            $this->form_validation->set_rules('ldap_filter', 'Ldap filter', 'trim');
            if ( empty($this->input->post('anonymous_bind'))) {
                $this->form_validation->set_rules('admin_username', 'LDAP bind user name', 'required');
                $this->form_validation->set_rules('admin_password', 'LDAP bind password', 'required');
            } else {
                unset($post['admin_username']);
                unset($post['admin_password']);
            }
            $this->form_validation->set_rules('anonymous_bind', 'Anonymous bind', 'trim');
            $this->form_validation->set_rules('group_attribute', 'User group attribute', 'trim');
            $this->form_validation->set_rules('settings[ldap_enable_role_syncing]', 'settings[ldap_enable_role_syncing]', 'trim');
            $this->form_validation->set_rules('settings[ldap_roles_list_to_sync]', 'settings[ldap_roles_list_to_sync]', 'trim');
            $this->form_validation->set_rules('settings[ldap_perform_sync_on_login]', 'settings[ldap_perform_sync_on_login]', 'trim');
            $this->form_validation->set_rules('settings[ldap_remove_role_on_syncing]', 'settings[ldap_remove_role_on_syncing]', 'trim');
            if ($this->form_validation->run($this) == true) {

                // update general LDAP settings
                $settingsToSave = $post['settings'];
                $settingsToSave['ldap_enable_role_syncing'] = key_exists('ldap_enable_role_syncing', $settingsToSave);
                $settingsToSave['ldap_perform_sync_on_login'] = key_exists('ldap_perform_sync_on_login', $settingsToSave);
                $settingsToSave['ldap_remove_role_on_syncing'] = key_exists('ldap_remove_role_on_syncing', $settingsToSave);
                $settingsToSave['ldap_roles_list_to_sync'] = key_exists('ldap_roles_list_to_sync', $settingsToSave) ?
                    json_encode(explode("\r\n", $settingsToSave['ldap_roles_list_to_sync'])) :
                    '';
                $this->settings_rest_model->updateData($settingsToSave);


                unset($post['settings']);
                $response = $this->ldap_api->saveSettings($pestClient, $post);
                if (!$response['success']) {
                    $message = [
                        'content' => ucfirst($response['message']),
                        'type'    => 'alert-error'
                    ];
                } else {
                    $message = [
                        'content' => 'Settings were successfully saved.',
                        'type'    => 'alert-success'
                    ];
                }
                $apiServerLDAPSettings = $this->input->post();
                $this->session->set_flashdata('message', $message);
                redirect('settings/authentication');
            } else {
                $apiServerLDAPSettings = $this->input->post();
                $this->session->set_flashdata('message', validation_errors());
            }
        }


        $data = array_merge($apiServerLDAPSettings, $data);
        $data['admin_password'] = isset($data['admin_password']) && $data['admin_password'] != '' ? 'Password is set' : 'Password is not set';
        $data['anonymous_bind'] = isset($data['anonymous_bind']) && $data['anonymous_bind'];
        $this->template->load('template', 'appsetting/auth_settings', $data);
    }

    /**
     * @rbacName Change authentication method
     * @rbacGroup Settings
     * @rbacAlias settings.saveAuthenticationMethod
     */
    public function saveAuthenticationMethod()
    {
        $returnData = $data = [];
        if ((!empty($this->input->post('mode')))) {
            $data['ldapEnabled'] = boolval($this->input->post('mode') == 'ldap');
        }

        try {
            log_message('debug', "saveAuthenticationMethod(): Trying to update data via REST ");
            $this->settings_rest_model->updateData($data);
            $returnData['status']  = true;
            $returnData['message'] = 'Data saved successfully.';
        } catch (Exception $e) {
            log_message('debug', "saveAuthenticationMethod(): REST - Could not save your settings");
            $returnData['status']  = false;
            $returnData['message'] = "Could not save your settings. " . $e->getMessage();
        }
        echo json_encode($returnData);
    }
}
