/*jslint node: true */
/*jslint browser: true*/
/*jslint nomen: true*/
/*global angular, BaseController, CFE_mapping, defaultSQL, STRING_CONDITION, NUMERIC_CONDITION, LIST_CONDITION, DATE_CONDITION, CONTEXT_CONDITION, FIELD_TYPES  */
'use strict';

/**
 *  Directive
 *
 * Use of Class.js
 *
 */
var inventoryFilterDirectiveCtrl = BaseController.extend({
    init: function ($scope, $elm, $attrs, $q, inventoryHelper, inventoryDataService, $location) {
        var self = this;
        this._super($scope);
        this._element = $elm;
        this._attrs   = $attrs;
        this._$q      = $q;
        this._inventoryHelper = inventoryHelper;
        this._inventoryDataService = inventoryDataService;
        this.$location = $location;

        self.$scope.filters = []; // all selected filters collected here, and only after user click apply we will move them to the SQL filter

        self.$scope.defaultSQL = defaultSQL;

        /* example of data  type in array
         self.$scope.SQL_EXPRESSION_ARRAY[field]['SELECT'] = string
         self.$scope.SQL_EXPRESSION_ARRAY[field]['FROM']  = string
         self.$scope.SQL_EXPRESSION_ARRAY[field]['WHERE'] = array;
        */

        self.$scope.SQL_EXPRESSION_ARRAY = {};


        self.$scope.STRING_CONDITION  = STRING_CONDITION;
        self.$scope.NUMERIC_CONDITION = NUMERIC_CONDITION;
        self.$scope.LIST_CONDITION    = LIST_CONDITION;
        self.$scope.DATE_CONDITION    = DATE_CONDITION;
        self.$scope.CONTEXT_CONDITION = CONTEXT_CONDITION;

        self.$scope.FIELD_TYPES       = FIELD_TYPES;

        self.$scope.openedDate = []; // for data field

        self._initInventoryObject();
        self._prepareInventoryVariables(self.$scope.cfemappingaddfields, self.$scope.cfemappingremovefields, self.$scope.cfemappingaddtofilters);
        $scope.data = {};
        this.defaultClassList = DefaultInventoryFilterClasses;
        self.$scope.data.classes = this.defaultClassList;
    },

    defineScope: function () {
        var self = this;
        self.$scope.addFilter    = this.addFilter.bind(this);
        self.$scope.applyFilter  = this.applyFilter.bind(this);
        self.$scope.updateFilterState = this.updateFilterState.bind(this);
        self.$scope.regexValidation = this.regexValidation.bind(this);
        self.$scope.removeFilter      = this.removeFilter.bind(this);
        self.$scope.clearAll     = this.clearAll.bind(this);
    },

    defineListeners: function () {
        var self = this;

        self.$scope.$on("clearAll_EVENT", function (event) {
            self.clearAll();
        });

        self.$scope.$on("runDefaultReport_EVENT", function (event, columns) {
            self._runDefaultReport(columns);
        });

        self.$scope.$on("modifyColumn_EVENT", function (event, column, pos) {
            self._modifyColumn(column, pos);
        });

        self.$scope.$on("modifyColumns_EVENT", function (event, columns) {
            self._modifyColumns(columns);
        });

        self.$scope.$on("environmentFilter_EVENT", function (event) {
            self.applyFilter();
        });

        self.$scope.$on("applyInvFilters_EVENT", function (event) {
            self._updateSQLAfterFilterChange();
        });

        self.$scope.$watch('sql', function (sql) {
            self.$scope.sql = sql;
            self.$scope.filters = !sql.filters || sql.filters.length == 0 ? [{}] : sql.filters;
        }, true);

        self.$scope.$watch('data.searchFilter', function (searchFilter) {
            if (searchFilter !== undefined && searchFilter.length > 0) {
                self._inventoryDataService.getClasses(searchFilter).then(function (r) {
                    self.$scope.data.classes = r.data.map(item => item[0])
                })
            } else {
                self.$scope.data.classes = self.defaultClassList;
            }
        })

        self.$scope.$on("reportLoaded_EVENT", function (event, SQL) {
            self.$scope.sql = SQL;

            self.$scope.filters = SQL.filters;
            self._updateSQL_ExpressionArrays();

            self.$scope.sql.SQLSTRING = self._refreshSQL(); // we must refresh SQL string accordingly to what we have as form input.

            self._runQuery();
        });

        self.$scope.$on("updateVariablesDictionary_EVENT", function(event) {
            self._updateVariablesDictionary();
        });

        self.$scope.$on("getInventoryAlertData_EVENT", function(event, defaultColumns) {
            self.$scope.sql.columns = [];

            // refresh filters, this will also remove empty filters and set sql.filters
            self._updateSQLAfterFilterChange(); // this will set all properties to SQL object

            // for inventory report we set columns = filter, and remove duplicates
            angular.forEach(self.$scope.sql.filters, function(column, index) {
                var pos = self._findIndexInObjectByAttributeName(self.$scope.sql.columns, column.attribute_name);
                if (pos === -1) {
                    //unset attr which we don't need
                    var tmp = angular.copy(column);
                    delete (tmp.condition);
                    delete (tmp.value);
                    self.$scope.sql.columns.push(tmp);
                }
            });

            // append default col - usually hostname
            angular.forEach(defaultColumns, function(column, index) {
                var pos = self._findIndexInObjectByAttributeName(self.$scope.sql.columns, column.attribute_name);
                if (pos === -1) {
                 self.$scope.sql.columns.push(column);
                }
            });

            // update SELECT part with columns from filter + default
            self._updateSQL_ExpressionArrays();

            self.$scope.sql.SQLSTRING = self._refreshSQL();
        });
    },



////////////////////////////////////////////////////////////////////////////////
//
//                  UI - related
//
////////////////////////////////////////////////////////////////////////////////


    /**
     * Get an element from filters array and add it to the filters.
     * set condition properties
     *
     *@param {object} field
     */
    addFilter: function(field, index) {
        var self = this;
        if (angular.equals({}, field)) {
            return;
        }

        var filterField = {};
        angular.copy(field, filterField);
        self.$scope.filters = self.$scope.filters ? self.$scope.filters : [];

        // check for software fields: combine as one field
        if (field.attribute_name === 'Software installed' || field.attribute_name === 'Software version') {

            var tmpField  = {};
            filterField['type']   = 'software_group';
            filterField['fields'] = {};

            // 1. get software name field
            tmpField = self._findObjectByAttributeName(self.$scope.ui.InventoryItems.filters.softwareFilter, 'Software installed');

            if (!angular.equals({}, tmpField)) {
                filterField['fields']['softwarename'] = angular.copy(tmpField);
            }

            // 2. get Software version  field
            tmpField = self._findObjectByAttributeName(self.$scope.ui.InventoryItems.filters.softwareFilter, 'Software version');

            if (!angular.equals({}, tmpField)) {
                filterField['fields']['softwareversion'] = angular.copy(tmpField);
            }
        }

        if (index !== undefined) {
            // preserve previous value
            let previousValue = angular.copy(self.$scope.filters[index].value);
            // in case of int or real convert value to number
            if (filterField.type === 'int' || filterField.type === 'real') {
                let parsedValue = parseFloat(previousValue);
                previousValue = isNaN(parsedValue) ? '' : parsedValue;
                filterField.value = previousValue;
            } else if (filterField.type === 'class') {
                filterField.value = field.value;
            } else {
                filterField.value = previousValue;
            }

            // overwrite
            self.$scope.filters[index] = filterField;

            // set default condition
            if (!self.$scope.filters[index].condition) {
                self.$scope.filters[index].condition = self._setDefaultCondition(field.type);
            }
        } else {
            // add new filter
            self.$scope.filters.push(filterField);
        }
        this.updateFilterState();
    },

    _setDefaultCondition: function(type) {
        var self = this;
        var defaultCondition;
        switch (type) {
            case 'slist':
                defaultCondition = self.$scope.LIST_CONDITION[0].value;
                break;
            case 'string':
                defaultCondition = self.$scope.STRING_CONDITION[0].value;
                break;
            case 'date':
                defaultCondition = self.$scope.DATE_CONDITION[0].value;
                break;
            case 'context':
                defaultCondition = self.$scope.CONTEXT_CONDITION[0].value;
                break;
            case 'int':
            case 'real':
                defaultCondition = self.$scope.NUMERIC_CONDITION[0].value;
                break;
            default:
                defaultCondition = self.$scope.STRING_CONDITION[0].value;
                break

        }

        return defaultCondition;
    },

    /**
     * Remove filter
     * @param {int} idx index
     */
    removeFilter: function(idx) {
        var self = this;
        var key = self.$scope.filters[idx].attribute_name;

        if (self.$scope.SQL_EXPRESSION_ARRAY[key]) {
            self.$scope.SQL_EXPRESSION_ARRAY[key]['WHERE'] = '';
        }
        //remove filter from URL when filter removed on the UI
        self.$location.search('filter', null)

        self.$scope.filters.splice(idx, 1);

        self.$scope.sql.filters = angular.copy(self.$scope.filters);

        self._refreshWHEREArray();

        if (self.$scope.filters.length === 0) {
            self._updateSQLAfterFilterChange(); // clear filter
            self.$scope.sql.SQLSTRING = self._refreshSQL();
            self._runQuery();
        }
    },

    /**
     * Apply filters
     */
    applyFilter: function() {
        var self = this;
        //don't update session storage if report has no columns
        if (self.$scope.sql.columns != undefined && self.$scope.sql.columns.length > 0) {
            self._updateSQLAfterFilterChange();
            self._runQuery();
        }


        self.$scope.showAllFilters = false;
    },

    /**
     * Clear all fields and checkboxes in form
     * Clear SQL string
     */
    clearAll: function() {
        var self = this;

        self.$scope.SQL_EXPRESSION_ARRAY = {};
        self.$scope.filters     = [];
        self.$scope.sql.filters = [];
        self.$scope.sql.hostsFilters = {includes: {}, excludes: {}};

        angular.forEach(self.$scope.hardwareGroup, function(col) {
            col.selected = false;
        });
        angular.forEach(self.$scope.softwareGroup, function(col) {
            col.selected = false;
        });
        angular.forEach(self.$scope.networkGroup, function(col) {
            col.selected = false;
        });
        angular.forEach(self.$scope.noGroupGroup, function(col) {
            col.selected = false;
        });
        angular.forEach(self.$scope.noCategoryGroup, function(col) {
            col.selected = false;
        });
        self._updateSQLAfterFilterChange();
    },

    /**
     * Used in datablicker for blur event
     */
    updateFilterState: function(currentFilter) {
        var self = this;
        self._updateSQL_ExpressionArrays();

        self.$scope.filters = self.$scope.filters.map(function (filter) {
            if (filter.value == undefined) {
                filter.value = '';
            }
            return filter;
        })

        if(currentFilter){
            var newFilter = $.grep(self.$scope.filters, function (filter) {
                return filter.id_key == currentFilter.id_key;
            })[0];
            self._inventoryDataService.updateVariable(newFilter);
        }
        self._updateSQLAfterFilterChange();
        self.$scope.$emit('INV_FILTER_CHANGED');
    },

    regexValidation: function(formElement, filter) {

        let self = this;
        clearTimeout(self.timeOut);
        clearTimeout(self.timeOutRemoveBlock);

        self.templates = {
            inProgress: `<span class="warning"><span class="icon-spinner icon-spin"></span> Validating regular expression</span>`,
            succeed : `<span class="green"><span class="icon-ok"></span> Validation successful</span>`,
            failed: `<span class="red"<span class="icon-warning-sign"></span> error_msg</span>`
        };

        const inventoryHelpBlock =  $(formElement.$$element).next('.inventory_help');

        if (filter.condition.indexOf('regex') == 0 && formElement.$viewValue) {
            // show message about validation when user changing value
            inventoryHelpBlock.html(self.templates.inProgress).show();
        }

        const validationFn = (formElement, filter, inventoryHelpBlock, self) => {
        const value = formElement.$viewValue;
        if (filter.condition.indexOf('regex') == 0 && formElement.$viewValue) {
            let params = {
                select: ['Host name'],
                filter: {}
            };
            params['filter'][filter.label] = {};
            params['filter'][filter.label][filter.condition] = value

            // clear interval, otherwise validation help block will be blinking when typing
            return this._inventoryDataService.getData(params, {"hide-spinner": true}).then(
                function () {
                    formElement.$setValidity('regex', true);
                    inventoryHelpBlock.html(self.templates.succeed);
                    self.timeOutRemoveBlock = setTimeout(() => $('.inventory_help .green').hide(), 2000);
                },
                function (error) {
                    clearTimeout(self.timeOutRemoveBlock);
                    const extractedError = error.data.substr(error.data.indexOf('invalid regular expression'))
                        || 'Regex validation error';
                    formElement.$setValidity('regex', false);
                    // set error message
                    inventoryHelpBlock.html(self.templates.failed.replace('error_msg', extractedError.capitalizeFirstLetter()));
                }
            );
        } else {
            formElement.$setValidity('regex', true);
        }
    }
    self.timeOut = setTimeout(() => validationFn(formElement, filter, inventoryHelpBlock, self), 500);
    },

////////////////////////////////////////////////////////////////////////////////
//
//                  SQL - related
//
////////////////////////////////////////////////////////////////////////////////

    /**
     * Emit event to run query
     */
    _runQuery: function() {
        var self = this;
        self.$scope.$emit('runQuery_EVENT');
    },


    /**
     * Refresh SQL string
     *
     * Note - There is no limit in SQL string!! Limit is set by pagination
     *
     */
    _refreshSQL: function() {
        var self = this, SQLSTRING = '',  tmp = '';
        if (!angular.equals({}, self.$scope.SQL_EXPRESSION_ARRAY)) {
            SQLSTRING = self.$scope.defaultSQL.SELECT;

            tmp = self._getArrayForSQLString('SELECT');

            if (tmp.length)
            {
                SQLSTRING += tmp.join(', ');
            }


            SQLSTRING += " " + self.$scope.defaultSQL.FROM;
            tmp = self._getArrayForSQLString('FROM');
            if (tmp.length)
            {
                SQLSTRING += " " + tmp.join(' ');
            }


            tmp = self._getArrayForSQLString('WHERE');


            if (tmp.length)
            {
                SQLSTRING += " " + self.$scope.defaultSQL.WHERE;
                SQLSTRING += " " + tmp.join(' AND ');
            }


            SQLSTRING += " " + self.$scope.defaultSQL.GROUP;
            tmp = self._getArrayForSQLString('GROUPBY');
            if (tmp.length)
            {
                SQLSTRING += ", " + tmp.join(', ');
            }
        }
        return SQLSTRING;
    },

    /**
     * Return from statement from field
     * @param {object} field
     */
    _getFromStr: function(field) {
        var self = this, tablename_alias = '', from_str = '';
        tablename_alias = self._getTableNameAlias(field);

        // Note: software table shouldn't be here
        switch (field.source_table) {
            case 'inventory':
                // if field contain attribute_name - join by meta
                if (field.attribute_name.indexOf('attribute_name=') !== -1) {
                    from_str = " LEFT JOIN getHostkeyAndValuesForAttribute('" + field.attribute_name + "') " + tablename_alias  + " ON  "  + tablename_alias +".hostkey = hosts.hostkey";
                }
                else
                {
                    // if no attribute name - use keyname.
                    // usually this is for columns/filters without attribute_name set
                    from_str = " LEFT JOIN getHostkeyAndValuesForKeyName('" + field.keyname + "') " + tablename_alias  + " ON  "  + tablename_alias +".hostkey = hosts.hostkey";
                }
                break;

            case "hosts":  // host table included by default
                from_str = '';
                break;
        }
        return from_str;
    },

    /**
     * Return group by statement from field
     * @param {object} field
     */
    _getGroupByStr: function(field) {
        var self = this, tablename_alias = '', groupby_str = '';

        tablename_alias = self._getTableNameAlias(field);

        // Note: software table shouldn't be here
        switch (field.source_table) {
            case 'inventory':
                // we can not use label here, because for items without attribute_name label is empty
                groupby_str = tablename_alias + '.inventoryValue';
                break;
                // use keyname for hosts
            case 'hosts':
                groupby_str = 'hosts.' + field.keyname;
                break;
        }

        return groupby_str;
    },

    /**
     * Refresh SQL-related arrays based on form
     */
    _updateSQL_ExpressionArrays: function() {
        var self = this;
        self.$scope.SQL_EXPRESSION_ARRAY = {};

        if (self.$scope.sql.filters && self.$scope.sql.filters.length > 0)
        {
            self._refreshWHEREArray();
        }
        self._refreshSelectFromGroupByArray();
    },

    /**
     * Create array for SELECT FROM GROUPBY statement
     */
    _refreshSelectFromGroupByArray: function() {
        var self = this;

        if (
                 (self.$scope.sql.columns === undefined || self.$scope.sql.columns.length === 0)
           )
        {
            self.$scope.SQL_EXPRESSION_ARRAY = {};
            return;
        }

        angular.forEach(self.$scope.sql.columns, function(field, id) {
            if (field) {
                self.$scope.SQL_EXPRESSION_ARRAY[field.attribute_name] = self.$scope.SQL_EXPRESSION_ARRAY[field.attribute_name] || {};
                self.$scope.SQL_EXPRESSION_ARRAY[field.attribute_name]['SELECT']  = self._getSelectStr(field);
                self.$scope.SQL_EXPRESSION_ARRAY[field.attribute_name]['FROM']    = self._getFromStr(field);
                self.$scope.SQL_EXPRESSION_ARRAY[field.attribute_name]['GROUPBY'] = self._getGroupByStr(field);
            }
        });
    },

    _getSelectStr: function(field) {

        var self = this, tablename_alias = '', sqlTableAndValueField = '', select_str = '';

        tablename_alias = self._getTableNameAlias(field);
        sqlTableAndValueField = self._getTableAndValueField(field);

        // Note: software table shouldn't be here

        switch (field.source_table) {
            case 'inventory':

                switch (field.type) {
                    case 'int':
                    case 'real':
                        select_str = self._utils_castToNumeric(sqlTableAndValueField) + ' AS ' + '"' + field.label + '"';
                        break;

                    case 'slist':
                        if (field.convert_function)
                        {
                            select_str = field.convert_function + '(' + sqlTableAndValueField + ') AS ' + '"' + field.label + '"';
                        }
                        else
                        {
                            select_str = SLIST_DEFAULT_CONVERT_FUNCTION + '(' + sqlTableAndValueField + ") AS " + '"' + field.label + '"';
                        }
                        break;

                    case 'context':
                        select_str = sqlTableAndValueField+ " AS " + '"' + field.label + '"';
                        break;

                    default:
                        select_str = sqlTableAndValueField + " AS " + '"' + field.label + '"';
                }

                break;
            case 'hosts': // because host table is included by default
                select_str = field.keyname + " AS " + '"' + field.label + '"';
                break;
        }

        return select_str;
    },

    /**
     * Create array for WHERE statement
     */
    _refreshWHEREArray: function() {
        var self = this,  SOFTWARE_WHERE = [], softwareSubQuery = '';
        // we have to link software table as subquery in the end of where statement.
        // that is why we will append all software-related items into separated string

        angular.forEach(self.$scope.sql.filters, function(filter, index) {

            if (filter.type === 'software_group')
            {
                softwareSubQuery = self._utils_getSoftwareSubQuery(filter);
                // add to the global array
                if (softwareSubQuery !== '') {
                    SOFTWARE_WHERE.push(softwareSubQuery);
                }
            }
            else {
                self._utils_WHERE_UPDATE_SQL_EXPRESSION_ARRAYS(filter);
            } // else

        });

        if (SOFTWARE_WHERE.length)
        {
            var softwareCondString = ' hosts.hostkey in ( ' + SOFTWARE_WHERE.join(' INTERSECT ') + ' ) ';
            // append software
            self.$scope.SQL_EXPRESSION_ARRAY['software_where'] = [];
            self.$scope.SQL_EXPRESSION_ARRAY['software_where']['WHERE'] = [];
            self.$scope.SQL_EXPRESSION_ARRAY['software_where']['WHERE'].push(softwareCondString);
        }
    },

    /**
     * Special wrapper to cast values into nummber
     *
     * @param {string} fieldName
     * @returns {String}
     */
    _utils_castToNumeric: function(fieldName) {
        //return 'cf_convertToNum(' + fieldName + ')';
    },

    /**
     * Return sql string for software filter
     *
     * @param {object} filter
     * @returns {String}
     */
    _utils_getSoftwareSubQuery: function (filter) {
        var softwarePair = [], // contain pair of softwarename and software version sql conditions
            SOFTWARE_WHERE_STR = '',
            softwareSubQuery = '';


        if(!filter['fields']['softwarename']['value']) {
            return '';
        }
        /**
         * SQL query for like and = different to NOT LIKE and !=
         * because in case of NOT LIKE and != we are looking for hosts where this package is NOT EXISTS
         *
         * simple check for NOT LIKE or != won't help, because host has a lot of different packages and
         * "not like" simple return wrong result, (any package is not like to what user is looking for)
         *
         * in case of LIKE and "=" subquery should look like
         * SELECT hostkey FROM software WHERE softwarename ILIKE '%cfengine-nova%'
         *
         * in case of NOT LIKE and "!="
         * SELECT hostkey FROM software except(SELECT hostkey FROM software WHERE softwarename ILIKE '%cfengine-nova-hub%')
         */

        // note: we use ILIKE for both becuase 'except' is used for negative queries
        switch (filter['fields']['softwarename']['condition']) {
            case 'ILIKE':
            case "NOT ILIKE" :
                SOFTWARE_WHERE_STR = 'softwarename ILIKE' + " '%" + filter['fields']['softwarename']['value'] + "%'";
                break;
            case "!="   :
            case "="    :
                SOFTWARE_WHERE_STR = 'softwarename ILIKE ' + " '" + filter['fields']['softwarename']['value'] + "' ";
                break;
        }

        softwarePair.push(SOFTWARE_WHERE_STR);

        if (filter['fields']['softwareversion']['value'])
        {
            switch (filter['fields']['softwareversion']['condition']) {
                case 'ILIKE':
                    SOFTWARE_WHERE_STR = "softwareversion " + filter['fields']['softwareversion']['condition'] + " '%" + filter['fields']['softwareversion']['value'] + "%'";
                    break;
                case "NOT ILIKE" :
                    SOFTWARE_WHERE_STR = "softwareversion " + filter['fields']['softwareversion']['condition'] + " '%" + filter['fields']['softwareversion']['value'] + "%'";

                    // if using 'except' query, must make condition opposite
                    if (filter['fields']['softwarename']['condition'] === 'NOT ILIKE' || filter['fields']['softwarename']['condition'] === '!=') {
                        SOFTWARE_WHERE_STR = "LOWER( softwareversion ) ILIKE " + "'%" + filter['fields']['softwareversion']['value'] + "%'";
                    }
                    break;
                case "!="   :
                    SOFTWARE_WHERE_STR = "LOWER( softwareversion ) NOT ILIKE " + "'" + filter['fields']['softwareversion']['value'] + "'";

                    // if using 'except' query, must make condition opposite
                    if (filter['fields']['softwarename']['condition'] === 'NOT ILIKE' || filter['fields']['softwarename']['condition'] === '!=') {
                        SOFTWARE_WHERE_STR = "LOWER( softwareversion ) ILIKE " + "'" + filter['fields']['softwareversion']['value'] + "'";
                    }
                    break;
                case "="    :
                    SOFTWARE_WHERE_STR = "LOWER( softwareversion ) ILIKE " + "'" + filter['fields']['softwareversion']['value'] + "'";
                    break;
            }

            softwarePair.push(SOFTWARE_WHERE_STR);
        }



        // create subquery for pair condition depends on software_version condition, see comment above
        if (filter['fields']['softwarename']['condition'] === 'NOT ILIKE' || filter['fields']['softwarename']['condition'] === '!=') {
            softwareSubQuery = " SELECT hostkey FROM software EXCEPT ( SELECT hostkey FROM software WHERE " + softwarePair.join(' AND ') + " )";
        } else {
            softwareSubQuery = " SELECT hostkey FROM software WHERE " + softwarePair.join(' AND ');
        }

        return softwareSubQuery;
    },

    /**
     * Convert field type and trim values for string
     * in case of datapicker - convert date object to string
     *
     * @param {object} filter
     * @returns {} value
     */
    _utils_convertFieldValueByType: function(filter) {
        var self = this;
        // TODO: cast field type for numberic data
        //
        if (filter.type !== 'date' && typeof filter.value === 'string') {
            filter.value = filter.value.trim();
        }
        else
        { // check and convert date object form datepicker to string
            if (self._getVariableJavasctiptType(filter.value) === 'date')
            {
                filter.value = self._getDateFromDateObject(filter.value);
            }
            else if (typeof filter.value === 'string') {
                filter.value = filter.value.trim();
            }
        }

        return filter.value;
    },


    /**
     * Return WHERE string for slist type variables
     *
     * @param {object} filter
     * @returns {String}
     */
    _utils_getSlistWHERE: function(filter) {

        var self = this, tmp = [], listArray = [], sqlValueField = '', realCondition = '',  WHERE = '';

        listArray = filter.value.split(',');

        sqlValueField = self._getTableAndValueField(filter);

        if (listArray.length)
        {
            switch (filter.condition)
            {
                // we must use case insensitive strict like
                case "="  :
                    realCondition = 'ILIKE';
                    for (var i = 0; i < listArray.length; i++)
                    {
                        tmp.push(sqlValueField + realCondition + " '%" + listArray[i].trim() + "%' ");
                    }
                    break;

                case "!=" :
                    realCondition = 'NOT ILIKE';
                    for (var i = 0; i < listArray.length; i++)
                    {
                        tmp.push(sqlValueField + realCondition + " '%" + listArray[i].trim() + "%' ");
                    }
                    break;

                case 'ILIKE':
                case 'NOT ILIKE':
                    for (var i = 0; i < listArray.length; i++)
                    {
                        tmp.push(sqlValueField + filter.condition + "('%' || '" + listArray[i].trim() + "' || '%')");
                    }
                    break;
            }

            if (tmp.length) {
                WHERE = tmp.join(' AND ');
            }
        }

        return WHERE;
    },

    /**
     * Return  WHERE string for string type variables
     *
     * @param {object} filter
     * @returns {String}
     */
    _utils_getStringWHERE: function(filter) {
        var self = this, sqlValueField = '', WHERE = '';

        sqlValueField = self._getTableAndValueField(filter);

        switch (filter.condition)
        {
            case "ILIKE"     :
            case "NOT ILIKE" :
                WHERE = sqlValueField + filter.condition + " '%" + filter.value + "%'";
                break;

            // we must use case insensitive strict like
            case "="        :
                WHERE = sqlValueField + 'ILIKE' + " '" + filter.value + "'";
                break
            case "!="       :
                WHERE = sqlValueField + 'NOT ILIKE' + " '" + filter.value + "'";
                break;
        }

        return WHERE;
    },


    /**
     * Return WHERE string for numeric fields
     *
     * @param {object} filter
     * @returns {string}
     */
    _utils_getNumericWHERE: function(filter) {
        var self = this, sqlValueField = '', WHERE = '';

        sqlValueField = self._getTableAndValueField(filter);

        switch (filter.condition)
        {
            case "=" :
            case "!=":
            case ">" :
            case "<" :
                 WHERE = self._utils_castToNumeric(sqlValueField) + filter.condition + filter.value;
                 break;
        }

        return WHERE;
    },


     /**
     * Return WHERE string for date field
     *
     * @param {object} filter
     * @returns {String}
     */
    _utils_getDateWHERE: function(filter) {
        var self = this, sqlValueField = '', WHERE = '';

        sqlValueField = self._getTableAndValueField(filter);

        switch (filter.condition)
        {
            case ">" :
            case "<" :
                 WHERE = sqlValueField + filter.condition + "'" + filter.value + "'";
                 break;
        }

        return WHERE;
    },

    _utils_WHERE_UPDATE_SQL_EXPRESSION_ARRAYS: function(filter) {
        var self = this, WHERE = '';

        //ignore empty items
        if (filter.value == null || !filter.condition) {
            return;
        }

        self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name] = self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name] || {};


        WHERE = self._getWhereStr(filter);
        self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name]['WHERE'] = self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name]['WHERE'] || [];
        self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name]['WHERE'].push(WHERE);

        self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name]['FROM']    = self._getFromStr(filter);
        self.$scope.SQL_EXPRESSION_ARRAY[filter.attribute_name]['GROUPBY'] = self._getGroupByStr(filter);
    },

    _getWhereStr: function(filter) {

        var self = this, WHERE = '';

        filter.value = self._utils_convertFieldValueByType(filter);

        switch (filter.type)
        {
            case 'slist':
                WHERE = self._utils_getSlistWHERE(filter);
                break;

            case 'string':
                WHERE = self._utils_getStringWHERE(filter);
                break;

            case 'int':
            case 'real':
                WHERE = self._utils_getNumericWHERE(filter);
                break;

            case 'date':
                WHERE = self._utils_getDateWHERE(filter);
                break;

            case 'context':
                WHERE = self._utils_getStringWHERE(filter);
                break;
        }

        return WHERE;
    },


    /**
     * Return part of SQL string as array from SQL_EXPRESSION_ARRAY array, used to construct final SQL expression
     *
     * @param {string} SQL_PART  part of SQL string to construct. Allowed: SELECT, FROM, GROUPBY, WHERE
     */
    _getArrayForSQLString: function(SQL_PART) {
        var self = this, elements = [], pos = -1;

        // for select we must maintain proper order of columns.
        // and if hostname is selected it alwas should be first
        if (SQL_PART === 'SELECT') {
            // search for host and move it to the first place if exist
            self.$scope.ui.SELECT_HEADER_ARRAY = [];

            pos = self._findIndexInObjectByAttributeName(self.$scope.sql.columns, 'Host name');


            if (pos !== -1 && pos !== 0) {
                // remove and add to the first place
                self.$scope.sql.columns.splice(pos, 1);
                self.$scope.sql.columns.unshift(self.$scope.ui.CFE_mapping['Host name']);
            }


            // construct array for sql string and append data to SELECT_HEADER_ARRAY
            angular.forEach(self.$scope.sql.columns, function(column, index) {
                if (self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name]['SELECT']) {
                    elements.push(self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name][SQL_PART]);
                    self.$scope.ui.SELECT_HEADER_ARRAY.push(column);
                }
            });
        }
        else {
            angular.forEach(self.$scope.SQL_EXPRESSION_ARRAY, function(filter, index) {

                if (!filter[SQL_PART]) {
                    return;
                }
                switch (SQL_PART) {
                    case "FROM":
                    case "GROUPBY":
                        elements.push(filter[SQL_PART]);
                        break;

                    case "WHERE":
                        if (filter[SQL_PART].length) {
                            elements.push(filter[SQL_PART].join(' AND '));
                        }
                        break;
                }
            });
         }

        return elements;
    },


////////////////////////////////////////////////////////////////////////////////
//
//                  Utils
//
////////////////////////////////////////////////////////////////////////////////

    /**
     *  Update SQL accordingly to selected filters
     *
     *  it will also remove all filters which are empty
     */
    _updateSQLAfterFilterChange: function() {
        var self = this;
        var tmpFilter = new Array();

        // cleanup filters - remove filters without value
        if (self.$scope.filters) {
            angular.forEach(self.$scope.filters, function(filter, index) {
                //ignore empty items
                if (filter.type!== 'software_group') {
                    if (filter.value != null && filter.condition) {
                        tmpFilter.push(filter);
                    }
                } else {
                    // if softwarename is empty - do not use that filter
                    if (filter.fields['softwarename']['value'] && filter.fields['softwarename']['condition']) {
                        tmpFilter.push(filter);
                    }
                }
            });
            self.$scope.filters = tmpFilter;
        }


        self.$scope.sql.filters = angular.copy(self.$scope.filters);

        self._updateSQL_ExpressionArrays();

        self.$scope.sql.SQLSTRING = self._refreshSQL();
        sessionStorage.setItem('sql',JSON.stringify(self.$scope.sql));
    },

    /**
     * Update variables dictionary. Usually called on save report
     */
    _updateVariablesDictionary: function() {
        var self = this, variablesList = {};
        if (self.$scope.sql.filters) {

            angular.forEach(self.$scope.sql.filters, function(field, index) {
                if (field.readonly !== "1") { // do not update readonly fields
                    variablesList[field.attribute_name] = variablesList[field.attribute_name] || {};
                    variablesList[field.attribute_name] = field;
                }
            });
        }
        if (!angular.equals({}, variablesList)) {
            var data = angular.toJson(variablesList);
            self._inventoryHelper.updateVariablesDictionary(data).then(
                    function(result) {
                        // do nothing
                    },
                    function(error) {
                        error.data[0].text = error.data[0].text || 'Error. Unable to update variable type';
                        self._notificationService.setNotification('error', error.data);

                    }
            );
        }
    },

    /**
     * Run default report - no filtering only preselected columns
     *
     * @param {array} defaultColumns - array of default columns, format - see DCA
     */
    _runDefaultReport: function(defaultColumns) {
        var self = this, isFound = false;
        if (defaultColumns) {
            self.$scope.sql.columns = [];
            angular.forEach(defaultColumns, function(label) {
                 angular.forEach(self.$scope.ui.InventoryItems.grouped, function(groupField, groupIndex) {
                    if (groupField.label === label)
                    {
                        // mark field as checked
                        groupField.selected = true;
                        self.$scope.sql.columns.push(groupField);
                        isFound = true;
                    }
                });
            });
        }

        if (isFound === true) {
            self._refreshSelectFromGroupByArray();
            self.$scope.sql.SQLSTRING = self._refreshSQL();
            self._runQuery();
        }
    },

    /**
     * When someone selected new column - re-run query  this function usually called from event
     * @param {object} field
     */
    _modifyColumn: function (col, position) {
        var self = this;
        var pos = -1;
        var field, isInInventoryItems = false;

        angular.forEach(self.$scope.ui.InventoryItems.grouped, function (groupField) {
            if (groupField.label === col) {
                // mark field as checked
                field = groupField;
                isInInventoryItems = true;
                pos = self._findIndexInObjectByAttributeName(self.$scope.sql.columns, field.attribute_name);

                if (pos !== -1) {
                    self._removeSqlColumn(field, pos);
                } else {
                    var newPos = position ? position : self.$scope.sql.columns.length;
                    self.$scope.sql.columns.splice(newPos, 0, field);
                }

                self._updateSqlQueries();
            }
        });

        if (!isInInventoryItems) {
            angular.forEach(self.$scope.sql.columns, function (column, index) {
                if ((col === column.attribute_name) || (('attribute_name=' + col) === column.attribute_name)) {
                    self._removeSqlColumn(column, index);
                    self._updateSqlQueries();
                }
            });
        }
    },

    /**
     * Modify all columns
     * @param {array} columns
     */
    _modifyColumns: function (columns) {
        let self = this;

        self.$scope.sql.columns = [];
        angular.forEach(columns, function (label) {
            angular.forEach(self.$scope.ui.InventoryItems.grouped, function (groupField) {
                if (groupField.label === label) {
                    groupField.selected = true;
                    self.$scope.sql.columns.push(groupField);
                }
            });
        });
        self._updateSqlQueries();
    },

    /**
     * Remove sql column
     *
     * @param {object} column
     * @param {integer} index
     * @private
     */
    _removeSqlColumn: function (column, index) {
        var self = this;
        self.$scope.sql.columns.splice(index, 1);
        if (self.$scope.SQL_EXPRESSION_ARRAY.hasOwnProperty(column.attribute_name)) {
            delete (self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name]['SELECT']);


            // if this field is not part of filter - remove FROM and GROUPBY
            if (!self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name]['WHERE']) {
                delete (self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name]['FROM']);
                delete (self.$scope.SQL_EXPRESSION_ARRAY[column.attribute_name]['GROUPBY']);
            }
        }
    },

    /**
     * Updates sql queries
     * @private
     */
    _updateSqlQueries: function () {
        var self = this;
        self._refreshSelectFromGroupByArray();
        self._updateSQL_ExpressionArrays();

        self.$scope.sql.SQLSTRING = self._refreshSQL();
        sessionStorage.setItem('sql', JSON.stringify(self.$scope.sql));
        self._runQuery();
    },

    /**
     * Search for item in object by attribute_name
     *
     * @param {object} object
     * @param {string} attribute_name
     *
     * @return {object} item if empty object
     */
    _findObjectByAttributeName: function(object, attribute_name) {
        var result = {};

        angular.forEach(object, function(groupItem, index) {
            if (groupItem.attribute_name === attribute_name) {
                result = groupItem;
                return;
            }
        });
        return result;
    },

   /**
     * Search for item in object by attribute_name and return index
     *
     * @param {object} object
     * @param {string} attribute_name
     *
     * @return {int} index of element in array or -1
     */
    _findIndexInObjectByAttributeName: function(object, attribute_name) {
        var result = -1;

        angular.forEach(object, function(groupItem, index) {
            if (groupItem.attribute_name === attribute_name) {
                result = index;
                return;
            }
        });
        return result;
    },


     /**
      * Return table name alias
      * @param {object} field  - field object
      */
    _getTableNameAlias: function(field) {
        var tablename_alias = '';
        // Note: software table shouldn't be here
        if (field) {
            switch(field.source_table) {
                case "hosts":
                    tablename_alias = 'hosts';
                    break;

                default:
                    // todo: check this!
                    tablename_alias = field.keyname? "table_" + field.keyname.replace(/[^a-zA-Z0-9]/g, '_') : '';
            }
        }

        return tablename_alias;
    },

    /**
     * Returns sql string for field value
     *
     * @param {object} field
     *
     */
     _getTableAndValueField: function(field) {
        var self = this, valueField = '', sqlValueField = '', tablename_alias = '';

        tablename_alias = self._getTableNameAlias(field);

        if (field.source_table === 'inventory')
        {
            valueField = 'inventoryValue'; // this field is taken from getHostkeyAndValuesForAttribute() result
        }
        else
        {
            valueField = field.keyname;
        }

        sqlValueField = tablename_alias + "." + valueField + " "; // !note space at the end
        return sqlValueField;
    },

    /*Try to get javascript variable type, used for data.picker*/
    _getVariableJavasctiptType: function(variable) {
        return ({}).toString.call(variable).match(/\s([a-zA-Z]+)/)[1].toLowerCase();
    },

    /**
     * Used when selecting date from datapicker
     * @param {date} dateObject
     */
    _getDateFromDateObject: function(dateObject) {
        var monthStr = '', dayStr = '';
        var day   = dateObject.getDate();
        var year  = dateObject.getFullYear();
        var month = dateObject.getMonth() + 1;

        monthStr = month < 10 ? "0" + month : month;
        dayStr   = day   < 10 ? "0" + day   : day;

        return year + "-" + monthStr + "-" + dayStr;
    },

    /**
     * Initialize inventory object
     */
    _initInventoryObject: function() {
        var self = this;
        //self.$scope.UI = {}; - initialized outside direcive
        self.$scope.ui.SELECT_HEADER_ARRAY    = []; // to convert result based on type, for example remove quotes from slist
        self.$scope.ui.CFE_mapping            = {};
        self.$scope.ui.InventoryItems         = {};
        self.$scope.ui.InventoryItems.groups  = {};
        self.$scope.ui.InventoryItems.filters = {};

        self.$scope.ui.InventoryItems.groups.hardwareGroup   = {};
        self.$scope.ui.InventoryItems.groups.softwareGroup   = {};
        self.$scope.ui.InventoryItems.groups.networkGroup    = {};
        self.$scope.ui.InventoryItems.groups.noCategoryGroup = {};
        self.$scope.ui.InventoryItems.groups.noGroupGroup    = {};
        self.$scope.ui.InventoryItems.grouped = {};

        self.$scope.ui.InventoryItems.filters.hardwareFilter   = {};
        self.$scope.ui.InventoryItems.filters.softwareFilter   = {};
        self.$scope.ui.InventoryItems.filters.networkFilter    = {};
        self.$scope.ui.InventoryItems.filters.noCategoryFilter = {};
        self.$scope.ui.InventoryItems.filters.noGroupFilter    = {};
    },

    /**
     * Load variables from DB
     */


    _prepareInventoryVariables: function(addFields, removeFields, addFilters) {
        var self = this, deferred = self._$q.defer();

        self._inventoryHelper.getInventoryVariables(addFields, removeFields, addFilters).then(
                function(result) {
                    // extend scope with new variables
                    angular.extend(self.$scope.ui, result);
                    deferred.resolve();
                    self.$scope.$emit('inventory_variables_loaded', result); // notify that we loaded all variables
                    self.$scope.$emit('inventoryDirectiveReady_EVENT'); // notify that we loaded all variables
                },
                function(error) {
                    // error handled in helper
                    deferred.reject(error);
                }
        );
        return deferred.promise;
    }
});