<?php

/**
 * @uri /inventory/attributes-dictionary
 */
class VariablesDictionary extends CfProtectedResource
{

    /**
     * @var VariablesDictionaryModel
     */
    private $model;


    /**
     * VariablesDictionary constructor.
     * @param array $args
     */
    public function __construct(array $args)
    {
        $this->model = new VariablesDictionaryModel();
        parent::__construct($args);
    }

    /**
     * @param $request
     * @return Response
     *
     * @rbacName Get inventory attributes
     * @rbacGroup Inventory API
     * @rbacAlias variablesDictionary.get
     * @rbacAllowedByDefault
     */
    function get($request)
    {
        $response = new Response($request);
        $data = $this->model->getAll();
        $response->body = json_encode($data);
        $response->code = Response::OK;

        return $response;
    }

    /**
     * Update variables dictionary
     *
     * @param $request
     * @return Response
     */
    function post($request)
    {
        $response = new Response($request);
        $this->model->updateCustomVariables();
        $response->code = Response::OK;
        return $response;
    }


}

/**
 * @uri /inventory/attributes-dictionary/:id
 */
class VariablesDictionaryUpdate extends CfProtectedResource
{

    /**
     * @var VariablesDictionaryModel
     */
    private $model;


    /**
     * VariablesDictionaryUpdate constructor.
     * @param array $args
     */
    public function __construct(array $args)
    {
        $this->model = new VariablesDictionaryModel();
        parent::__construct($args);
    }


    /**
     * @param $request
     * @param $id
     * @return Response
     * @throws Exception
     *
     * @rbacName Update inventory attributes
     * @rbacGroup Inventory API
     * @rbacAlias variablesDictionaryUpdate.post
     * @rbacAllowedByDefault
     */
    public function patch($request, $id)
    {
        $response = new Response($request);

        if (!$id) {
            throw new Exception('Id is required field.');
        }
        $inputData = Utils::getValidJsonData($request->data);
        $variable = $this->model->getById($id);

        if (!$variable) {
            $response->code = Response::NOTFOUND;
            $response->body = 'Variable not found.';
            return $response;
        }

        if ($variable['readonly'] == 1) {
            $response->code = Response::FORBIDDEN;
            $response->body = 'You cannot edit read only variable';
            return $response;
        }

        if (!$this->model->validate($inputData)) {
            $response->body = json_encode($this->model->errors);
            $response->code = Response::BADREQUEST;
            return $response;
        }

        $this->model->update($id, $inputData);

        $response->body = json_encode($this->model->getById($id));
        $response->code = Response::OK;

        return $response;
    }

}

/**
 * @api {get} inventory/attributes-dictionary List of variables
 *
 * @apiName VariablesDictionaryList
 * @apiGroup Variables Dictionary
 * @apiVersion 1.0.0
 * @apiHeader {String} Authorization User token. Exmp.: `Bearer f9c6000853a90e4043b745263786777e0ff0011e`
 *
 *
 * @apiSuccessExample Success-Response:
 * HTTP/1.1 200 OK
 * [
 *   {
 *     "id": 2,
 *     "attribute_name": "BIOS vendor",
 *     "category": "Hardware",
 *     "readonly": 1,
 *     "type": "string",
 *     "convert_function": null
 *   },
 *   {
 *     "id": 3,
 *     "attribute_name": "BIOS version",
 *     "category": "Hardware",
 *     "readonly": 1,
 *     "type": "string",
 *     "convert_function": null
 *   }
 * ]
 */

/**
 * @api {post} inventory/attributes-dictionary Add custom variables to data dictionary
 *
 * @apiName VariablesDictionaryUpdate
 * @apiGroup Variables Dictionary
 * @apiVersion 1.0.0
 * @apiHeader {String} Authorization User token. Exmp.: `Bearer f9c6000853a90e4043b745263786777e0ff0011e`
 *
 * @apiSuccessExample Success-Response:
 * HTTP/1.1 200 OK
 */

/**
 * @api {patch} inventory/attributes-dictionary/:id Edit variable
 *
 * @apiName VariablesDictionaryEdit
 * @apiGroup Variables Dictionary
 * @apiVersion 1.0.0
 * @apiHeader {String} Authorization User token. Exmp.: `Bearer f9c6000853a90e4043b745263786777e0ff0011e`
 * @apiHeader {String=application/json} Content-Type
 *
 * @apiParam {int} id   Variable Id
 * @apiParam {string} category  Category
 * @apiParam {string=int,real,slist,string} type   Variable type
 * @apiParam {string} convert_function   Convert Function
 *
 *
 * @apiSuccessExample Success-Response:
 * HTTP/1.1 200 OK
 *
 * {
 *   "id": 1,
 *   "attribute_name": "Architecture",
 *   "category": "Hardware",
 *   "readonly": 0,
 *   "type": "slist",
 *   "convert_function": "cf_clearSlist"
 * }
 *
 * @apiErrorExample {string} Validation errors:
 * HTTP/1.1 400 Bad Request
 * [
 * "Type is wrong. Allowed values: int,real,slist,string",
 * "Convert function is wrong. Allowed values: cf_clearSlist"
 * ]
 *
 * @apiErrorExample {string} Forbidden error:
 * HTTP/1.1 403 Forbidden
 * You cannot edit read only variable
 *
 */