<?php


namespace CMDB\Models;

use \CMDB\Entities\CmdbItemEntity;

class CmdbItemModel
{
    private \PDO $dbConnection;
    private $table = '__cmdb';
    private $view = 'cmdb';
    private $username;


    public function __construct(string $username)
    {
        $this->username = $username;
        $this->dbConnection = \CfdbPdo::getInstance()->getConnection();
    }


    public function createOrUpdate(CmdbItemEntity $entity)
    {
        $sql = "INSERT INTO {$this->table} (hostkey, value, epoch) 
                VALUES (:hostkey, :value, :epoch) 
                ON CONFLICT (hostkey) 
                DO UPDATE SET 
                value = EXCLUDED.value, epoch = EXCLUDED.epoch, updated_at = 'now()'";

        $this->dbConnection->prepare($sql)->execute([
            'hostkey' => $entity->getHostKey(),
            'value' => $entity->valuesToJson(),
            'epoch' => $this->maxEpoch() + 1,
        ]);
    }

    public function maxEpoch()
    {
        $sql = "SELECT max(epoch) FROM {$this->table}";
        $result = $this->dbConnection->query($sql)->fetch(\PDO::FETCH_ASSOC);
        return $result['max'] ?? 0;
    }

    public function update(CmdbItemEntity $entity)
    {
        $sql = "UPDATE {$this->table} SET value = :value, epoch = :epoch, updated_at = 'now()'";
        $this->dbConnection->prepare($sql)->execute([
            'value' => $entity->valuesToJson(),
            'epoch' => $this->maxEpoch() + 1
        ]);
    }

    public function updateByName(string $hostkey, string $type, string $name, array $requestData, array $existingData)
    {
        // get name from requested data if exists to update var or class name
        $newName = isset($requestData['name']) && $requestData['name'] !== $name ? $requestData['name'] : null;
        unset($requestData['name']);
        $mergedData = array_merge((array) $existingData[$type][$name], $requestData);

        $params = [
            'value' => json_encode($mergedData),
            'epoch' => $this->maxEpoch() + 1,
            'hostkey' => $hostkey
        ];

        if ($newName !== null) {
            $sql = "UPDATE {$this->table} SET value = jsonb_set(value #- :oldPath, :newPath, :value), epoch = :epoch, updated_at = 'now()' WHERE hostkey LIKE :hostkey";
            $params['oldPath'] = '{' . "$type,$name" . '}';
            $params['newPath'] = '{' . "$type,$newName" . '}';
        } else {
            $sql = "UPDATE {$this->table} SET value = jsonb_set(value, :path, :value), epoch = :epoch, updated_at = 'now()' WHERE hostkey LIKE :hostkey";
            $params['path'] = '{' . "$type,$name" . '}';
        }

        $this->dbConnection->prepare($sql)->execute($params);
    }

    public function insertOrUpdateByName(string $hostkey, string $type, string $name, array $requestData)
    {
        $newValue = ['classes' => new \stdClass(), 'variables' => new \stdClass()];
        $newValue[$type] = [$name => $requestData];

        $sql = "INSERT INTO {$this->table} 
                (hostkey, value, epoch, updated_at) 
                VALUES (:hostkey,  :newValue, :epoch, 'now()') 
                ON CONFLICT (hostkey) 
                DO UPDATE SET epoch = EXCLUDED.epoch, updated_at = EXCLUDED.updated_at, value = jsonb_set(
                    jsonb_build_object(
                        'classes', 
                        coalesce(__cmdb.value->>'classes','{}')::jsonb, 
                        'variables', 
                        coalesce(__cmdb.value->>'variables','{}')::jsonb
                    ), 
                    :path, 
                    :value, 
                    TRUE
                    )";

        $this->dbConnection->prepare($sql)->execute([
            'value' => empty($requestData) ? '{}' : json_encode($requestData),
            'path' => '{' . "$type, $name" . '}',
            'epoch' => $this->maxEpoch() + 1,
            'newValue' => json_encode($newValue),
            'hostkey' => $hostkey
        ]);
    }

    public function getByName(string $hostkey, string $type, string $name)
    {
        $params = \Utils::processQueryApiParams([]);
        $params['query'] = "SELECT value->'$type'->'$name' as data FROM {$this->view} WHERE hostkey LIKE '$hostkey' AND (value->'$type'->'$name') is not null";

        $result = json_decode(
            cfapi_query_post(
                $this->username,
                $params['query'],
                $params['sortColumn'] ?? '',
                $params['sortDescending'] ?? false,
                $params['skip'] ?? -1,
                $params['limit'] ?? -1,
                $params['hostContextInclude'] ?? [],
                $params['hostContextExclude'] ?? []
            ), true);

        $data = null;
        if (isset($result['data'][0]['rows'][0][0])) {
           $data = ['hostkey' => $hostkey];
           $data[$type][$name] = json_decode($result['data'][0]['rows'][0][0]);
       }

        return $data;
    }


    public function deleteByName(string $hostkey, string $type, string $name)
    {
        $sql = "UPDATE {$this->table} SET value = value #- :path, epoch = :epoch, updated_at = 'now()' WHERE hostkey LIKE :hostkey";
        $this->dbConnection->prepare($sql)->execute([
            'path' => '{' . "$type,$name" . '}',
            'epoch' => $this->maxEpoch() + 1,
            'hostkey' => $hostkey,
        ]);
    }

    public function list(array $params)
    {
        $where = [];
        if (isset($params['fromEpoch'])) {
            $where[] = "epoch > {$params['fromEpoch']} ";
        }

        if (isset($params['fromTime'])) {
            $where[] = "updated_at > '{$params['fromTime']}'";
        }

        if (isset($params['toTime'])) {
            $where[] = "updated_at < '{$params['toTime']}'";
        }

        $whereCond = sizeof($where) > 0 ? ' WHERE ' . implode(' AND ', $where) : '';
        $limit = $params['limit'] > 0 ? "LIMIT {$params['limit']} " : '';
        $offset = $params['skip'] > 0 ? "OFFSET {$params['skip']} " : '';
        $params['query'] = "
                WITH 
                main_cte AS (SELECT  hostkey, value, epoch FROM {$this->view} $whereCond $limit $offset),
                total_cte AS (SELECT hostkey FROM {$this->view} $whereCond)
                SELECT 
                json_object_agg(main_cte.hostkey, main_cte.value) as value,
                count(main_cte.hostkey) as count,
                (SELECT COUNT(*) FROM total_cte) as total
                FROM main_cte";

        $result = json_decode(
            cfapi_query_post(
                $this->username,
                $params['query'],
                $params['sortColumn'] ?? '',
                $params['sortDescending'] ?? false,
                $params['skip'] ?? -1,
                $params['limit'] ?? -1,
                $params['hostContextInclude'] ?? [],
                $params['hostContextExclude'] ?? []
            ), true);

        $data = $this->prepareResponse($result);
        //get epoch value individually to return the latest value even if there are no data in the response
        $data['meta']['cmdb_epoch'] = $this->getMaxEpoch($params);
        return $data;
    }

    private function getMaxEpoch($params)
    {
        $params['query'] = "SELECT max(epoch) as cmdb_epoch FROM {$this->view}";
        $result = json_decode(
            cfapi_query_post(
                $this->username,
                $params['query'],
                $params['sortColumn'] ?? '',
                $params['sortDescending'] ?? false,
                $params['skip'] ?? -1,
                $params['limit'] ?? -1,
                $params['hostContextInclude'] ?? [],
                $params['hostContextExclude'] ?? []
            ), true);
        return $result['data'][0]['rows'][0][0] ?? 0;
    }

    private function prepareResponse(array $data)
    {
        $response = [
            'data' => [],
            'meta' => array_merge($data['meta'], ["cmdb_epoch" => 0])
        ];

        if (isset($data['data'][0]['rows'][0])) {
            $row = $data['data'][0]['rows'][0];
            $response['data'] = json_decode($row[0]);
            $response['meta']['count'] = $row[1];
            $response['meta']['total'] = $row[2];
        }

        return $response;
    }

    public function get(string $hostkey)
    {
        $params = \Utils::processQueryApiParams([]);
        $params['query'] = "SELECT hostkey, value FROM {$this->view} WHERE hostkey LIKE '$hostkey'";

        $result = json_decode(
            cfapi_query_post(
                $this->username,
                $params['query'],
                $params['sortColumn'] ?? '',
                $params['sortDescending'] ?? false,
                $params['skip'] ?? -1,
                $params['limit'] ?? -1,
                $params['hostContextInclude'] ?? [],
                $params['hostContextExclude'] ?? []
            ), true);

        return isset($result['data'][0]['rows'][0][0]) ?
            [
                'hostkey' => $result['data'][0]['rows'][0][0],
                'value' => json_decode($result['data'][0]['rows'][0][1])
            ] :
            null;
    }

    public function delete(string $hostkey)
    {
        $sql = "DELETE FROM {$this->table}  WHERE hostkey LIKE :hostkey";
        $this->dbConnection->prepare($sql)->execute(['hostkey' => $hostkey]);
    }
}
