<?php

/**
 *  CfLoginAttempts class is responsible for logging failed authentication attempts and 
 * returning time out before the next available attempt.
 *  IP address is used as user identifier. That means that a pause between attempts 
 * will be applied to the IP not the real user.
 */
class CfLoginAttempts
{
    // fallback ip address. in case if the real IP address cannot be received for unknown reason
    const FALLBACK_IP = '0.0.0.0';
    //  reset attempts if more than 1 hour has passed since the last attempt
    const RESET_ATTEMPTS_AFTER_SEC = 3600;
    // 30 seconds, 1 minute, 2 minutes, 5 minutes, 10 minutes.
    const DELAYS_IN_SECONDS = [30, 60, 120, 300, 600];
    // apply delays when the threshold is reached
    const ATTEMPTS_THRESHOLD = 3;

    protected PDO $db;
    protected string $ipAddress;
    private string $table = 'failed_login_attempts';

    public function __construct($ipAddress)
    {
        $this->db = CfSettings::getInstance()->getConnection();
        $this->ipAddress = $ipAddress ?: self::FALLBACK_IP;
    }

    /**
     * @return int|null
     *
     * Return delay in seconds if time out until time for the given IP address is greater than the current time.
     */
    public function timeOutBeforeNextAttempt(): ?int
    {
        $currentTimestamp = time();
        $fa = $this->getFailedAttempt();

        return ($fa && ($currentTimestamp < $fa->timeoutUntil)) ? $fa->timeoutSeconds : null;
    }

    public function logFailedAttempt(): void
    {
        $fa = $this->getFailedAttempt();
        
        if ($fa === null) {
            $this->createFailedAttempt();
        } else {
            $currentTimestamp = time();

            // set attempts to 1 when RESET_ATTEMPTS_AFTER_SEC passed since the last failed attempt
            $attempts = (($currentTimestamp - $fa->lastAttempt) > self::RESET_ATTEMPTS_AFTER_SEC) ?
                1 :
                $fa->attempts + 1;

            $timeOutSeconds = $attempts >= self::ATTEMPTS_THRESHOLD ? $this->getNextDelay($fa->timeoutSeconds) : 0;
            $timeOutUntil = date('Y-m-d H:i:s', $currentTimestamp + $timeOutSeconds);

            $this->updateFailedAttempt($attempts, $timeOutUntil, $timeOutSeconds);
        }
    }

    private function getFailedAttempt(): ?FailedAttemptDTO
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE ip_address = ?");
        $stmt->execute([$this->ipAddress]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? FailedAttemptDTO::fromArray($row) : null;
    }

    private function createFailedAttempt(): void
    {
        $statement = $this->db->prepare("INSERT INTO {$this->table} (ip_address, attempts, last_attempt) VALUES (?, 1, NOW())");
        $statement->execute([$this->ipAddress]);
    }

    private function updateFailedAttempt(int $attempts, string $timeOutUntil, int $timeOutSeconds): void
    {
        $statement = $this->db->prepare(
            "UPDATE {$this->table}
                   SET attempts = :attempts, last_attempt = NOW(), timeout_until = :timeout_until, timeout_seconds = :timeout_seconds 
                   WHERE ip_address = :ip_address"
        );

        $statement->execute([
            'ip_address' => $this->ipAddress,
            'attempts' => $attempts,
            'timeout_until' => $timeOutUntil,
            'timeout_seconds' => $timeOutSeconds,
        ]);
    }

    /**
     * @param int $currentDelay
     * @return int
     *
     * Returns the first delay that is greater than the current one
     * or returns the last delay from DELAYS_IN_SECONDS
     * what happens when the delay reaches its maximum.
     */
    private function getNextDelay(int $currentDelay): int
    {
        $delaysGreaterThanCurrent = array_filter(self::DELAYS_IN_SECONDS, fn($delay) => $delay > $currentDelay);
        $firstDelay = reset($delaysGreaterThanCurrent);
        return $firstDelay ?: self::DELAYS_IN_SECONDS[array_key_last(self::DELAYS_IN_SECONDS)];
    }
}

readonly class FailedAttemptDTO
{
    public function __construct(
        public string $ipAddress,
        public int $attempts,
        public int $lastAttempt,
        public int $timeoutUntil,
        public int $timeoutSeconds
    )
    {
    }

    public static function fromArray(array $data): self
    {
        return new self(
            $data['ip_address'] ?? '',
            $data['attempts'] ?? 0,
            strtotime($data['last_attempt'] ?? '') ?: 0,
            strtotime($data['timeout_until'] ?? '') ?: 0,
            $data['timeout_seconds'] ?? 0
        );
    }
}
