\set ON_ERROR_STOP true

-------------------------------------------------------------------------------
-- Create settings database:
-------------------------------------------------------------------------------

CREATE TABLE IF NOT EXISTS ScheduledReports (
	username		TEXT		NOT NULL,
	query			TEXT		NOT NULL,
	query_id		TEXT		NOT NULL,
	run_classes		TEXT		NOT NULL,
	last_executed 		TEXT,
	email			TEXT,
	email_title		TEXT,
	email_description	TEXT,
	host_include		TEXT[],
	host_exclude		TEXT[],
        already_run		BOOLEAN		DEFAULT FALSE,
        enabled			BOOLEAN		DEFAULT TRUE,
	output			TEXT[]		NOT NULL
    );

ALTER TABLE ScheduledReports
ADD COLUMN IF NOT EXISTS excludedhosts json DEFAULT NULL;

CREATE TABLE IF NOT EXISTS Settings (
        key                     TEXT		NOT NULL,
        value			JSON
    );
CREATE UNIQUE INDEX IF NOT EXISTS settings_unique_key_idx ON settings (key);

CREATE TABLE IF NOT EXISTS Users (
        username                TEXT		NOT NULL,
        password                TEXT,
        salt                    TEXT,
        name                    TEXT,
        email                   TEXT,
        external                BOOLEAN         DEFAULT FALSE,
        active                  BOOLEAN         DEFAULT FALSE,
        roles                   TEXT[]          DEFAULT '{}',
        time_zone               TEXT            DEFAULT 'Etc/GMT+0',
        ChangeTimestamp         timestamp with time zone       DEFAULT now()
    );

-- Add time_zone column to users --
-- Default value is Etc/GMT+0 what means no time offset (GMT) by default if time_zone is not specified --
ALTER TABLE Users ADD COLUMN IF NOT EXISTS time_zone TEXT DEFAULT 'Etc/GMT+0';

ALTER TABLE Users ADD COLUMN IF NOT EXISTS password_expiration_time timestamp DEFAULT NULL;

CREATE UNIQUE INDEX IF NOT EXISTS users_uniq ON Users (username, external);

CREATE TABLE IF NOT EXISTS Roles (
        name                    TEXT		NOT NULL,
        description             TEXT,
        include_rx              TEXT,
        exclude_rx              TEXT,
        ChangeTimestamp         timestamp with time zone       DEFAULT now()
    );

ALTER TABLE Roles ADD COLUMN IF NOT EXISTS is_default boolean NULL DEFAULT 'false';

-- 3.12.1 drop sketches
ALTER TABLE Roles DROP COLUMN IF EXISTS sketches;

DROP INDEX IF EXISTS roles_uniq CASCADE;
CREATE UNIQUE INDEX IF NOT EXISTS roles_uniq ON Roles (name);

CREATE TABLE IF NOT EXISTS LicenseInfo (
        ExpireTimeStamp         timestamp with time zone    NOT NULL,
        InstallTimestamp        timestamp with time zone,
        Organization            TEXT,
        LicenseType             TEXT,
        LicenseCount            integer
    );

CREATE TABLE IF NOT EXISTS KeysPendingForDeletion (
        HostKey     TEXT
    );


 -- FOR OAUTH

CREATE TABLE IF NOT EXISTS  oauth_clients (client_id VARCHAR(80) NOT NULL, client_secret VARCHAR(80) NOT NULL, redirect_uri VARCHAR(2000) NOT NULL, grant_types VARCHAR(80), scope VARCHAR(100), user_id VARCHAR(80), CONSTRAINT client_id_pk PRIMARY KEY (client_id));
CREATE TABLE IF NOT EXISTS  oauth_access_tokens (access_token VARCHAR(40) NOT NULL, client_id VARCHAR(80) NOT NULL, user_id VARCHAR(255), expires TIMESTAMP NOT NULL, scope VARCHAR(2000), CONSTRAINT access_token_pk PRIMARY KEY (access_token));
CREATE TABLE IF NOT EXISTS  oauth_authorization_codes (authorization_code VARCHAR(40) NOT NULL, client_id VARCHAR(80) NOT NULL, user_id VARCHAR(255), redirect_uri VARCHAR(2000), expires TIMESTAMP NOT NULL, scope VARCHAR(2000), CONSTRAINT auth_code_pk PRIMARY KEY (authorization_code));
CREATE TABLE IF NOT EXISTS  oauth_refresh_tokens (refresh_token VARCHAR(40) NOT NULL, client_id VARCHAR(80) NOT NULL, user_id VARCHAR(255), expires TIMESTAMP NOT NULL, scope VARCHAR(2000), CONSTRAINT refresh_token_pk PRIMARY KEY (refresh_token));
--CREATE TABLE IF NOT EXISTS  oauth_users (username VARCHAR(255) NOT NULL, password VARCHAR(2000), first_name VARCHAR(255), last_name VARCHAR(255), CONSTRAINT username_pk PRIMARY KEY (username));
CREATE TABLE IF NOT EXISTS  oauth_scopes (scope TEXT, is_default BOOLEAN);
CREATE TABLE IF NOT EXISTS  oauth_jwt (client_id VARCHAR(80) NOT NULL, subject VARCHAR(80), public_key VARCHAR(2000), CONSTRAINT client_id_pk_jwt PRIMARY KEY (client_id));




CREATE OR REPLACE FUNCTION delete_expired_access() RETURNS TRIGGER AS $_$
BEGIN
    DELETE FROM oauth_access_tokens WHERE expires <  now() AT TIME ZONE 'UTC';
    DELETE FROM oauth_refresh_tokens WHERE expires < now() AT TIME ZONE 'UTC';
    RETURN NULL;
END $_$ LANGUAGE 'plpgsql';

DROP TRIGGER IF EXISTS delete_expired_access_token ON oauth_access_tokens;
CREATE TRIGGER delete_expired_access_token AFTER INSERT ON oauth_access_tokens FOR EACH ROW EXECUTE PROCEDURE delete_expired_access();

CREATE TABLE IF NOT EXISTS external_roles_map (
        external_role                    TEXT		NOT NULL,
        internal_role                TEXT,
        ChangeTimestamp         timestamp with time zone       DEFAULT now()
    );

CREATE UNIQUE INDEX IF NOT EXISTS external_roles_uniq ON external_roles_map (external_role);


CREATE TABLE IF NOT EXISTS rbac_permissions (
    "alias" character varying(100) PRIMARY KEY,
    "group" character varying(50),
    "name" character varying(100),
    "description" character varying(200),
    "application" character varying(50),
    "allowed_by_default" boolean DEFAULT false NOT NULL
);
COMMENT ON TABLE rbac_permissions IS 'This table stores RBAC permissions';

CREATE TABLE IF NOT EXISTS rbac_role_permission (
    "role_id" character varying NOT NULL,
    "permission_alias" character varying NOT NULL,
    CONSTRAINT "rbac_role_permission_role_id_permission_alias" UNIQUE ("role_id", "permission_alias"),
    CONSTRAINT "rbac_role_permission_permission_alias_fkey" FOREIGN KEY (permission_alias) REFERENCES rbac_permissions(alias) ON UPDATE CASCADE ON DELETE CASCADE,
    CONSTRAINT "rbac_role_permission_role_id_fkey" FOREIGN KEY (role_id) REFERENCES roles(name) ON UPDATE CASCADE ON DELETE CASCADE
);
COMMENT ON TABLE rbac_role_permission IS 'This table associates roles to permissions in a 1-to-many relationship';

CREATE OR REPLACE FUNCTION set_default_permissions() RETURNS TRIGGER AS $_$
BEGIN
    INSERT INTO rbac_role_permission (permission_alias, role_id) (
    SELECT alias as permission_alias, NEW.name::text as role_id
    FROM rbac_permissions
    WHERE allowed_by_default = true
    );
    RETURN NULL;
END $_$ LANGUAGE 'plpgsql';
COMMENT ON FUNCTION set_default_permissions IS 'This function assigns allowed by default permissions to new roles';

DROP TRIGGER IF EXISTS set_default_permissions_trigger ON roles;
CREATE TRIGGER set_default_permissions_trigger AFTER INSERT ON roles FOR EACH ROW EXECUTE PROCEDURE set_default_permissions();

CREATE OR REPLACE FUNCTION insert_default_permission() RETURNS TRIGGER AS $_$
DECLARE
    role record;
BEGIN
    FOR role IN
        EXECUTE 'SELECT name FROM roles WHERE name NOT IN (''admin'', ''cf_remoteagent'')'
    LOOP
        IF NEW.allowed_by_default  = TRUE THEN
            EXECUTE 'INSERT INTO rbac_role_permission (permission_alias, role_id) ' ||
                    'VALUES ($1, $2) ' ||
                    'ON CONFLICT (role_id, permission_alias) DO NOTHING' USING NEW.alias, role.name;
        END IF;
    END LOOP;
    RETURN NULL;
END $_$ LANGUAGE 'plpgsql';
COMMENT ON FUNCTION insert_default_permission IS 'This function assigns allowed by default permissions to existing roles';

DROP TRIGGER IF EXISTS insert_default_permission_trigger ON rbac_permissions;
CREATE TRIGGER insert_default_permission_trigger AFTER INSERT OR UPDATE ON rbac_permissions FOR EACH ROW EXECUTE PROCEDURE insert_default_permission();


CREATE TABLE IF NOT EXISTS remote_hubs (
    id bigserial  PRIMARY KEY,
    hostkey text,
    ui_name character varying(70) UNIQUE,
    api_url text NULL,
    target_state character varying(20) NOT NULL,
    transport json NOT NULL,
    role character varying(50) NOT NULL
);

COMMENT ON TABLE remote_hubs IS 'This table stores federated reporting remote hub';


CREATE TABLE  IF NOT EXISTS federated_reporting_settings (
    "key" character varying DEFAULT 'false' PRIMARY KEY,
    "value" text
);

COMMENT ON TABLE federated_reporting_settings IS 'This table stores federated reporting settings';


CREATE TABLE IF NOT EXISTS "inventory_aliases" (
    "inventory_attribute" text NOT NULL,
    "alias" text NOT NULL
);
CREATE UNIQUE INDEX IF NOT EXISTS inventory_aliases_unique ON inventory_aliases ("inventory_attribute", "alias");
COMMENT ON TABLE inventory_aliases IS 'This table stores inventory attributes aliases';


DO $$
BEGIN
    IF NOT EXISTS (SELECT 1 FROM pg_type WHERE typname = 'authentication_types') THEN
        CREATE TYPE authentication_types AS ENUM ('password', 'private_key');
    END IF;
END$$;

CREATE TABLE IF NOT EXISTS ssh_keys (
    "id" bigserial PRIMARY KEY,
    "public_key" text NOT NULL,
    "private_key" text NOT NULL,
    "generated_at" timestamp with time zone DEFAULT now(),
    "generated_by" text  NOT NULL
);

COMMENT ON TABLE ssh_keys IS 'This table stores generated ssh keys';

CREATE TABLE IF NOT EXISTS build_projects (
    "id" bigserial  PRIMARY KEY,
    "repository_url" TEXT,
    "branch" TEXT,
    "name" TEXT,
    "authentication_type" authentication_types,
    "username" TEXT,
    "password" TEXT,
    "ssh_private_key" TEXT,
    "ssh_key_id" integer REFERENCES ssh_keys ON DELETE SET NULL DEFAULT NULL,
    "created_at" timestamp with time zone DEFAULT now(),
    "pushed_at" timestamp with time zone DEFAULT NULL,
    "is_local" BOOLEAN DEFAULT FALSE
);

COMMENT ON TABLE build_projects IS 'This table stores build application projects';

DO $$
BEGIN
  IF EXISTS(SELECT * FROM information_schema.columns WHERE table_name='build_projects' and column_name='is_empty')
  THEN
    ALTER TABLE "build_projects" RENAME "is_empty" TO "is_local";
  END IF;
END $$;

ALTER TABLE build_projects ADD COLUMN IF NOT EXISTS is_deployed_locally BOOLEAN DEFAULT FALSE;
ALTER TABLE build_projects ADD COLUMN IF NOT EXISTS action TEXT DEFAULT NULL;

CREATE TABLE IF NOT EXISTS cfbs_requests (
    "id" bigserial  PRIMARY KEY,
    "request_name" TEXT NOT NULL,
    "arguments" JSONB,
    "created_at" timestamp with time zone DEFAULT now(),
    "finished_at" timestamp with time zone NULL,
    "response" JSONB
 );

COMMENT ON TABLE cfbs_requests IS 'This table stores cfbs requests and responses';

CREATE OR REPLACE FUNCTION process_cfbs_requests() RETURNS TRIGGER AS $$
BEGIN
IF (TG_OP = 'UPDATE') THEN
    IF (NEW.finished_at IS NOT NULL) THEN
        PERFORM pg_notify('cfbs_request_done', NEW.id::text);
    END IF;
ELSIF (TG_OP = 'INSERT') THEN
    PERFORM pg_notify('new_cfbs_request', NEW.id::text);
END IF;

RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE TRIGGER cfbs_requests_insert_update_trigger
AFTER INSERT OR UPDATE ON cfbs_requests
FOR EACH ROW EXECUTE FUNCTION process_cfbs_requests();

CREATE TABLE IF NOT EXISTS build_modules (
    "name" TEXT,
    "readme" TEXT NULL,
    "description" TEXT NULL,
    "version" TEXT NOT NULL,
    "author" JSONB DEFAULT NULL,
    "updated" timestamp with time zone NULL,
    "downloads" integer DEFAULT 0,
    "repo" TEXT NULL,
    "documentation" TEXT NULL,
    "website" TEXT NULL,
    "subdirectory" TEXT NULL,
    "commit" TEXT,
    "dependencies" JSONB DEFAULT NULL,
    "tags" JSONB DEFAULT NULL,
    "versions" JSONB DEFAULT NULL,
    "latest" BOOLEAN DEFAULT FALSE
);

COMMENT ON TABLE build_modules IS 'This table stores build modules';

ALTER TABLE build_modules ADD COLUMN IF NOT EXISTS ts_vector tsvector
GENERATED ALWAYS AS
(
  setweight(to_tsvector('simple', build_modules.name), 'A') ||
  setweight(to_tsvector('simple', coalesce(build_modules.description, '')), 'B')
) STORED;

COMMENT ON COLUMN build_modules.ts_vector IS 'Generated ts_vector column based on id and description.';

CREATE INDEX IF NOT EXISTS build_modules_ts_vector_idx ON build_modules USING GIN (ts_vector);

CREATE UNIQUE INDEX IF NOT EXISTS build_modules_unique_idx ON build_modules (name, version);

CREATE TABLE IF NOT EXISTS failed_login_attempts (
    ip_address VARCHAR(50) PRIMARY KEY,
    attempts INT NOT NULL DEFAULT 0,
    last_attempt TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    timeout_until TIMESTAMP DEFAULT NULL,
    timeout_seconds INT DEFAULT 0
);

COMMENT ON TABLE failed_login_attempts IS 'This table stores unsuccessful authentication attempts.';
