<?php

class CfUsers
{
    /**
     * @var PDO
     */
    private $dbConnection;

    /**
     * @var array
     */
    private $user = false;

    /**
     * @var string
     */
    private $adminRole = 'admin';

    /**
     * @var string
     */
    private $username = false;
    private $email = false;
    private $roles = [];



    public function __construct($username, $email = '', $roles = [])
    {
        $this->dbConnection = CfPdo::getInstance()->getConnection();
        $this->username = $username;
        $this->email = $email;
        $this->roles = $roles;
        $this->user = $this->getUser();
    }

    /**
     * @return bool
     */
    public function checkUserExistingAndCreateIfNot()
    {
        if (!$this->user) {
            $this->createUser();
        }

        return (boolean) $this->user;
    }


    /**
     * @return mixed
     */
    public function getUser()
    {
        $statement = $this->dbConnection->prepare('SELECT * FROM users WHERE username ILIKE :username');
        $statement->bindParam(':username', $this->username, PDO::PARAM_STR);
        $statement->execute();
        return $statement->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * @return bool
     */
    public function createUser()
    {
        $rolesList = '{' . implode(',', $this->roles) . '}';
        $sth = $this->dbConnection->prepare('INSERT INTO "users" 
                                     ("username", "name", "email", "external", "active", "roles", "changetimestamp") 
                                     VALUES (:username, :username, :email, \'1\', \'0\', :roles, \'now()\')');
        $sth->bindParam(':username', $this->username, PDO::PARAM_STR);
        $sth->bindParam(':email', $this->email, PDO::PARAM_STR);
        $sth->bindParam(':roles', $rolesList, PDO::PARAM_STR);
        return $sth->execute();
    }

    /**
     * @return mixed
     */
    public function getUserRoles()
    {
        /**
         * postgres returns TEXT[] as {item1,item2,item3}
         * see https://www.postgresql.org/docs/current/arrays.html#ARRAYS-IO
         *
         *   The array output routine will put double quotes around element values
         *   if they are empty strings, contain curly braces, delimiter characters,
         *   double quotes, backslashes, or white space, or match the word NULL. 
         *
         * So we must remove double quotes to get the true value in those cases.
         */
        return $this->user != false && array_key_exists('roles', $this->user) ?
            explode(',', str_replace(['{', '}', '"'], '', $this->user['roles'])) :
            false;
    }

    public function syncLdapRoles($settings, $rolesFromLdap, $isFirstLogin)
    {
        // role syncing should be turned on and roles from LDAP shouldn't be empty
        if (isset($settings['ldap_enable_role_syncing']) && $settings['ldap_enable_role_syncing'] == 'true' && sizeof($rolesFromLdap) > 0) {
            if ($isFirstLogin || $settings['ldap_perform_sync_on_login']) {
                $roleModel = new CfRole();
                $rolesListToSync = json_decode(
                    json_decode($settings['ldap_roles_list_to_sync'])
                );

                // text[] item surrounded by " when a role contains spaces. isn't our case, we don't allow any space, just to be sure
                $rolesFromProfile = $this->getUserRoles() ?
                    array_map(function ($item) { return trim($item, '"'); }, $this->getUserRoles()) :
                    [];


                //get roles for syncing
                $allowedRolesFromLdap = array_filter(
                    array_map(function ($item) use ($rolesListToSync) {
                        // role should be in roles list to sync
                        return in_array($item['name'], $rolesListToSync) ? $item['name'] : null;
                    }, $roleModel->getRoles($rolesFromLdap)),
                    'strlen'
                );

                // find missing roles in profile
                $rolesToAssign = array_diff($allowedRolesFromLdap, $rolesFromProfile);

                // if ldap_remove_role_on_syncing is on then find roles to unassign
                $rolesToUnassign = $settings['ldap_remove_role_on_syncing'] == 'true' ?
                    array_diff(
                        array_intersect($rolesListToSync, $rolesFromProfile),
                        $allowedRolesFromLdap
                    ) : [];

                $finalRoles = array_filter(
                    array_diff(
                        array_merge($rolesToAssign, $rolesFromProfile),
                        $rolesToUnassign
                    ), 'strlen');

                $this->dbConnection
                    ->prepare('UPDATE users SET roles = ? WHERE username = ?')
                    ->execute(['{' . implode(',', $finalRoles) . '}', $this->username]);
            }
        }
    }

    /**
     * @return bool
     */
    public function isAdminRole()
    {
        return (boolean)in_array($this->adminRole, $this->getUserRoles());
    }

    /**
     * @return mixed
     */
    public function getEmail()
    {
        return $this->user != false && in_array('email', $this->user) ? $this->user['email'] : false;
    }

    /**
     * @return mixed
     */
    public function getUserName()
    {
        return $this->username ;
    }

    /**
     * @return mixed
     */
    public function getActive()
    {
        return $this->user != false && in_array('active', $this->user) ? $this->user['active'] : false;
    }

    public function isExternal()
    {
        return $this->user != false && in_array('external', $this->user) ? $this->user['external'] : false;
    }
    
    public function setPasswordExpirationTime(?int $hoursOffset): bool
    {
        if ($hoursOffset !== null) {
            $currentDateTime = new DateTime();
            $currentDateTime->add(new DateInterval("PT{$hoursOffset}H"));
            $expirationTime = $currentDateTime->format('Y-m-d H:i:s');
        } else {
            $expirationTime = null;
        }
        
        $sth = $this->dbConnection->prepare('UPDATE "users" SET password_expiration_time = :expirationTime WHERE username = :username');
        $sth->bindParam(':expirationTime', $expirationTime, PDO::PARAM_STR);
        $sth->bindParam(':username', $this->username, PDO::PARAM_STR);
        return $sth->execute();
    }

    public function isPasswordExpired(): bool
    {
        $statement = $this->dbConnection->prepare('SELECT (now() > password_expiration_time) as is_password_expired FROM users WHERE username ILIKE :username');
        $statement->bindParam(':username', $this->username, PDO::PARAM_STR);
        $statement->execute();
        $row = $statement->fetch(PDO::FETCH_ASSOC);
        return $row['is_password_expired'] === true;
    }
}
