<?php
/**
 * @uri /oauth2/token
 */

class Oauth extends CfBaseResource
{

    function post($request)
    {
        $response = new Response($request);
        $data = Utils::getValidJsonData($request->data);
        
        $CfLoginAttempts = new CfLoginAttempts(ipAddress: $_SERVER['REMOTE_ADDR']);
        $timeout = $CfLoginAttempts->timeOutBeforeNextAttempt();

        if ($timeout !== null) {
            $response->code = Response::TOO_MANY_REQUESTS;
            $response->body = "We have detected multiple unsuccessful login attempts, please try again after $timeout seconds.";
            return $response;
        }
        
        $server = Utils::getOauthServer();
        $oAuthResponse = $server->handleTokenRequest(OAuth2\Request::createFromGlobals());
        $response->body = $oAuthResponse->getResponseBody('json');
        $response->code = $oAuthResponse->getStatusCode();

        foreach ($oAuthResponse->getHttpHeaders() as $key => $value)
        {
            $response->addHeader($key, $value);
        }

        if ($response->code === Response::OK && (new CfUsers($data->username))->isPasswordExpired()) {
            $response->code = Response::FORBIDDEN;
            $response->body = "Your password has expired. Please contact the administrator to request a new password.";
            return $response;
        } else if ($response->code == Response::UNAUTHORIZED) {
            try {
                $CfLoginAttempts->logFailedAttempt();
            }  catch (PDOException $exception) {
                $response->code = Response::INTERNALSERVERERROR;
                $response->body = 'An internal server error occurred. Please contact your server administrator.';
                syslog(LOG_ERR, $exception->getTraceAsString());
                return $response;
            }
        }
        
        return $response;
    }
    
    function get($request)
    {
        $response = new Response($request);
        $this->accessRequired();

        $permissions = (new CfRBAC())->getPermissionsByRoles((new CfUsers($this->username))->getUserRoles());
        $server = Utils::getOauthServer();
        $accessTokenStorage = $server->getStorage('access_token');
        $tokens = RbacAccessService::isActionAllowed('all-users-tokens.get', $permissions) ? 
            $accessTokenStorage->getActiveTokens() : 
            $accessTokenStorage->getActiveTokensByUserId($this->username);
        // decode the json from PGSQL else we will encode it twice when encoding to output
        array_walk($tokens, function (&$value, $index)
        {
            $value['details'] = json_decode($value['details'], true);
        });
        $response->code = Response::OK;
        $response->body = json_encode($tokens);
        return $response;
    }
}

/**
 * @uri /oauth2/token/revoke
 */
class OauthRevoke extends CfBaseResource
{

    function post($request)
    {
        $this->accessRequired();
        $response = new Response($request);
        $data = Utils::getValidJsonData($request->data);
        $deleteToken = $data->tokens;
        $permissions = (new CfRBAC())->getPermissionsByRoles((new CfUsers($this->username))->getUserRoles());

        if (empty($deleteToken))
        {
            throw new Exception("Error Processing Request. tokens parameter is missing or empty", 1);
        }
       
        syslog(LOG_DEBUG, 'DELETING ACCESS TOKEN:: ' . implode(',', $deleteToken));
        $server = Utils::getOauthServer();
        $accessTokenStorage = $server->getStorage('access_token');
        
        if (RbacAccessService::isActionAllowed('all-users-tokens.revoke', $permissions)) {
            $accessTokenStorage->unsetAccessToken($deleteToken);
        } else {
            // if a user does not have permissions to revoke all users tokens then pass its username as the second parameter
            // which will allow to delete only current user tokens
            $accessTokenStorage->unsetAccessToken($deleteToken, $this->username);
        }
        
        $response->code = Response::OK;
        $response->body = null;
        return $response;
    }
}

