<?php
require_once 'AlertSQL.php';
class SoftwareUpdateAlert extends AlertSQL
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Return SQL string with condition for software update
     *
     *
     * @param <CF_RULE> $rule
     * @return string
     */
    private function _getSoftwareUpdateConditions($rule)
    {
        $tmp = [];

        $condition = $rule->softwareUpdateConditions;

        $tmp[] = " PatchReportType='AVAILABLE' ";

        if (!empty($condition['patchname'])) {
            $escapedPatchname = $this->ci->db->escape_str($condition['patchname']);
            if (isset($condition['condition']) && $condition['condition'] == 'is') {
                $tmp[] = "patchname ILIKE '" . $escapedPatchname . "'";
            } else {
                $tmp[] = "patchname ILIKE '%" . $escapedPatchname . "%'";
            }
        }

        if (!empty($condition['patcharchitecture'])) {
            $escapedPatcharch = $this->ci->db->escape_str($condition['patcharchitecture']);
            $tmp[] = "patcharchitecture ILIKE '" . $escapedPatcharch . "'";
        }

        $WHERE_CONDITION = '';
        if (!empty($tmp)) {
            $WHERE_CONDITION = 'WHERE ' . implode(' AND ', $tmp);
        }

        return $WHERE_CONDITION;
    }

    /**
     * Return status info for software updates alerts.
     *
     * @param <CF_ALERT> $alert
     * @param <CF_RULE> $rule
     * @param <array> $includes
     * @param <array> $excludes
     * @return <array>
     */

    public function getSoftwareUpdateStatus($alert, $rule, $includes, $excludes)
    {
        $failHosts = $this->utils_getCountFailHosts($alert, $rule, $includes, $excludes);

        $totalHosts = $this->ci->host_model->getHostCountByContext($alert->username, $includes, $excludes);

        $alertStatus['status'] = $failHosts > 0 ? 'fail' : 'success';
        $alertStatus['failHosts'] = $failHosts;
        $alertStatus['totalHosts'] = $totalHosts;

        return $alertStatus;
    }

    /**
     * Return SQL string for hostkeys
     * Used by dashboard widget
     *
     * @param <CF_RULE> $rule
     * @return type
     */
    public function getStatSQLStringForSoftwareUpdate($rule)
    {
        $WHERE_CONDITION = $this->_getSoftwareUpdateConditions($rule);
        $SQL =
            'SELECT DISTINCT s_up.hostkey
                FROM SoftwareUpdates s_up ' . $WHERE_CONDITION;
        return $SQL;
    }

    /**
     * Return count of fail hosts
     *
     * @param <CF_ALERT> $alert
     * @param <CF_RULE> $rule
     * @return <int>
     */

    // NOTE: for future refactoring.
    // getCountFailHosts  and getSoftwareUpdateStatus use different includes, so in the getSoftwareUpdateStatus we can not use utils_getIncludesFromAlert function to get includes list,
    // because in this case includes is a combination of all includes from group of alerts.

    public function getCountFailHosts($alert, $rule)
    {
        $includes = $this->utils_getIncludesFromAlert($alert);
        $excludes = $this->utils_getExcludesFromAlert($alert);
        $failHosts = $this->utils_getCountFailHosts($alert, $rule, $includes, $excludes);

        return $failHosts;
    }

    // see note for getCountFailHosts function about WHY includes is send as param and not taken from $alert
    // utils_getCountFailHosts() used by dashboard widget and alerts overview
    private function utils_getCountFailHosts($alert, $rule, $includes = [], $excludes = [])
    {
        $WHERE_CONDITION = $this->_getSoftwareUpdateConditions($rule);

        $SQL = "SELECT COUNT(*) AS failHost
                FROM (
                    SELECT DISTINCT s_up.hostkey
                    FROM softwareUpdates s_up
                    $WHERE_CONDITION
                ) AS c_query";

        try {
            $result = $this->ci->advancedreports_model->runQuery(
                $alert->username,
                $SQL,
                '',
                '',
                0,
                0,
                $includes,
                $excludes,
            );
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), $e->getMessage() . ' ' . $e->getFile() . ' line:' . $e->getLine());
            throw $e;
        }

        if (isset($result['rows']) && isset($result['rows'][0])) {
            $failHosts = $result['rows'][0][0];
        } else {
            // result if nothing found - Assume success for now
            $failHosts = 0;
        }

        return $failHosts;
    }

    /**
     * Return list of hosts for alerts. Note, we MUST have hostkey here to set proper context
     *
     * @param <CF_ALERT> $alert
     * @param <CF_RULE> $rule
     * @param <int> $limit
     * @return <array>
     */
    public function getFailHostsList_ForSoftwareUpdateAlerts($alert, $rule, $limit = 5)
    {
        $WHERE_CONDITION = $this->_getSoftwareUpdateConditions($rule);

        $SQL =
            'SELECT h.hostkey, h.hostname, aggregated.c
                FROM hosts h
                INNER JOIN (
                  SELECT s_up.hostkey, count(s_up.PatchName) AS "c"
                  FROM SoftwareUpdates s_up
                  ' .
            $WHERE_CONDITION .
            '
                  GROUP BY s_up.hostkey
                ) aggregated ON h.hostkey = aggregated.hostkey
                ORDER BY aggregated.c DESC';

        $SQL .= ' LIMIT ' . (int) $limit;

        $includes = $this->utils_getIncludesFromAlert($alert);
        $excludes = $this->utils_getExcludesFromAlert($alert);

        $result = $this->ci->advancedreports_model->runQuery(
            $alert->username,
            $SQL,
            '',
            '',
            0,
            $limit,
            $includes,
            $excludes,
        );
        return $result['rows'];
    }

    /**
     * Return a link to the report without adding include/exclude or hostkey.
     *
     *
     * @param <CF_RULE> $rule
     * @param string $hostkey
     * @return type
     */
    public function getLinkToSoftwareUpdatesReport($rule, $hostkey)
    {
        $WHERE_CONDITION = $this->_getSoftwareUpdateConditions($rule);

        if (!empty($hostkey)) {
            $escapedHostkey = $this->ci->db->escape_str($hostkey);
            if (empty($WHERE_CONDITION)) {
                $WHERE_CONDITION = 'WHERE s_up.Hostkey = \'' . $escapedHostkey . '\'';
            } else {
                $WHERE_CONDITION .= ' AND s_up.Hostkey = \'' . $escapedHostkey . '\'';
            }
        }

        // create link str to use it in report
        $link =
            'SELECT
                        h.hostname AS "Host name",
                        s_up.patchname AS "Package name",
                        array_agg(s.softwareversion) AS "Version installed",
                        s_up.patchversion AS "Version available",
                        s_up.changetimestamp AS "Discovered at"
                    FROM
                        hosts h
                    INNER
                        JOIN softwareupdates s_up ON s_up.hostkey = h.hostkey
                    LEFT
                        JOIN software s ON s.hostkey = s_up.hostkey
                        AND s.softwarename = s_up.patchname
                        AND s.softwarearchitecture = s_up.patcharchitecture
                    ' .
            $WHERE_CONDITION .
            '
                    GROUP BY
                        h.hostkey,
                        h.hostname,
                        s_up.patchname,
                        s_up.patchversion,
                        s_up.changetimestamp';

        return rawurlencode(base64_encode($link));
    }

    /**
     * Return SQL string for hostlist with count. Note, we need hostkey in order to build proper include context
     *
     * @param <CF_RULE> $rule
     * @return string
     */
    public function get_SQLSTRING_HostlistAndCountAvailableUpdates($rule)
    {
        $WHERE_CONDITION = $this->_getSoftwareUpdateConditions($rule);

        $SQL =
            'SELECT h.hostkey AS "Host key", h.hostname AS "Host name", aggregated.c AS "Updates triggering", inventory_new.values->>\'OS\' as "OS"
                FROM hosts h
                INNER JOIN (
                  SELECT s_up.hostkey, count(s_up.PatchName) AS "c"
                  FROM SoftwareUpdates s_up
                  ' .
            $WHERE_CONDITION .
            '
                  GROUP BY s_up.hostkey
                ) aggregated ON h.hostkey = aggregated.hostkey
                LEFT JOIN inventory_new ON inventory_new.hostkey = h.hostkey
                ORDER BY aggregated.c DESC';

        return $SQL;
    }
}
