<?php

namespace AiChat\Models;

use LLMConfigEntity;
use LLMProviderType;
use LLPhant\AnthropicConfig;
use LLPhant\Chat\AnthropicChat;
use LLPhant\Chat\ChatInterface;
use LLPhant\Chat\MistralAIChat;
use LLPhant\Chat\OllamaChat;
use LLPhant\Chat\OpenAIChat;
use LLPhant\GeminiOpenAIConfig;
use LLPhant\MistralAIConfig;
use LLPhant\OllamaConfig;
use LLPhant\OpenAIConfig;

class LlmFactory
{
    public const modelOptions = ['temperature' => 0];
    public static function create(LLMConfigEntity $config): ChatInterface
    {
        return match ($config->provider) {
            LLMProviderType::OPENAI => self::createOpenAiLlm($config),
            LLMProviderType::ANTHROPIC => self::createAntropicLlm($config),
            LLMProviderType::GEMINI => self::createGeminiLlm($config),
            LLMProviderType::OLLAMA => self::creatOllamaLlm($config),
            LLMProviderType::OPEANAILIKE => self::creatOpenAiLikeLlm($config),
            LLMProviderType::MISTRAL => self::createMistralLlm($config),
        };
    }

    private static function createOpenAiLlm(LLMConfigEntity $config): ChatInterface
    {
        $openAIConfig = new OpenAIConfig(model: $config->model, apiKey: $config->token, modelOptions: self::modelOptions);
        return new OpenAIChat($openAIConfig);
    }

    private static function createAntropicLlm(LLMConfigEntity $config): ChatInterface
    {
        $anthropicConfig = new AnthropicConfig(model: $config->model, apiKey: $config->token, modelOptions: self::modelOptions);
        return new AnthropicChat($anthropicConfig);
    }

    private static function createGeminiLlm(LLMConfigEntity $config): ChatInterface
    {
        $geminiConfig = new GeminiOpenAIConfig();
        $geminiConfig->apiKey = $config->token;
        $geminiConfig->model = $config->model;
        $geminiConfig->modelOptions = self::modelOptions;
        return new OpenAIChat($geminiConfig);
    }

    private static function createMistralLlm(LLMConfigEntity $config): ChatInterface
    {
        $mistralConfig = new MistralAIConfig(model: $config->model, apiKey: $config->token, modelOptions: self::modelOptions);
        return new MistralAIChat($mistralConfig);
    }

    private static function creatOllamaLlm(LLMConfigEntity $config): ChatInterface
    {
        $ollamaConfig = new OllamaConfig();
        $ollamaConfig->apiKey = $config->token;
        $ollamaConfig->model = $config->model;
        $ollamaConfig->url = $config->base_url;
        $ollamaConfig->modelOptions = self::modelOptions;
        return new OllamaChat($ollamaConfig);
    }


    private static function creatOpenAiLikeLlm(LLMConfigEntity $config): ChatInterface
    {
        $openAIConfig = new OpenAIConfig(model: $config->model, apiKey: $config->token, url: $config->base_url, modelOptions: self::modelOptions);
        return new OpenAIChat($openAIConfig);
    }
}
