import { last } from '../utils.js';
const INDENT_TYPE_TOP_LEVEL = 'top-level';
const INDENT_TYPE_BLOCK_LEVEL = 'block-level';
/**
 * Manages indentation levels.
 *
 * There are two types of indentation levels:
 *
 * - BLOCK_LEVEL : increased by open-parenthesis
 * - TOP_LEVEL : increased by RESERVED_CLAUSE words
 */
export default class Indentation {
    /**
     * @param {string} indent A string to indent with
     */
    constructor(indent) {
        this.indent = indent;
        this.indentTypes = [];
    }
    /**
     * Returns indentation string for single indentation step.
     */
    getSingleIndent() {
        return this.indent;
    }
    /**
     * Returns current indentation level
     */
    getLevel() {
        return this.indentTypes.length;
    }
    /**
     * Increases indentation by one top-level indent.
     */
    increaseTopLevel() {
        this.indentTypes.push(INDENT_TYPE_TOP_LEVEL);
    }
    /**
     * Increases indentation by one block-level indent.
     */
    increaseBlockLevel() {
        this.indentTypes.push(INDENT_TYPE_BLOCK_LEVEL);
    }
    /**
     * Decreases indentation by one top-level indent.
     * Does nothing when the previous indent is not top-level.
     */
    decreaseTopLevel() {
        if (this.indentTypes.length > 0 && last(this.indentTypes) === INDENT_TYPE_TOP_LEVEL) {
            this.indentTypes.pop();
        }
    }
    /**
     * Decreases indentation by one block-level indent.
     * If there are top-level indents within the block-level indent,
     * throws away these as well.
     */
    decreaseBlockLevel() {
        while (this.indentTypes.length > 0) {
            const type = this.indentTypes.pop();
            if (type !== INDENT_TYPE_TOP_LEVEL) {
                break;
            }
        }
    }
}
//# sourceMappingURL=Indentation.js.map