<?php

namespace Build\Models;

use Build\Exceptions\InvalidCfbsRequestException;

class CfbsRequestsModel
{
    private \PDO $dbConnection;
    private $table = 'cfbs_requests';

    const INIT_PROJECT_REQUEST = 'init_project';
    const UPDATE_PROJECT_REQUEST = 'update_project';
    const DELETE_PROJECT_REQUEST = 'delete_project';
    const SYNC_PROJECT_REQUEST = 'sync_project';
    const REFRESH_PROJECT_REQUEST = 'refresh_project';
    const LOCAL_DEPLOY_REQUEST = 'local_deploy';

    const ADD_MODULES_REQUEST = 'add_modules';
    const UPDATE_MODULES_REQUEST = 'update_modules';
    const REMOVE_MODULES_REQUEST = 'remove_modules';

    const GET_MODULE_INPUT_REQUEST = 'get_module_input';
    const SET_MODULE_INPUT_REQUEST = 'set_module_input';

    const REQUEST_STATUS_OK = 'ok';
    const REQUEST_STATUS_ERROR = 'error';
    const REQUEST_DONE_CHANNEL = 'cfbs_request_done';

    const PUSH_SYNC_ACTION = 'push';
    const REBASE_SYNC_ACTION = 'rebase';
    const FORCE_PULL_SYNC_ACTION = 'force-pull';
    const FORCE_REBASE_SYNC_ACTION = 'force-rebase';

    const SYNC_ACTIONS = [
        self::PUSH_SYNC_ACTION,
        self::REBASE_SYNC_ACTION,
        self::FORCE_PULL_SYNC_ACTION,
        self::FORCE_REBASE_SYNC_ACTION
    ];

    public function __construct()
    {
        $this->dbConnection = \CfSettings::getInstance()->getConnection();
    }

    public function create(string $request, array $arguments): int
    {
        $sql = "INSERT INTO {$this->table} (request_name, arguments) VALUES (:request_name, :arguments)";
        $this->dbConnection->prepare($sql)->execute(['request_name' => $request, 'arguments' => json_encode($arguments)]);
        return (int) $this->dbConnection->lastInsertId();
    }

    public function get(int $id)
    {
        $stmt = $this->dbConnection->prepare("SELECT * FROM {$this->table} WHERE id = :id");
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }

    /**
     * @throws InvalidCfbsRequestException
     */
    public function processRequestResponse(int $requestId, $timeOut = 60): array
    {
        if ($this->listenWhileRequestDone($requestId, $timeOut)) {
            $request = $this->get($requestId);
            $response = json_decode($request['response'], $associative = true);
            if ($response['status'] !== self::REQUEST_STATUS_ERROR) {
                return $response;
            } else {
                throw new InvalidCfbsRequestException(json_encode($response));
            }
        } else {
            throw new InvalidCfbsRequestException('Timeout exceeded. Please try again.');
        }
    }

    private function listenWhileRequestDone(int $requestId, $timeOut = 60): bool
    {
        $this->dbConnection->exec(sprintf('LISTEN %s', self::REQUEST_DONE_CHANNEL));

        for ($i = 0; $i <= $timeOut; $i++) { // wait $timeOut seconds
            while ($result = $this->dbConnection->pgsqlGetNotify(\PDO::FETCH_ASSOC)) {
                if (isset($result['payload']) && intval($result['payload']) === $requestId) {
                    return true;
                }
            }
            sleep(1);
        }

        return false;
    }
}
