import { createSlice } from '@reduxjs/toolkit';
import {
    getConfig,
    updateConfig,
    createConfig,
    deleteConfig,
    updatePreferences,
    getPreferences
} from './thunks';

type Provider = 'openai' | 'gemini' | 'mistral' | 'ollama' | 'anthropic' | 'openai_like';
export interface LLMConfig {
  id?: number;
  token?: string;
  provider: Provider | '';
  model: string | null;
  base_url: string | null;
}
interface Preferences {
    allowLlmViewAccessToAttributesNames: boolean,
    disableAI: boolean
}

export interface LlmConfigState {
    config: LLMConfig | null;
    preferences: Preferences | null;
    loading: boolean;
    error: string | null;
}

const initialState: LlmConfigState = {
    config: null,
    preferences: null,
    loading: false,
    error: null
};

const llmConfigSlice = createSlice({
    name: 'llm',
    initialState,
    reducers: {
    },
    extraReducers: (builder) => {
        // Get config
        builder
            .addCase(getConfig.pending, (state) => {
                state.loading = true;
            })
            .addCase(getConfig.fulfilled, (state, action) => {
                state.loading = false;
                state.config = action.payload;
            })
            .addCase(getConfig.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to fetch config.';
            });

        // Create config
        builder
            .addCase(createConfig.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(createConfig.fulfilled, (state, action) => {
                state.loading = false;
            })
            .addCase(createConfig.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to create config.';
            });

        // Update config
        builder
            .addCase(updateConfig.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(updateConfig.fulfilled, (state, action) => {
                state.loading = false;
            })
            .addCase(updateConfig.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to update config.';
            });

        // Update prefences
        builder
            .addCase(updatePreferences.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(updatePreferences.fulfilled, (state, { payload = {} }) => {
                state.loading = false;
                state.preferences = { ...state.preferences, ...payload };
            })
            .addCase(updatePreferences.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to update preferences.';
            });

        // Get prefences
        builder
            .addCase(getPreferences.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(getPreferences.fulfilled, (state, { payload }) => {
                state.loading = false;
                state.preferences = {
                    allowLlmViewAccessToAttributesNames: !!payload?.allowLlmViewAccessToAttributesNames,
                    disableAI: !!payload?.disableAI
                };
            })
            .addCase(getPreferences.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to update preferences.';
            });

        // Delete config
        builder
            .addCase(deleteConfig.pending, (state) => {
                state.loading = true;
                state.error = null;
            })
            .addCase(deleteConfig.fulfilled, (state, action) => {
                state.loading = false;
                state.config = null;
            })
            .addCase(deleteConfig.rejected, (state, action) => {
                state.loading = false;
                state.error = action.error.message || 'Failed to delete config.';
            });
    },
});

export default llmConfigSlice.reducer;
