import {
  DEVICE_MESSAGE_PROTOCOLS,
  DEVICE_MESSAGE_TYPES,
  TIMEOUTS,
  apiUrl
} from "./chunk-7C4SQBEE.js";
import "./chunk-2DYA3X77.js";
import "./chunk-FAJ3I2UQ.js";
import "./chunk-ZKAH3APE.js";
import "./chunk-HFOTAPIJ.js";

// src/sockethook.ts
import { useCallback, useEffect, useRef, useState } from "react";
import { blobToString, byteArrayToString } from "@northern.tech/utils/helpers";
import msgpack5 from "msgpack5";
import Cookies from "universal-cookie";
var cookies = new Cookies();
var MessagePack = msgpack5();
var useSession = ({ onClose, onHealthCheckFailed, onMessageReceived, onNotify, onOpen, token }) => {
  const [sessionId, setSessionId] = useState();
  const healthcheckTimeout = useRef();
  const socketRef = useRef();
  const sendMessage = useCallback(({ typ, body, props }) => {
    if (!socketRef.current) {
      return;
    }
    const proto_header = { proto: DEVICE_MESSAGE_PROTOCOLS.Shell, typ, sid: socketRef.current.sessionId, props };
    const encodedData = MessagePack.encode({ hdr: proto_header, body });
    socketRef.current.send(encodedData);
  }, []);
  const close = useCallback(() => {
    if (!socketRef.current || socketRef.current?.readyState !== WebSocket.OPEN) {
      return;
    }
    sendMessage({ typ: DEVICE_MESSAGE_TYPES.Stop, body: {}, props: {} });
    socketRef.current.close();
    setSessionId();
  }, [sendMessage]);
  const healthcheckFailed = useCallback(() => {
    onHealthCheckFailed();
    close();
  }, [close, onHealthCheckFailed]);
  const onSocketMessage = useCallback(
    (event) => blobToString(event.data).then((data) => {
      const {
        hdr: { props = {}, proto, sid, typ },
        body
      } = MessagePack.decode(data);
      if (proto !== DEVICE_MESSAGE_PROTOCOLS.Shell) {
        return;
      }
      switch (typ) {
        case DEVICE_MESSAGE_TYPES.New: {
          if (props.status == WebSocket.CLOSING) {
            onNotify(`Error: ${byteArrayToString(body)}`);
            setSessionId();
            return close();
          } else {
            clearTimeout(healthcheckTimeout.current);
            healthcheckTimeout.current = setTimeout(healthcheckFailed, 65 * TIMEOUTS.oneSecond);
            socketRef.current.sessionId = sid;
            return setSessionId(sid);
          }
        }
        case DEVICE_MESSAGE_TYPES.Shell:
          return onMessageReceived(body);
        case DEVICE_MESSAGE_TYPES.Stop:
          return close();
        case DEVICE_MESSAGE_TYPES.Ping: {
          if (healthcheckTimeout.current) {
            clearTimeout(healthcheckTimeout.current);
          }
          sendMessage({ typ: DEVICE_MESSAGE_TYPES.Pong });
          const timeout = parseInt(props.timeout);
          if (timeout > 0) {
            healthcheckTimeout.current = setTimeout(healthcheckFailed, timeout * TIMEOUTS.oneSecond);
          }
          return;
        }
        default:
          break;
      }
    }),
    [close, healthcheckFailed, onMessageReceived, onNotify, sendMessage]
  );
  const onSocketError = useCallback(
    (error) => {
      onNotify(`WebSocket error: ${error.message}`);
      close();
      clearTimeout(healthcheckTimeout.current);
    },
    [close, onNotify]
  );
  const onSocketOpen = useCallback(() => {
    sendMessage({ typ: DEVICE_MESSAGE_TYPES.New, props: {} });
    onOpen(true);
  }, [onOpen, sendMessage]);
  const onSocketClose = useCallback(
    (e) => {
      console.log("closing");
      onClose(e);
      close();
    },
    [close, onClose]
  );
  const connect = useCallback(
    (deviceId) => {
      const uri = `wss://${window.location.host}${apiUrl.v1}/deviceconnect/devices/${deviceId}/connect`;
      setSessionId();
      cookies.set("JWT", token, { path: "/", secure: true, sameSite: "strict", maxAge: 5 });
      try {
        socketRef.current = new WebSocket(uri);
      } catch (error) {
        console.log(error);
      }
    },
    [token]
  );
  useEffect(() => {
    if (!socketRef.current) {
      return;
    }
    socketRef.current.addEventListener("close", onSocketClose);
    socketRef.current.addEventListener("error", onSocketError);
    socketRef.current.addEventListener("message", onSocketMessage);
    socketRef.current.addEventListener("open", onSocketOpen);
    return () => {
      socketRef.current.removeEventListener("close", onSocketClose);
      socketRef.current.removeEventListener("error", onSocketError);
      socketRef.current.removeEventListener("message", onSocketMessage);
      socketRef.current.removeEventListener("open", onSocketOpen);
    };
  }, [onSocketClose, onSocketError, onSocketMessage, onSocketOpen, socketRef.current?.readyState]);
  return [connect, sendMessage, close, socketRef.current?.readyState ?? WebSocket.CLOSED, sessionId];
};
export {
  useSession
};
//# sourceMappingURL=sockethook.js.map