<?php

namespace AiChat\Controllers;

use AiChat\Models\LlmFactory;
use AiChat\Models\Text2SqlModel;
use CfLLMConfig;
use Exception;
use LLMConfigEntity;
use Response;
use ResponseException;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Validation;

/**
 * @uri /ai/text-to-sql
 */
class TextToSqlResource extends \CfProtectedResource
{
    /**
     * @rbacName Send message to AI chat
     * @rbacGroup AI Chat
     * @rbacAlias ai.chat.message
     * @rbacAllowedByDefault
     */
    public function post($request): Response
    {
        $requestData = \Utils::getValidJsonData($request->data, returnArray: true);
        self::validateRequest($requestData);

        $response = new Response($request);
        $cfLLMConfig =  new CfLLMConfig(\CfSettings::getInstance()->getConnection());

        $configs = $cfLLMConfig->list();

        if (!$configs) {
            throw new ResponseException(
                'Add an LLM configuration before using the chat.',
                Response::UNPROCESSABLE_ENTITY
            );
        }

        /**
         * @var LLMConfigEntity
         */
        $llmConfig = $configs[0];
        $chat = LlmFactory::create($llmConfig);

        $text2sql = new Text2SqlModel($chat);
        try {
            $reply = $text2sql->getReplay($requestData['question'], $requestData['history'] ?? null);
        } catch (Exception $exception) {
            $response->code = Response::BADREQUEST;
            $response->body = $exception->getMessage();
            return $response;
        }

        $return = $reply->toArray();
        $return['meta'] = ['provider' => $llmConfig->provider, 'model' => $llmConfig->model];

        $response->body = json_encode(value: $return);
        return $response;
    }

    private static function validateRequest(array $requestData)
    {
        $validator = Validation::createValidator();
        $constraints = new Assert\Collection(
            fields: [
                'question' => new Assert\Required([
                    new Assert\Type(['type' => 'string']),
                    new Assert\Length(['max' => MAX_VALUE_LENGTH])
                ]),
                'history' => new Assert\Optional([
                    new Assert\Type(['type' => 'array'])
                ])
            ]
        );

        $issues = $validator->validate($requestData, $constraints);
        if (count($issues) > 0) {
            $response = [
                'success' => false,
                'errors' => []
            ];
            foreach ($issues as $issue) {
                $response['errors'][] = [
                    'field' => $issue->getPropertyPath(),
                    'message' => $issue->getMessage()
                ];
            }
            throw new \InvalidArgumentException(json_encode($response));
        }
    }
}
