<?php

class AuditLogModel
{
    private const TABLE = 'audit_log';
    public const DEFAULT_SORT_COLUMN = 'time';
    public const DEFAULT_SORT_DIRECTION = 'DESC';
    public const ALLOWED_SORT_COLUMNS = ['time', 'actor', 'action', 'object_id', 'object_name', 'object_type'];
    public const SELECT_COLUMNS = ['id', 'time', 'actor', 'action', 'object_type', 'object_id', 'object_name', 'details'];

    private \PDO $dbConnection;

    public function __construct()
    {
        $this->dbConnection = \CfSettings::getInstance()->getConnection();
    }

    public function create(array $data): string
    {
        $sql = "INSERT INTO " . self::TABLE . " 
                (actor, object_type, object_id, object_name, action, details, ip_address) 
                VALUES (:actor, :object_type, :object_id, :object_name, :action, :details, :ip_address)";

        $stmt = $this->dbConnection->prepare($sql);
        $stmt->execute($data);
        return $this->dbConnection->lastInsertId();
    }

    public function item(int $id): array
    {
        $query = "SELECT " . implode(', ', self::SELECT_COLUMNS) . " FROM " . self::TABLE . " WHERE id = ?";
        $stmt = $this->dbConnection->prepare($query);
        $stmt->execute([$id]);
        $log = $stmt->fetch(\PDO::FETCH_ASSOC);
        $log['details'] = json_decode($log['details']);
        return $log;
    }

    public function list(AuditLogListArgumentsDto $arguments): array
    {
        $query = $this->buildListQuery($arguments);
        $stmt = $this->dbConnection->prepare($query['sql']);
        $stmt->execute($query['params']);

        $logs = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        $logs = array_map(fn ($log) => [...$log, 'details' => json_decode($log['details'])], $logs);

        return [
            'data' => $logs,
            'meta' => [
                'count' => count($logs),
                'page' => floor($arguments->offset / $arguments->limit) + 1,
                'timestamp' => time(),
                'total' => $this->getTotalLogsCount($query['whereSql'], $query['whereParams'])
            ]
        ];
    }

    public function namesByType(string $type): array
    {
        $query = "SELECT distinct object_name FROM " . self::TABLE . " WHERE object_type = ?";
        $stmt = $this->dbConnection->prepare($query);
        $stmt->execute([$type]);
        return array_column($stmt->fetchAll(\PDO::FETCH_ASSOC), 'object_name');
    }

    public function actors(): array
    {
        $query = "SELECT distinct actor FROM " . self::TABLE;
        $stmt = $this->dbConnection->prepare($query);
        $stmt->execute();
        return array_column($stmt->fetchAll(\PDO::FETCH_ASSOC), 'actor');
    }

    private function buildListQuery(AuditLogListArgumentsDto $arguments): array
    {
        $whereConditions = [];
        $whereParams = [];

        if ($arguments->actor !== null) {
            $whereConditions[] = 'actor = :actor';
            $whereParams['actor'] = $arguments->actor;
        }

        if ($arguments->objectType !== null) {
            $whereConditions[] = 'object_type = :object_type';
            $whereParams['object_type'] = $arguments->objectType;
        }

        if ($arguments->action !== null) {
            $whereConditions[] = 'action = :action';
            $whereParams['action'] = $arguments->action;
        }

        if ($arguments->objectId === 'none') {
            $whereConditions[] = 'object_id IS NULL';
        } elseif ($arguments->objectId !== null) {
            $whereConditions[] = 'object_id = :object_id';
            $whereParams['object_id'] = $arguments->objectId;
        }

        if ($arguments->objectName !== null) {
            $whereConditions[] = 'object_name = :object_name';
            $whereParams['object_name'] = $arguments->objectName;
        }

        if ($arguments->created_after) {
            $whereConditions[] = 'time >= to_timestamp(:created_after)';
            $whereParams['created_after'] = $arguments->created_after;
        }

        if ($arguments->created_before) {
            $whereConditions[] = 'time <= to_timestamp(:created_before)';
            $whereParams['created_before'] = $arguments->created_before;
        }

        $whereSql = $whereConditions ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

        $sql = "SELECT " . implode(', ', self::SELECT_COLUMNS) . " FROM " . self::TABLE . " 
                $whereSql 
                ORDER BY {$arguments->sort_column} {$arguments->sort_direction} 
                LIMIT :limit OFFSET :offset";

        return [
            'sql' => $sql,
            'params' => [...$whereParams, 'limit' => $arguments->limit, 'offset' => $arguments->offset],
            'whereSql' => $whereSql,
            'whereParams' => $whereParams
        ];
    }

    private function getTotalLogsCount(string $where, array $params): int
    {
        $sql = "SELECT COUNT(*) FROM " . self::TABLE . " $where";
        $stmt = $this->dbConnection->prepare($sql);
        $stmt->execute($params);
        return (int) $stmt->fetchColumn();
    }
}
