import {
  commonErrorFallback,
  commonErrorHandler,
  createAppAsyncThunk
} from "./chunk-V3QEONX7.js";
import {
  general_api_default
} from "./chunk-RS37Y2PQ.js";
import {
  actions_default
} from "./chunk-F26IKSML.js";
import {
  actions,
  sliceName
} from "./chunk-OOICURFL.js";
import {
  formatReleases
} from "./chunk-KC7SKI3U.js";
import {
  getSearchEndpoint
} from "./chunk-SCZLNFB5.js";
import {
  getReleasesById
} from "./chunk-MSOXMRF7.js";
import {
  convertDeviceListStateToFilters,
  progress
} from "./chunk-UE5N5VO3.js";
import {
  DEVICE_LIST_DEFAULTS,
  SORTING_OPTIONS,
  TIMEOUTS,
  deploymentsApiUrl,
  deploymentsApiUrlV2,
  emptyFilter,
  headerNames
} from "./chunk-7C4SQBEE.js";
import {
  ARTIFACT_GENERATION_TYPE
} from "./chunk-2DYA3X77.js";

// src/releasesSlice/thunks.ts
import { customSort, deepCompare, duplicateFilter, extractSoftwareItem } from "@northern.tech/utils/helpers";
import { isCancel } from "axios";
import { v4 as uuid } from "uuid";
var { setSnackbar, initUpload, uploadProgress, cleanUpUpload } = actions_default;
var { page: defaultPage, perPage: defaultPerPage } = DEVICE_LIST_DEFAULTS;
var sortingDefaults = { direction: SORTING_OPTIONS.desc, key: "modified" };
var flattenRelease = (release, stateRelease) => {
  const updatedArtifacts = release.artifacts?.sort(customSort(true, "modified")) || [];
  const { artifacts, deviceTypes, modified } = updatedArtifacts.reduce(
    (accu, item) => {
      accu.deviceTypes.push(...item.device_types_compatible);
      const stateArtifact = stateRelease.artifacts?.find((releaseArtifact) => releaseArtifact.id === item.id) || {};
      accu.modified = accu.modified ? accu.modified : item.modified;
      accu.artifacts.push({
        ...stateArtifact,
        ...item
      });
      return accu;
    },
    { artifacts: [], deviceTypes: [], modified: void 0 }
  );
  return { ...stateRelease, ...release, artifacts, device_types_compatible: deviceTypes.filter(duplicateFilter), modified };
};
var reduceReceivedReleases = (releases, stateReleasesById) => releases.reduce((accu, release) => {
  const stateRelease = stateReleasesById[release.name] || {};
  accu[release.name] = flattenRelease(release, stateRelease);
  return accu;
}, {});
var findArtifactIndexInRelease = (releases, id) => Object.values(releases).reduce(
  (accu, item) => {
    const index = item.artifacts.findIndex((releaseArtifact) => releaseArtifact.id === id);
    if (index > -1) {
      accu = { release: item, index };
    }
    return accu;
  },
  { release: null, index: -1 }
);
var getArtifactInstallCount = createAppAsyncThunk(`${sliceName}/getArtifactInstallCount`, (id, { dispatch, getState }) => {
  const { release, index } = findArtifactIndexInRelease(getReleasesById(getState()), id);
  if (!release || index === -1) {
    return;
  }
  const releaseArtifacts = [...release.artifacts];
  const artifact = releaseArtifacts[index];
  const { key, name, version } = extractSoftwareItem(artifact.artifact_provides) ?? {};
  const attribute = `${key}${name ? `.${name}` : ""}.version`;
  const { filterTerms } = convertDeviceListStateToFilters({
    filters: [{ ...emptyFilter, key: attribute, value: version, scope: "inventory" }]
  });
  return general_api_default.post(getSearchEndpoint(getState()), {
    page: 1,
    per_page: 1,
    filters: filterTerms,
    attributes: [{ scope: "identity", attribute: "status" }]
  }).catch((err) => commonErrorHandler(err, `Retrieving artifact installation count failed:`, dispatch, commonErrorFallback)).then(({ headers }) => {
    const foundRelease = findArtifactIndexInRelease(getReleasesById(getState()), id);
    const index2 = foundRelease.index;
    let release2 = foundRelease.release;
    if (!release2 || index2 === -1) {
      return;
    }
    const installCount = Number(headers[headerNames.total]);
    const releaseArtifacts2 = [...release2.artifacts];
    releaseArtifacts2[index2] = { ...releaseArtifacts2[index2], installCount };
    release2 = {
      ...release2,
      artifacts: releaseArtifacts2
    };
    return dispatch(actions.receiveRelease(release2));
  });
});
var getArtifactUrl = createAppAsyncThunk(
  `${sliceName}/getArtifactUrl`,
  (id, { dispatch, getState }) => general_api_default.get(`${deploymentsApiUrl}/artifacts/${id}/download`).then((response) => {
    const foundRelease = findArtifactIndexInRelease(getReleasesById(getState()), id);
    const index = foundRelease.index;
    let release = foundRelease.release;
    if (!release || index === -1) {
      return dispatch(getReleases());
    }
    const releaseArtifacts = [...release.artifacts];
    releaseArtifacts[index] = {
      ...releaseArtifacts[index],
      url: response.data.uri
    };
    release = {
      ...release,
      artifacts: releaseArtifacts
    };
    return dispatch(actions.receiveRelease(release));
  })
);
var createArtifact = createAppAsyncThunk(`${sliceName}/createArtifact`, ({ file, meta }, { dispatch }) => {
  const formData = Object.entries(meta).reduce((accu, [key, value]) => {
    if (Array.isArray(value)) {
      accu.append(key, value.join(","));
    } else if (value instanceof Object) {
      accu.append(key, JSON.stringify(value));
    } else {
      accu.append(key, value);
    }
    return accu;
  }, new FormData());
  formData.append("type", ARTIFACT_GENERATION_TYPE.SINGLE_FILE);
  formData.append("file", file);
  const uploadId = uuid();
  const cancelSource = new AbortController();
  return Promise.all([
    dispatch(setSnackbar("Generating artifact")),
    dispatch(initUpload({ id: uploadId, upload: { name: file.name, size: file.size, progress: 0, cancelSource } })),
    general_api_default.upload(
      `${deploymentsApiUrl}/artifacts/generate`,
      formData,
      (e) => dispatch(uploadProgress({ id: uploadId, progress: progress(e) })),
      cancelSource.signal
    )
  ]).then(() => {
    setTimeout(() => {
      dispatch(getReleases());
      dispatch(selectRelease(file.name));
    }, TIMEOUTS.oneSecond);
    return Promise.resolve(dispatch(setSnackbar({ message: "Upload successful", autoHideDuration: TIMEOUTS.fiveSeconds })));
  }).catch((err) => {
    if (isCancel(err)) {
      return dispatch(setSnackbar({ message: "The artifact generation has been cancelled", autoHideDuration: TIMEOUTS.fiveSeconds }));
    }
    return commonErrorHandler(err, `Artifact couldn't be generated.`, dispatch);
  }).finally(() => dispatch(cleanUpUpload(uploadId)));
});
var uploadArtifact = createAppAsyncThunk(`${sliceName}/uploadArtifact`, ({ file, meta }, { dispatch }) => {
  const formData = new FormData();
  formData.append("size", file.size.toString());
  formData.append("description", meta.description);
  formData.append("artifact", file);
  const uploadId = uuid();
  const cancelSource = new AbortController();
  return Promise.all([
    dispatch(setSnackbar("Uploading artifact")),
    dispatch(initUpload({ id: uploadId, upload: { name: file.name, size: file.size, progress: 0, cancelSource } })),
    general_api_default.upload(
      `${deploymentsApiUrl}/artifacts`,
      formData,
      (e) => dispatch(uploadProgress({ id: uploadId, progress: progress(e) })),
      cancelSource.signal
    )
  ]).then(() => {
    const tasks = [
      dispatch(setSnackbar({ message: "Upload successful", autoHideDuration: TIMEOUTS.fiveSeconds })),
      dispatch(getReleases())
    ];
    if (meta.name) {
      tasks.push(dispatch(selectRelease(meta.name)));
    }
    return Promise.all(tasks);
  }).catch((err) => {
    if (isCancel(err)) {
      return dispatch(setSnackbar({ message: "The upload has been cancelled", autoHideDuration: TIMEOUTS.fiveSeconds }));
    }
    return commonErrorHandler(err, `Artifact couldn't be uploaded.`, dispatch);
  }).finally(() => dispatch(cleanUpUpload(uploadId)));
});
var cancelFileUpload = createAppAsyncThunk(`${sliceName}/cancelFileUpload`, (id, { dispatch, getState }) => {
  const { [id]: current } = getState().app.uploadsById;
  current.cancelSource.abort();
  return Promise.resolve(dispatch(cleanUpUpload(id)));
});
var editArtifact = createAppAsyncThunk(
  `${sliceName}/editArtifact`,
  ({ id, body }, { dispatch, getState }) => general_api_default.put(`${deploymentsApiUrl}/artifacts/${id}`, body).catch((err) => commonErrorHandler(err, `Artifact details couldn't be updated.`, dispatch)).then(() => {
    const state = getState();
    const { release, index } = findArtifactIndexInRelease(getReleasesById(state), id);
    if (!release || index === -1) {
      return dispatch(getReleases());
    }
    const updatedRelease = {
      ...release,
      artifacts: release.artifacts.map((artifact, i) => i === index ? { ...artifact, description: body.description || "" } : artifact)
    };
    return Promise.all([
      dispatch(actions.receiveRelease(updatedRelease)),
      dispatch(setSnackbar({ message: "Artifact details were updated successfully.", autoHideDuration: TIMEOUTS.fiveSeconds, action: "" })),
      dispatch(getRelease(release.name)),
      dispatch(selectRelease(release.name))
    ]);
  })
);
var removeArtifact = createAppAsyncThunk(
  `${sliceName}/removeArtifact`,
  (id, { dispatch, getState }) => general_api_default.delete(`${deploymentsApiUrl}/artifacts/${id}`).then(() => {
    const state = getState();
    const { release, index } = findArtifactIndexInRelease(getReleasesById(state), id);
    if (!release || index === -1) {
      return dispatch(getReleases());
    }
    const releaseArtifacts = [...release.artifacts];
    releaseArtifacts.splice(index, 1);
    if (!releaseArtifacts.length) {
      const { releasesList } = state.releases;
      const releaseIds = releasesList.releaseIds.filter((id2) => release.name !== id2);
      return Promise.all([
        dispatch(actions.removeRelease(release.name)),
        dispatch(
          setReleasesListState({
            releaseIds,
            searchTotal: releasesList.searchTerm ? releasesList.searchTotal - 1 : releasesList.searchTotal,
            total: releasesList.total - 1
          })
        )
      ]);
    }
    return Promise.all([
      dispatch(setSnackbar({ message: "Artifact was removed", autoHideDuration: TIMEOUTS.fiveSeconds, action: "" })),
      dispatch(actions.receiveRelease(release))
    ]);
  }).catch((err) => commonErrorHandler(err, `Error removing artifact:`, dispatch))
);
var removeRelease = createAppAsyncThunk(
  `${sliceName}/removeRelease`,
  (releaseId, { dispatch, getState }) => Promise.all(getReleasesById(getState())[releaseId].artifacts.map(({ id }) => dispatch(removeArtifact(id)))).then(() => dispatch(selectRelease(null)))
);
var removeReleases = createAppAsyncThunk(`${sliceName}/removeReleases`, (releaseIds, { dispatch, getState }) => {
  const deleteRequests = releaseIds.reduce((accu, releaseId) => {
    const releaseArtifacts = getReleasesById(getState())[releaseId].artifacts;
    accu.push(releaseArtifacts.map(({ id }) => dispatch(removeArtifact(id))));
    return accu;
  }, []);
  return Promise.all(deleteRequests);
});
var selectRelease = createAppAsyncThunk(`${sliceName}/selectRelease`, (release, { dispatch }) => {
  const name = (release && typeof release === "object" ? release.name : release) || null;
  const tasks = [dispatch(actions.selectedRelease(name))];
  if (name) {
    tasks.push(dispatch(getRelease(name)));
  }
  return Promise.all(tasks);
});
var setReleasesListState = createAppAsyncThunk(
  `${sliceName}/setReleasesListState`,
  (selectionState, { dispatch, getState }) => {
    const currentState = getState().releases.releasesList;
    const nextState = {
      ...currentState,
      ...selectionState,
      sort: { ...currentState.sort, ...selectionState.sort }
    };
    const tasks = [];
    const { isLoading: currentLoading, ...currentRequestState } = currentState;
    const { isLoading: selectionLoading, ...selectionRequestState } = nextState;
    if (!deepCompare(currentRequestState, selectionRequestState)) {
      nextState.isLoading = true;
      tasks.push(dispatch(getReleases(nextState)).finally(() => dispatch(setReleasesListState({ isLoading: false }))));
    }
    tasks.push(dispatch(actions.setReleaseListState(nextState)));
    return Promise.all(tasks);
  }
);
var releaseListRetrieval = (config) => {
  const { searchTerm = "", page = defaultPage, perPage = defaultPerPage, sort = sortingDefaults, selectedTags = [], type = "" } = config;
  const { key: attribute, direction } = sort;
  const filterQuery = formatReleases({ pageState: { searchTerm, selectedTags } });
  const updateType = type ? `update_type=${type}` : "";
  const sorting = attribute ? `sort=${attribute}:${direction}`.toLowerCase() : "";
  return general_api_default.get(
    `${deploymentsApiUrlV2}/deployments/releases?${[`page=${page}`, `per_page=${perPage}`, filterQuery, updateType, sorting].filter((i) => i).join("&")}`
  );
};
var deductSearchState = (receivedReleases, config, total, state) => {
  let releaseListState = { ...state.releasesList };
  const { searchTerm, searchOnly, sort = {}, selectedTags = [], type } = config;
  const flattenedReleases = Object.values(receivedReleases).sort(customSort(sort.direction === SORTING_OPTIONS.desc, sort.key));
  const releaseIds = flattenedReleases.map((item) => item.name);
  const isFiltering = !!(selectedTags.length || type || searchTerm);
  if (searchOnly) {
    releaseListState = { ...releaseListState, searchedIds: releaseIds };
  } else {
    releaseListState = {
      ...releaseListState,
      releaseIds,
      searchTotal: isFiltering ? total : state.releasesList.searchTotal,
      total: !isFiltering ? total : state.releasesList.total
    };
  }
  return releaseListState;
};
var getReleases = createAppAsyncThunk(`${sliceName}/getReleases`, (passedConfig = {}, { dispatch, getState }) => {
  const config = { ...getState().releases.releasesList, ...passedConfig };
  return releaseListRetrieval(config).then(({ data: receivedReleases = [], headers = {} }) => {
    const total = headers[headerNames.total] ? Number(headers[headerNames.total]) : 0;
    const state = getState().releases;
    const flatReleases = reduceReceivedReleases(receivedReleases, state.byId);
    const combinedReleases = { ...state.byId, ...flatReleases };
    const tasks = [dispatch(actions.receiveReleases(combinedReleases))];
    const releaseListState = deductSearchState(receivedReleases, config, total, state);
    tasks.push(dispatch(actions.setReleaseListState(releaseListState)));
    return Promise.all(tasks);
  }).catch((err) => commonErrorHandler(err, `Please check your connection`, dispatch));
});
var getRelease = createAppAsyncThunk(`${sliceName}/getReleases`, async (name, { dispatch, getState }) => {
  const releaseResponse = await general_api_default.get(`${deploymentsApiUrlV2}/deployments/releases/${name}`);
  const { data: release } = releaseResponse;
  if (release) {
    const stateRelease = getReleasesById(getState())[release.name] || {};
    await dispatch(actions.receiveRelease(flattenRelease(release, stateRelease)));
  }
  return Promise.resolve(null);
});
var updateReleaseInfo = createAppAsyncThunk(
  `${sliceName}/updateReleaseInfo`,
  ({ name, info }, { dispatch, getState }) => general_api_default.patch(`${deploymentsApiUrlV2}/deployments/releases/${name}`, info).catch((err) => commonErrorHandler(err, `Release details couldn't be updated.`, dispatch)).then(
    () => Promise.all([
      dispatch(actions.receiveRelease({ ...getReleasesById(getState())[name], ...info, name })),
      dispatch(setSnackbar({ message: "Release details were updated successfully.", autoHideDuration: TIMEOUTS.fiveSeconds, action: "" }))
    ])
  )
);
var setSingleReleaseTags = createAppAsyncThunk(
  `${sliceName}/setSingleReleaseTags`,
  ({ name, tags }, { dispatch, getState }) => general_api_default.put(`${deploymentsApiUrlV2}/deployments/releases/${name}/tags`, tags).then(
    () => Promise.resolve(dispatch(actions.receiveRelease({ ...getReleasesById(getState())[name], name, tags })))
  )
);
var setReleaseTags = createAppAsyncThunk(
  `${sliceName}/setReleaseTags`,
  ({ name, tags = [] }, { dispatch }) => dispatch(setSingleReleaseTags({ name, tags })).catch((err) => commonErrorHandler(err, `Release tags couldn't be set.`, dispatch)).then(() => Promise.resolve(dispatch(setSnackbar({ message: "Release tags were set successfully.", autoHideDuration: TIMEOUTS.fiveSeconds, action: "" }))))
);
var setReleasesTags = createAppAsyncThunk(
  `${sliceName}/setReleasesTags`,
  ({ releases, tags = [] }, { dispatch }) => {
    const addRequests = releases.reduce((accu, release) => {
      accu.push(dispatch(setSingleReleaseTags({ name: release.name, tags: [.../* @__PURE__ */ new Set([...release.tags ? release.tags : [], ...tags])] })));
      return accu;
    }, []);
    return Promise.all(addRequests).catch((err) => commonErrorHandler(err, `Releases couldn't be tagged.`, dispatch)).then(
      () => Promise.resolve(dispatch(setSnackbar({ message: "Releases were tagged successfully.", autoHideDuration: TIMEOUTS.fiveSeconds, action: "" })))
    );
  }
);
var getExistingReleaseTags = createAppAsyncThunk(
  `${sliceName}/getReleaseTags`,
  (_, { dispatch }) => general_api_default.get(`${deploymentsApiUrlV2}/releases/all/tags`).catch((err) => commonErrorHandler(err, `Existing release tags couldn't be retrieved.`, dispatch)).then(({ data: tags }) => Promise.resolve(dispatch(actions.receiveReleaseTags(tags))))
);
var getUpdateTypes = createAppAsyncThunk(
  `${sliceName}/getReleaseTypes`,
  (_, { dispatch }) => general_api_default.get(`${deploymentsApiUrlV2}/releases/all/types`).catch((err) => commonErrorHandler(err, `Existing update types couldn't be retrieved.`, dispatch)).then(({ data: types }) => Promise.resolve(dispatch(actions.receiveReleaseTypes(types))))
);

export {
  getArtifactInstallCount,
  getArtifactUrl,
  createArtifact,
  uploadArtifact,
  cancelFileUpload,
  editArtifact,
  removeArtifact,
  removeRelease,
  removeReleases,
  selectRelease,
  setReleasesListState,
  getReleases,
  getRelease,
  updateReleaseInfo,
  setSingleReleaseTags,
  setReleaseTags,
  setReleasesTags,
  getExistingReleaseTags,
  getUpdateTypes
};
//# sourceMappingURL=chunk-QKPV5RRN.js.map