/*
  Copyright 2024 Northern.tech AS

  This file is part of CFEngine 3 - written and maintained by Northern.tech AS.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  To the extent this program is licensed as part of the Enterprise
  versions of CFEngine, the applicable Commercial Open Source License
  (COSL) may apply to this file if you as a licensee so wish it. See
  included file COSL.txt.
*/

#ifndef CFENGINE_WRITER_H
#define CFENGINE_WRITER_H

/*
 * Abstract "writer".
 *
 * Writes passed data either to
 *   passed FILE*, or
 *   memory buffer
 */

typedef struct Writer_ Writer;

#include <stdio.h> // FILE
#include <stdbool.h> // bool
#include <stdarg.h> // va_list
#include <compiler.h>

/* Set by ./configure allowing us to avoid #include <config.h> here. */
// #undef HAVE_GETOPT_H

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#ifndef _GETOPT_H /* in case something included it from somewhere anyway  */
/* We actually only need the 'struct option' type from the header here so just
 * defined it ourselves if we cannot get it from the header. */
struct option {
    const char *name;
    int has_arg;
    int *flag;
    int val;
};
#endif /* _GETOPT_H */
#endif  /* HAVE_GETOPT_H */

Writer *FileWriter(FILE *);
Writer *StringWriter(void);

size_t WriterWriteF(Writer *writer, const char *fmt, ...) FUNC_ATTR_PRINTF(2, 3);
size_t WriterWriteVF(Writer *writer, const char *fmt, va_list ap) FUNC_ATTR_PRINTF(2, 0);

size_t WriterWrite(Writer *writer, const char *str);
size_t WriterWriteLen(Writer *writer, const char *str, size_t len);
size_t WriterWriteChar(Writer *writer, char c);

size_t StringWriterLength(const Writer *writer);
const char *StringWriterData(const Writer *writer);

void WriterClose(Writer *writer);

/* Returns modifiable string and destroys itself */
char *StringWriterClose(Writer *writer) FUNC_WARN_UNUSED_RESULT;

/* Returns the open file and destroys itself */
FILE *FileWriterDetach(Writer *writer);
/* Commonly used on a FileWriter(stdout), ignoring return; so don't
 * try to warn on unused result ! */

typedef struct
{
    const char *name;
    const char *description;
    const char *usage;
} Description;

typedef struct
{
    const char *name;
    const char *website;
    const char *copyright;
} Component;

void WriterWriteHelp(Writer *w, const Component *component,
                     const struct option options[], const char *const hints[],
                     const Description *commands, bool command_first,
                     bool accepts_file_argument);

#endif
