<?php

/**
 * Class MigrationModel
 */
class MigrationModel extends Cf_Model
{

    private $cfdb;
    private $cfsettings;
    private $db;
    private $cfmp;

    /**
     * @var string $migrationPath Path of folder with migrations
     */
    protected $migrationPath = __DIR__ . '/../migrations';

    /**
     * MigrationModel constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
        $this->cfdb = $this->load->database('cfdb', true);
        $this->cfsettings = $this->load->database('cfsettings', true);
        $this->cfmp = $this->load->database('cfmp', true);
        $this->db = $this->load->database('default', true);
        $this->load->config('migration/sql_commands_set');
        $this->checkTableMigrationExists();
    }

    /**
     * @return bool
     */
    public function migrate()
    {
        $migrationFiles = $this->getMigrationFiles();
        foreach ($migrationFiles as $migration) {
            if ($this->checkMigrationIsExecuted($migration)) {
                continue;
            }
            $migrationObject = $this->getMigrationObject($migration);
            if(isset($migrationObject->useCFDB) && $migrationObject->useCFDB == true){
                $this->db = $this->cfdb;
            }elseif(isset($migrationObject->useCFSettings) && $migrationObject->useCFSettings == true){
                $this->db = $this->cfsettings;
            }elseif(isset($migrationObject->runAsSU) && $migrationObject->runAsSU == true){
                $this->db = $this->cfmp;
            }
            $sql = $migrationObject->up();
            $this->db->trans_start();

            try {
                if (is_array($sql)) {
                    foreach ($sql as $s) {
                        $this->db->query($s);
                    }
                } else {
                    $this->db->query($sql);
                }
                $this->db->trans_commit();
                $this->setMigrationAsCompleted($migration);
                $successMessage = 'Migration: ' . $migration . ' was executed successful.';
                print_r($successMessage . "\n");
                log_message('info', $successMessage);
            } catch (Exception $e) {
                log_message('debug', $e->getMessage());
                print_r($e->getMessage());
                $this->db->trans_rollback();
            }
        }
        return true;
    }

    /**
     * @param int $count
     * @return bool
     */
    public function down($count = 1)
    {

        $this->db->order_by('apply_time', 'DESC');
        $migrationsToDown = $this->db->get('migrations', $count, 0)->result();

        if ($migrationsToDown) {
            foreach ($migrationsToDown as $migration) {
                $migrationObject = $this->getMigrationObject($migration->version);
                $sql = $migrationObject->down();
                $this->db->trans_start();

                try {
                    if (is_array($sql)) {
                        foreach ($sql as $s) {
                            $this->db->query($s);
                        }
                    } else {
                        if ($sql != '') {
                            $this->db->query($sql);
                        }
                    }

                    $this->db->trans_commit();
                    $this->setMigrationAsRolledBack($migration->version);
                    $successMessage = 'Migration: ' . $migration->version . ' was rolled back successful.';
                    print_r($successMessage . "\n");
                    log_message('info', $successMessage);
                } catch (Exception $e) {
                    log_message('debug', $e->getMessage());
                    print_r($e->getMessage());
                    $this->db->trans_rollback();
                }
            }
        }
        return true;
    }

    /**
     * @return array
     */
    private function getMigrationFiles()
    {
        return array_filter(scandir($this->migrationPath), function ($item) {
            if (strstr($item, '.php')) {
                return $item;
            }
        });
    }

    /**
     *
     */
    private function checkTableMigrationExists()
    {

        $sql = $this->config->item('sql_migration_table_exists');
        $res = $this->cfmp->query($sql)->row();

        if (!$res->exists) {
            $sql = $this->config->item('sql_create_migration_table');
            $this->cfmp->query($sql);
            log_message('info', 'migration table was created');
        }
    }

    /**
     * @param $name
     * @return bool
     */
    private function checkMigrationIsExecuted($name)
    {
        $sql = sprintf($this->config->item('sql_check_migration_is_done'), $name);
        if ($this->cfmp->query($sql)->num_rows() > 0) {
            return true;
        }
        return false;
    }

    /**
     * @param $path
     * @return mixed
     */
    private function getMigrationObject($path)
    {
        try {
            include_once $this->migrationPath . '/' . $path;
            $className = basename($path, ".php");
            return new $className();
        } catch (Exception $e) {
            log_message('debug', 'Migration loading error: ' . $e->getMessage());
        }
    }

    /**
     * @param $name
     */
    private function setMigrationAsCompleted($name)
    {
        $sql = sprintf($this->config->item('sql_set_migration_as_completed'), $name);
        $this->cfmp->query($sql);
    }

    /**
     * @param $migration
     */
    private function setMigrationAsRolledBack($migration)
    {
        $data = array(
            'version' => $migration,
        );
        $this->cfmp->delete('migrations', $data);
    }

}