// src/helpers.ts
import dayjs from "dayjs";
import utc from "dayjs/plugin/utc.js";
import pluralize from "pluralize";
import Cookies from "universal-cookie";
dayjs.extend(utc);
var isEncoded = (uri = "") => uri !== decodeURIComponent(uri);
var fullyDecodeURI = (uri) => {
  while (isEncoded(uri)) {
    uri = decodeURIComponent(uri);
  }
  return uri;
};
var isEmpty = (obj) => {
  for (const _ in obj) {
    return false;
  }
  return true;
};
var yes = () => true;
var canAccess = yes;
var versionCompare = (v1, v2) => {
  const partsV1 = `${v1}`.split(".");
  const partsV2 = `${v2}`.split(".");
  for (let index = 0; index < partsV1.length; index++) {
    const numberV1 = partsV1[index];
    const numberV2 = partsV2[index];
    if (numberV1 > numberV2) {
      return 1;
    }
    if (numberV2 > numberV1) {
      return -1;
    }
  }
  return 0;
};
function deepCompare(...args) {
  let i, l, leftChain, rightChain;
  function compare2Objects(x, y) {
    let p;
    if (isNaN(x) && isNaN(y) && typeof x === "number" && typeof y === "number") {
      return true;
    }
    if (x === y) {
      return true;
    }
    if (typeof x === "function" && typeof y === "function" || x instanceof Date && y instanceof Date || x instanceof RegExp && y instanceof RegExp || x instanceof String && y instanceof String || x instanceof Number && y instanceof Number) {
      return x.toString() === y.toString();
    }
    if (!(x instanceof Object && y instanceof Object)) {
      return false;
    }
    if (x.isPrototypeOf(y) || y.isPrototypeOf(x)) {
      return false;
    }
    if (x.constructor !== y.constructor) {
      return false;
    }
    if (x.prototype !== y.prototype) {
      return false;
    }
    if (leftChain.indexOf(x) > -1 || rightChain.indexOf(y) > -1) {
      return false;
    }
    for (p in y) {
      if (y.hasOwnProperty(p) !== x.hasOwnProperty(p) || typeof y[p] !== typeof x[p]) {
        return false;
      }
    }
    for (p in x) {
      if (y.hasOwnProperty(p) !== x.hasOwnProperty(p) || typeof y[p] !== typeof x[p]) {
        return false;
      }
      switch (typeof x[p]) {
        case "object":
        case "function":
          leftChain.push(x);
          rightChain.push(y);
          if (!compare2Objects(x[p], y[p])) {
            return false;
          }
          leftChain.pop();
          rightChain.pop();
          break;
        default:
          if (x[p] !== y[p]) {
            return false;
          }
          break;
      }
    }
    return true;
  }
  if (args.length < 1) {
    return true;
  }
  for (i = 1, l = args.length; i < l; i++) {
    leftChain = [];
    rightChain = [];
    if (!compare2Objects(args[0], args[i])) {
      return false;
    }
  }
  return true;
}
var stringToBoolean = (content) => {
  if (!content) {
    return false;
  }
  const string = content + "";
  switch (string.trim().toLowerCase()) {
    case "true":
    case "yes":
    case "1":
      return true;
    case "false":
    case "no":
    case "0":
    case null:
      return false;
    default:
      return Boolean(string);
  }
};
var toggle = (current) => !current;
var formatTime = (date) => {
  if (date && Object.prototype.toString.call(date) === "[object Date]" && !isNaN(date)) {
    return date.toISOString().slice(0, -1);
  } else if (date) {
    return date.replace(" ", "T").replace(/ /g, "").replace("UTC", "");
  }
};
var customSort = (direction, field) => (a, b) => {
  if (typeof a[field] === "string") {
    const result = a[field].localeCompare(b[field], navigator.language, { sensitivity: "case" });
    return direction ? result * -1 : result;
  }
  if (a[field] > b[field]) return direction ? -1 : 1;
  if (a[field] < b[field]) return direction ? 1 : -1;
  return 0;
};
var duplicateFilter = (item, index, array) => array.indexOf(item) == index;
var attributeDuplicateFilter = (filterableArray, attributeName = "key") => filterableArray.filter(
  (item, index, array) => array.findIndex((filter) => filter[attributeName] === item[attributeName] && filter.scope === item.scope) == index
);
var unionizeStrings = (someStrings, someOtherStrings) => {
  const startingPoint = new Set(someStrings.filter((item) => item.length));
  const uniqueStrings = someOtherStrings.length ? someOtherStrings.reduce((accu, item) => {
    if (item.trim().length) {
      accu.add(item.trim());
    }
    return accu;
  }, startingPoint) : startingPoint;
  return [...uniqueStrings];
};
var getFormattedSize = (bytes) => {
  const suffixes = ["Bytes", "KB", "MB", "GB"];
  const i = Math.floor(Math.log(bytes) / Math.log(1024));
  if (!bytes) {
    return "0 Bytes";
  }
  return `${(bytes / Math.pow(1024, i)).toFixed(2)} ${suffixes[i]}`;
};
var collectAddressesFrom = (devices) => devices.reduce((collector, { attributes = {} }) => {
  const ips = Object.entries(attributes).reduce((accu, [name, value]) => {
    if (name.startsWith("ipv4")) {
      if (Array.isArray(value)) {
        const texts = value.map((text) => text.slice(0, text.indexOf("/")));
        accu.push(...texts);
      } else {
        const text = value.slice(0, value.indexOf("/"));
        accu.push(text);
      }
    }
    return accu;
  }, []);
  collector.push(...ips);
  return collector;
}, []);
var getDemoDeviceAddress = (devices, onboardingApproach) => {
  const defaultVitualizedIp = "10.0.2.15";
  const addresses = collectAddressesFrom(devices);
  const address = addresses.reduce((accu, item) => {
    if (accu && item === defaultVitualizedIp) {
      return accu;
    }
    return item;
  }, null);
  if (!address || onboardingApproach === "virtual" && (navigator.appVersion.indexOf("Win") != -1 || navigator.appVersion.indexOf("Mac") != -1)) {
    return "localhost";
  }
  return address;
};
var detectOsIdentifier = () => {
  if (navigator.appVersion.indexOf("Win") != -1) return "Windows";
  if (navigator.appVersion.indexOf("Mac") != -1) return "MacOs";
  if (navigator.appVersion.indexOf("X11") != -1) return "Unix";
  return "Linux";
};
var standardizePhases = (phases) => phases.map((phase, index) => {
  const standardizedPhase = { batch_size: phase.batch_size, start_ts: index };
  if (phase.delay) {
    standardizedPhase.delay = phase.delay;
    standardizedPhase.delayUnit = phase.delayUnit || "hours";
  }
  if (index === 0) {
    delete standardizedPhase.start_ts;
  }
  return standardizedPhase;
});
var getInstallScriptArgs = ({ isHosted, isPreRelease, hasMonitor }) => {
  let installScriptArgs = "--demo";
  installScriptArgs = isPreRelease ? `${installScriptArgs} -c experimental` : installScriptArgs;
  installScriptArgs = isHosted && hasMonitor ? `${installScriptArgs} --commercial` : installScriptArgs;
  installScriptArgs = isHosted ? `${installScriptArgs} --jwt-token $JWT_TOKEN` : installScriptArgs;
  return installScriptArgs;
};
var getSetupArgs = ({ deviceType = "generic-armv6", ipAddress, isDemoMode, tenantToken, isOnboarding }) => {
  let menderSetupArgs = `--quiet --device-type "${deviceType}"`;
  menderSetupArgs = tenantToken ? `${menderSetupArgs} --tenant-token $TENANT_TOKEN` : menderSetupArgs;
  menderSetupArgs = isDemoMode || isOnboarding ? `${menderSetupArgs} --demo` : `${menderSetupArgs} --retry-poll 300 --update-poll 1800 --inventory-poll 28800`;
  if (isDemoMode) {
    menderSetupArgs = `${menderSetupArgs}${ipAddress ? ` --server-ip ${ipAddress}` : ""}`;
  } else {
    menderSetupArgs = `${menderSetupArgs} --server-url https://${window.location.hostname} --server-cert=""`;
  }
  return menderSetupArgs;
};
var installComponents = "--force-mender-client4";
var getDebConfigurationCode = (props) => {
  const { tenantToken, token, isPreRelease } = props;
  const envVars = tenantToken ? `JWT_TOKEN="${token}"
TENANT_TOKEN="${tenantToken}"
` : "";
  const installScriptArgs = getInstallScriptArgs(props);
  const scriptUrl = isPreRelease ? "https://get.mender.io/staging" : "https://get.mender.io";
  const menderSetupArgs = getSetupArgs(props);
  return `${envVars}wget -O- ${scriptUrl} | sudo bash -s -- ${installScriptArgs} ${installComponents} -- ${menderSetupArgs}`;
};
var getSnackbarMessage = (skipped, done) => {
  pluralize.addIrregularRule("its", "their");
  const skipText = skipped ? `${skipped} ${pluralize("devices", skipped)} ${pluralize("have", skipped)} more than one pending authset. Expand ${pluralize(
    "this",
    skipped
  )} ${pluralize("device", skipped)} to individually adjust ${pluralize("their", skipped)} authorization status. ` : "";
  const doneText = done ? `${done} ${pluralize("device", done)} ${pluralize("was", done)} updated successfully. ` : "";
  return `${doneText}${skipText}`;
};
var extractSoftware = (attributes = {}) => {
  const softwareKeys = Object.keys(attributes).reduce((accu, item) => {
    if (item.endsWith(".version")) {
      accu.push(item.substring(0, item.lastIndexOf(".")));
    }
    return accu;
  }, []);
  return Object.entries(attributes).reduce(
    (accu, item) => {
      if (softwareKeys.some((key) => item[0].startsWith(key))) {
        accu.software.push(item);
      } else {
        accu.nonSoftware.push(item);
      }
      return accu;
    },
    { software: [], nonSoftware: [] }
  );
};
var extractSoftwareItem = (artifactProvides = {}) => {
  const { software } = extractSoftware(artifactProvides);
  return software.reduce((accu, item) => {
    const infoItems = item[0].split(".");
    if (infoItems[infoItems.length - 1] !== "version") {
      return accu;
    }
    accu.push({
      key: infoItems[0],
      name: infoItems.slice(1, infoItems.length - 1).join("."),
      version: item[1],
      nestingLevel: infoItems.length
    });
    return accu;
  }, []).sort((a, b) => a.nestingLevel - b.nestingLevel).reduce((accu, item) => accu ?? item, void 0);
};
var cookies = new Cookies();
var createDownload = (target, filename, token) => {
  const link = document.createElement("a");
  link.setAttribute("href", target);
  link.setAttribute("download", filename);
  link.style.display = "none";
  document.body.appendChild(link);
  cookies.set("JWT", token, {
    path: "/",
    secure: true,
    sameSite: "strict",
    maxAge: 5
  });
  link.click();
  document.body.removeChild(link);
};
var createFileDownload = (content, filename, token) => createDownload("data:text/plain;charset=utf-8," + encodeURIComponent(content), filename, token);
var getISOStringBoundaries = (currentDate) => {
  const date = [currentDate.getUTCFullYear(), `0${currentDate.getUTCMonth() + 1}`.slice(-2), `0${currentDate.getUTCDate()}`.slice(-2)].join("-");
  return { start: `${date}T00:00:00.000`, end: `${date}T23:59:59.999` };
};
var extractErrorMessage = (err, fallback = "") => err.response?.data?.error?.message || err.response?.data?.error || err.error || err.message || fallback;
var preformatWithRequestID = (res, failMsg) => {
  if (failMsg.length > 100) failMsg = `${failMsg.substring(0, 220)}...`;
  try {
    if (res?.data && Object.keys(res.data).includes("request_id")) {
      const shortRequestUUID = res.data["request_id"].substring(0, 8);
      return `${failMsg} [Request ID: ${shortRequestUUID}]`;
    }
  } catch (e) {
    console.log("failed to extract request id:", e);
  }
  return failMsg;
};
var dateRangeToUnix = (startDate = null, endDate = null) => {
  const unixRange = { start: null, end: null };
  const format = "YYYY-MM-DD";
  if (startDate !== null) {
    const start = dayjs.utc(dayjs(startDate).format(format));
    if (start.isValid()) {
      unixRange.start = start.startOf("day").unix();
    }
  }
  if (endDate !== null) {
    const end = dayjs.utc(dayjs(endDate).format(format));
    if (end.isValid()) {
      unixRange.end = end.endOf("day").unix();
    }
  }
  return unixRange;
};
var byteArrayToString = (body) => String.fromCharCode(...body);
var blobToString = (blob) => new Promise((resolve) => {
  const fr = new FileReader();
  fr.onload = () => {
    resolve(fr.result);
  };
  fr.readAsArrayBuffer(blob);
});

export {
  fullyDecodeURI,
  isEmpty,
  yes,
  canAccess,
  versionCompare,
  deepCompare,
  stringToBoolean,
  toggle,
  formatTime,
  customSort,
  duplicateFilter,
  attributeDuplicateFilter,
  unionizeStrings,
  getFormattedSize,
  getDemoDeviceAddress,
  detectOsIdentifier,
  standardizePhases,
  getDebConfigurationCode,
  getSnackbarMessage,
  extractSoftware,
  extractSoftwareItem,
  createDownload,
  createFileDownload,
  getISOStringBoundaries,
  extractErrorMessage,
  preformatWithRequestID,
  dateRangeToUnix,
  byteArrayToString,
  blobToString
};
//# sourceMappingURL=chunk-YKX7CGDK.js.map