import React, { useCallback, useMemo, useState, useRef, useEffect } from 'react';
import {
  Box,
  Button
} from '@mui/material';
import { useDispatch, useSelector } from 'react-redux';

import { CmdbItem, updateItem, getItems, getPolicyConfigurationsIds } from '../store/cmdbSlice/thunks';
import { selectPagination } from '../store/cmdbSlice/selectors';
import { ITEM_TYPES } from '../store/constants';
import { useStyles } from './CmdbCreate';
import ComfirmationDialog from '../../common/ComfirmationDialog';

interface CmdbEditProps {
  item: CmdbItem;
  identifier: string;
  onCancel: () => void;
  setHasUnsavedChanges: (value: boolean) => void;
  actionInterrupted: boolean;
}

interface FormRef {
  validate: () => Promise<boolean>;
}

const CmdbEdit: React.FC<CmdbEditProps> = ({ 
  item, 
  identifier, 
  onCancel, 
  setHasUnsavedChanges, 
  actionInterrupted 
}) => {
  const { classes } = useStyles();
  const [cmdbItem, setCmdbItem] = useState<CmdbItem | undefined>();
  const [originalCmdbItem, setOriginalCmdbItem] = useState<CmdbItem | undefined>();
  const dispatch = useDispatch();
  const formRef = useRef<FormRef | null>(null);
  const pagination = useSelector(selectPagination);

  const selectedType = useMemo(() => {
    return ITEM_TYPES.find(typeItem => typeItem.value === item.type);
  }, [item.type]);

  const FormComponent = useMemo(() => {
    if (!selectedType) return null;

    return selectedType.component({ 
      identifier, 
      setCmdbItem, 
      formRef, 
      item,
      classes 
    });
  }, [selectedType, identifier, setCmdbItem, item, classes]);

  useEffect(() => {
    if (!originalCmdbItem && cmdbItem) {
      setOriginalCmdbItem(cmdbItem);
    }
  }, [cmdbItem, originalCmdbItem]);


  useEffect(() => {
    if (!cmdbItem || !originalCmdbItem) {
      setHasUnsavedChanges(false);
      return;
    }
    
    const hasChanges = JSON.stringify(cmdbItem) !== JSON.stringify(originalCmdbItem);
    setHasUnsavedChanges(hasChanges);
  }, [cmdbItem, originalCmdbItem, setHasUnsavedChanges]);

  const onSave = useCallback(async () => {
    const isValid = await formRef.current.validate();
    if (isValid) {
      await dispatch(updateItem({
        id: String(item.id),
        identifier,
        entry: cmdbItem
      }));

      dispatch(getItems({
        identifier,
        page: pagination.page || 1
      }));

      dispatch(getPolicyConfigurationsIds({ identifier }));
      onCancel();
    }
  }, [cmdbItem, item.id, identifier, pagination.page, dispatch, onCancel]);


  return (
    <>
      <Box className={classes.root}>
        <Box>
          {FormComponent}
        </Box>
        <Box sx={{ mb: 3, mt: 1 }}>
          <Button 
            onClick={onCancel} 
            variant="outlined" 
            sx={{ mr: 1 }} 
            size="small"
          >
            Cancel
          </Button>
          <Button 
            onClick={onSave} 
            variant="contained" 
            color="secondary" 
            size="small"
            disabled={!cmdbItem}
          >
            Save
          </Button>
        </Box>
      </Box>
      <ComfirmationDialog
        open={actionInterrupted}
        title="Unsaved changes"
        confirmText="Save changes"
        cancelText="Continue without saving"
        size="xs"
        message="You're about to leave the editing without saving. All changes will be lost. Do you really want to leave without saving?"
        onClose={onCancel}
        onConfirm={onSave}
      />
    </>
  );
};

export default React.memo(CmdbEdit);
