<?php

namespace Build\Controllers;

use AuditLogActions;
use AuditLogObjectTypes;
use AuditLogFields;
use AuditLogService;
use Build\Entities\ProjectEntity;
use Build\Entities\ProjectEntityValidator;
use Build\Exceptions\InvalidCfbsRequestException;
use Build\Models\CfbsRequestsModel;
use Build\Models\ProjectModel;

class BaseProjects extends \CfProtectedResource
{
    /**
     * @var ProjectModel
     */
    protected $projectModel;

    /**
     * @var CfbsRequestsModel
     */
    protected $cfbsRequestsModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->projectModel = new ProjectModel($this->username);
        $this->cfbsRequestsModel = new CfbsRequestsModel();
    }

    protected function createProject(ProjectEntity $projectEntity, \Response $response, array $cfbsArgs = []) : \Response
    {
        try {
            $id = $this->projectModel->create($projectEntity);
            $requestID = $this->cfbsRequestsModel->create(cfbsRequestsModel::INIT_PROJECT_REQUEST, ['project_id' => $id, ...$cfbsArgs]);
            $response->code = \Response::CREATED;
            $response->body = json_encode(['id' => $id]);
        } catch (\PDOException $exception) {
            syslog(LOG_ERR, 'Database error occurred while adding build project. Err.:' . $exception->getTraceAsString());
            $response->code = \Response::INTERNALSERVERERROR;
            return $response;
        }

        $deleteProjectAndSendCfbsRequest = function ($id) {
            $this->projectModel->delete($id);
            $this->cfbsRequestsModel->create(cfbsRequestsModel::DELETE_PROJECT_REQUEST, ['project_id' => $id]);
        };

        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $id,
                AuditLogFields::OBJECT_NAME => $projectEntity->getName(),
                AuditLogFields::ACTION => AuditLogActions::CREATE,
                AuditLogFields::DETAILS => ["Created Build project #$id.", [
                    'name' => $projectEntity->getName(),
                    'url' => valueOrNA($projectEntity->getRepositoryUrl()),
                    'branch' => valueOrNA($projectEntity->getBranch()),
                    'isLocal' => boolToYesNo($projectEntity->getIsLocal()),
                ]]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $deleteProjectAndSendCfbsRequest($id);
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $deleteProjectAndSendCfbsRequest($id);
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while adding build project.';
            syslog(LOG_ERR, 'An error occurred while adding build project. Err.:' . $exception->getTraceAsString());
        }
        return $response;
    }
}

/**
 * @uri /build/projects
 */
class Projects extends BaseProjects
{
    /**
     * @rbacName Get project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.get
     */
    function get($request): \Response
    {
        $response = new \Response($request);
        $projects = $this->projectModel->list($_REQUEST['limit'] ?? 10, $_REQUEST['skip'] ?? 0);

        $response->code = \Response::OK;
        $response->body = json_encode($projects);

        return $response;
    }

    /**
     * @rbacName Create project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.create
     */
    function post($request): \Response
    {
        $requestData = \Utils::getValidJsonData($request->data);
        $response = new \Response($request);

        $projectEntity = ProjectEntity::fromRequest($requestData);
        ProjectEntityValidator::validateBeforeInsertion($projectEntity);
        $cfbsArgs = $projectEntity->getClassicPolicySet() === true ? ['skip_cfbs_init' => true] : [];

        return $this->createProject($projectEntity, $response, $cfbsArgs);
    }
}


/**
 * @uri /build/projects/:id
 */
class ProjectsItem extends BaseProjects
{
    /**
     * @rbacName Get project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.get
     */
    function get($request, $id): \Response
    {
        $response = new \Response($request);
        $project = $this->projectModel->get((int)$id);

        if ($project) {
            $response->code = \Response::OK;
            $response->body = json_encode($project);
        } else {
            $response->code = \Response::NOTFOUND;
        }

        return $response;
    }

    /**
     * @rbacName Update project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.update
     *
     * To update a project we delete the current project after successfully re-initialized a new one
     * in case when branch or repository are changed
     */
    function patch($request, $updatingProjectId): \Response
    {
        $requestData = \Utils::getValidJsonData($request->data);
        $response = new \Response($request);

        if (!$project = $this->projectModel->get((int)$updatingProjectId)) {
            $response->code = \Response::NOTFOUND;
            return $response;
        }
        $requestProjectEntity = ProjectEntity::fromRequest($requestData);
        ProjectEntityValidator::validateBeforeUpdating($requestProjectEntity);
        
        foreach ($requestProjectEntity->toArray() as $key => $value) {
            // Skip keys that start with 'is_' or key does not exist in the $project array
            if (str_starts_with($key, 'is_') || !array_key_exists($key, $project)) {
                continue;
            }

            // Check for value difference and non-empty values
            if ($project[$key] !== $value && !empty($value)) {
                $changedProperties[] = $key;
            }
        }

        $auditLog = fn() => AuditLogService::register([
            AuditLogFields::ACTOR => $this->username,
            AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
            AuditLogFields::OBJECT_ID => $updatingProjectId,
            AuditLogFields::OBJECT_NAME => $project['name'],
            AuditLogFields::ACTION => AuditLogActions::UPDATE,
            AuditLogFields::DETAILS => ["Updated Build project #$updatingProjectId.", ['changed properties' => $changedProperties]]
        ]);
        
        if ( // if branch and repository are not changed then just update project row in the DB
            ($requestProjectEntity->getBranch() == null || $requestProjectEntity->getBranch() == $project['branch']) &&
            ($requestProjectEntity->getRepositoryUrl() == null || $requestProjectEntity->getRepositoryUrl() == $project['repository_url'])
        ) {
            $this->projectModel->update((int)$updatingProjectId, $requestProjectEntity);
            $response->code = \Response::OK;
            $response->body = json_encode(['id' => $updatingProjectId]);
            $auditLog();
            return $response;
        }

        // get data from the DB about a project that is going to be updated
        $originalProjectEntity = ProjectEntity::fromArray($this->projectModel->getWithSensitiveData((int)$updatingProjectId));
        $originalProjectEntity->combine($requestProjectEntity);

        $createdResponse = $this->createProject($originalProjectEntity, $response);
        if ($createdResponse->code !== \Response::CREATED) {
            return $createdResponse;
        }

        $requestID = $this->cfbsRequestsModel->create(cfbsRequestsModel::DELETE_PROJECT_REQUEST, ['project_id' => $updatingProjectId]);
        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
        } catch (\Exception $exception) {
            syslog(
                LOG_ERR,
                "An error occurred while deleting updated project files with ID=$updatingProjectId. Err.:" . $exception->getTraceAsString()
            );
        }
        $this->projectModel->delete((int) $updatingProjectId);
        $auditLog();
        return $response;
    }

    /**
     * @rbacName Delete project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.delete
     */
    function delete($request, $id): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;

        if ($project = $this->projectModel->get((int)$id)) {
            $projectEntity = ProjectEntity::fromArray($project);
            $this->projectModel->delete((int)$id);
            $this->cfbsRequestsModel->create(cfbsRequestsModel::DELETE_PROJECT_REQUEST, ['project_id' => $id]);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $id,
                AuditLogFields::OBJECT_NAME => $projectEntity->getName(),
                AuditLogFields::ACTION => AuditLogActions::DELETE,
                AuditLogFields::DETAILS => ["Deleted Build project #$id.", [
                    'name' => $projectEntity->getName(),
                    'url' => valueOrNA($projectEntity->getRepositoryUrl()),
                    'branch' => valueOrNA($projectEntity->getBranch()),
                    'isLocal' => boolToYesNo($projectEntity->getIsLocal()),
                ]]
            ]);
        } else {
            $response->code = \Response::NOTFOUND;
        }

        return $response;
    }
}

/**
 * @uri /build/projects/empty
 * @uri /build/projects/local
 */
class EmptyProject extends BaseProjects
{

    /**
     * @rbacName Create project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.create
     */
    function post($request): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::CREATED;
        $requestData = !empty($request->data) ? \Utils::getValidJsonData($request->data) : [];
        $masterfiles = $requestData->masterfiles ?? null;

        $projectModel = new ProjectModel($this->username);

        $projectEntity = new ProjectEntity();
        $projectEntity->setIsLocal(true);

        if (!$projectEntity->getName()) {
            $projectEntity->setName($projectModel->getNextAvailableLocalProjectName());
        }

        return $this->createProject(
            $projectEntity, 
            $response, 
            ['git' => getGitDataFromRequest($requestData), ...($masterfiles ? ['masterfiles' => $masterfiles] : [])]
        );
    }
}

/**
 * @uri /build/projects/:id/sync
 */
class ProjectSync extends \CfProtectedResource
{
    /**
     * @rbacName Sync project changes
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.sync
     */
    function post($request, $id): \Response
    {
        $cfbsRequestsModel = new CfbsRequestsModel();
        $projectModel = new ProjectModel($this->username);

        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;

        $requestData = \Utils::getValidJsonData($request->data);

        if (
            !isset($requestData->action) ||
            empty($requestData->action) ||
            !in_array($requestData->action, cfbsRequestsModel::SYNC_ACTIONS)
        ) {
            $response->code = \Response::BADREQUEST;
            $response->body = 'Action is empty or invalid. Allowed actions: ' . implode(', ', cfbsRequestsModel::SYNC_ACTIONS);
            return $response;
        }

        if (!($project = $projectModel->get((int)$id))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        $requestID = $cfbsRequestsModel->create(
            cfbsRequestsModel::SYNC_PROJECT_REQUEST,
            [
                'project_id' => $id,
                'action' => $requestData->action,
                'git' => getGitDataFromRequest($requestData)
            ]
        );
        try {
            $reply = $cfbsRequestsModel->processRequestResponse($requestID);
            $projectModel->updatePushedAt((int) $id);
            $response->body = json_encode($reply);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $id,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::PUSHED,
                AuditLogFields::DETAILS => ["Pushed Build project #$id to upstream using git."]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while syncing build project.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }
}

/**
 * @uri /build/projects/:id/localDeploy
 */
class ProjectLocalDeploy extends \CfProtectedResource
{
    /**
     * @rbacName Deploy project locally
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.local_deploy
     */
    function post($request, $id): \Response
    {
        $cfbsRequestsModel = new CfbsRequestsModel();
        $projectModel = new ProjectModel($this->username);

        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!($project = $projectModel->get((int)$id))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        $requestID = $cfbsRequestsModel->create(cfbsRequestsModel::LOCAL_DEPLOY_REQUEST, ['project_id' => $id]);
        try {
            $reply = $cfbsRequestsModel->processRequestResponse($requestID);
            $projectModel->updateIsDeployedLocally((int) $id, true);
            $response->body = json_encode($reply);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $id,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::DEPLOY,
                AuditLogFields::DETAILS => ["Deployed Build project #$id"],
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while deploying build project locally.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }
}


/**
 * @uri /build/projects/:id/configureVCS
 */
class ProjectConfigureVCS extends \CfProtectedResource
{
    const DEFAULT_LOCAL_DEPLOY_PATH = '/opt/cfengine/build/local_deploy';
    const DEFAULT_LOCAL_DEPLOY_BRANCH = 'main';

    private $projectModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);

        $this->projectModel = new ProjectModel($this->username);
    }

    /**
     * @rbacAlias build.projects.get
     */
    function get($request, $id): \Response
    {
        $vcsSettings = new \VcsSettingsLib();
        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!($project = $this->projectModel->getWithSensitiveData((int)$id))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        $paramFile = $vcsSettings->get_dc_params_file_path();

        if (!file_exists($paramFile)) {
            $response->body = json_encode(['isSynchronised' => false]);
            return $response;
        }

        $vcsSettings = $vcsSettings->parseSettings($paramFile, $hideSensitiveData = false);

        $isLocalDeployment = (
            $vcsSettings['GIT_URL'] == self::DEFAULT_LOCAL_DEPLOY_PATH &&
            $vcsSettings['GIT_REFSPEC'] == self::DEFAULT_LOCAL_DEPLOY_BRANCH
        );
        $isSynchronised = false;
        $deployedLocally = false;

        if ($isLocalDeployment) {
            $isSynchronised = $vcsSettings['BUILD_PROJECT'] == $id;
            $deployedLocally = true;
        } else {
            $repositoryUrl = $project['repository_url'];
            $branch = $project['branch'];
            $isBranchAndRepoSynchronised = (
                $vcsSettings['GIT_URL'] == $repositoryUrl &&
                $vcsSettings['GIT_REFSPEC'] == $branch
            );

            if ($project['authentication_type'] == ProjectEntityValidator::KEY_AUTHENTICATION) {
                $isSynchronised = (
                    $isBranchAndRepoSynchronised &&
                    $project['ssh_private_key'] == file_get_contents($vcsSettings['PKEY'])
                );
            } else if ($project['authentication_type'] == ProjectEntityValidator::PASS_AUTHENTICATION) {
                $isSynchronised = (
                    $isBranchAndRepoSynchronised &&
                    $project['password'] ==  $vcsSettings['GIT_PASSWORD'] &&
                    $project['username'] ==  $vcsSettings['GIT_USERNAME']
                );
            }
        }

        $response->body = json_encode(['isSynchronised' => $isSynchronised, 'deployedLocally' => $deployedLocally]);
        return $response;
    }

    /**
     * @rbacAlias build.projects.get
     *
     * User should have permission to get project
     */
    function post($request, $id): \Response
    {
        $requestHeaders = getallheaders();
        $httpClient = new \PestHttpClient(API_URL);

        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;

        $localDeploy = isset($_GET['localDeploy']) && $_GET['localDeploy'] === 'true';

        if (!($project = $this->projectModel->getWithSensitiveData((int)$id))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        try {
            $data = $this->projectToVcsRequestData($project, $localDeploy);
            $httpClient->post('/vcs/settings', json_encode($data), ['Authorization: ' . $requestHeaders['Authorization']]);
        } catch (\Pest_BadRequest $e) {
            throw new \InvalidArgumentException($e->getMessage());
        }

        return $response;
    }

    private function projectToVcsRequestData(array $project, bool $localDeploy = false): array
    {
        return [
            'vcsType' => $project['classic_policy_set'] ? \VcsApi::GIT_TYPE : \VcsApi::CFBS_TYPE,
            'gitServer' => ($project['is_local'] || $localDeploy) ? self::DEFAULT_LOCAL_DEPLOY_PATH : $project['repository_url'],
            'gitRefspec' => ($project['is_local'] || $localDeploy) ?  self::DEFAULT_LOCAL_DEPLOY_BRANCH : $project['branch'],
            'gitUsername' => $project['username'],
            'gitPassword' => $project['password'],
            'gitPrivateKey' => $project['ssh_private_key'],
            'buildProject' => $project['id']
        ];
    }
}

/**
 * @uri /build/projects/:id/refresh
 */
class ProjectRefresh extends \CfProtectedResource
{
    /**
     * @rbacName Refresh project
     * @rbacGroup Build Projects API
     * @rbacAlias build.projects.refresh
     */
    function post($request, $id): \Response
    {
        $cfbsRequestsModel = new CfbsRequestsModel();
        $projectModel = new ProjectModel($this->username);

        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!$projectModel->get((int)$id)) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        $requestID = $cfbsRequestsModel->create(cfbsRequestsModel::REFRESH_PROJECT_REQUEST, ['project_id' => $id]);
        try {
            $reply = $cfbsRequestsModel->processRequestResponse($requestID);
            $response->body = json_encode($reply);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while refreshing build project.';
            syslog(LOG_ERR,  $response->body .'. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }
}

/**
 * @uri /build/projects/:id/default-action
 */
class ProjectDefaultAction extends \CfProtectedResource
{
    const ALLOWED_ACTIONS = ['push', 'pushAndDeploy', 'localDeploy'];
    /**
     * @rbacAlias build.projects.update
     *
     * Allow to set default action to user who has update role enabled
     */
    function post($request, $id): \Response
    {
        $projectModel = new ProjectModel($this->username);
        $requestData = \Utils::getValidJsonData($request->data);

        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!$projectModel->get((int)$id)) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found';
            return $response;
        }

        if (!isset($requestData->action) || !in_array($requestData->action, self::ALLOWED_ACTIONS)) {
            $response->code = \Response::BADREQUEST;
            $response->body = 'Action is missing or not allowed. Allowed actions: ' . implode(', ', self::ALLOWED_ACTIONS);
            return $response;
        }

        $projectModel->setDefaultAction($id, $requestData->action);

        return $response;
    }
}
