<?php

class Utils
{
    public const ERRID_UNKNOWN = 0;
    public const ERRID_SUCCESS = 1;
    public const ERRID_DBCONNECT = 2;
    public const ERRID_DBCLOSE = 3;
    public const ERRID_DB_OPERATION = 4;
    public const ERRID_ARGUMENT_MISSING = 5;
    public const ERRID_ARGUMENT_WRONG = 6;
    public const ERRID_BUFFER_FULL = 7;
    public const ERRID_ITEM_NONEXISTING = 8;
    public const ERRID_ITEM_EXISTS = 9;
    public const ERRID_ITEM_MULTIPLE = 10;
    public const ERRID_RBAC_DISABLED = 11;
    public const ERRID_RBAC_ACCESS_DENIED = 12;
    public const ERRID_DATA_UNAVAILABLE = 13;
    public const ERRID_HOST_NOT_FOUND = 14;
    public const ERRID_ACCESS_DENIED = 15;
    public const ERRID_ACCESS_DENIED_EXTERNAL = 16;

    public static function baseUri()
    {
        if (isset($_SERVER['HTTPS']) && (strtolower($_SERVER["HTTPS"]) == "on")) {
            $prot = 'https://';
        } else {
            $prot = 'http://';
        }

        return $prot . $_SERVER['SERVER_NAME'] ;
    }

    public static function getOauthServer($tokenLifeTimeInSeconds = OAUTH_ACCESSTOKEN_TIME)
    {
        $dbPort = getenv('CFENGINE_API_DATABASE_PORT') ? getenv('CFENGINE_API_DATABASE_PORT') : OAUTH_DATABASE_PORT;
        $dbHost = getenv('CFENGINE_API_DATABASE_ADDRESS') ? getenv('CFENGINE_API_DATABASE_ADDRESS') : OAUTH_DATABASE_HOST;

        $dsn = sprintf('pgsql:host=%s;port=%s;dbname=%s', $dbHost, $dbPort, OAUTH_DATABASE_NAME);
        $username = OAUTH_DATABASE_USER;
        $storage = new OauthCfPdo(array(
            'dsn' => $dsn
                    ));
        $options = array(
            'always_issue_new_refresh_token' => true,
            'refresh_token_lifetime' => OAUTH_REFRESHTOKEN_TIME,
            'access_lifetime' => $tokenLifeTimeInSeconds
        );
        $server = new OAuth2\Server($storage, $options);
        // create the grant type
        $server->addGrantType(new OAuth2\GrantType\UserCredentials($storage));
        $server->addGrantType(new OAuth2\GrantType\RefreshToken($storage, $options));

        return $server;
    }

    public static function getVcsConfigFilePath()
    {

        $configFileName = getenv('CFENGINE_TEST_OVERRIDE_DC_CONFIG_FILE_NAME') ? getenv('CFENGINE_TEST_OVERRIDE_DC_CONFIG_FILE_NAME') : 'vcsconfig.json';
        $filePath = FCPATH . 'config' . DIRECTORY_SEPARATOR . $configFileName;
        return $filePath;
    }

    public static function queryParam($name, $defaultValue = null)
    {
        if (!array_key_exists($name, $_GET)) {
            return $defaultValue;
        } else {
            return $_GET[$name];
        }
    }

    public static function bodyParam($name, $defaultValue = null)
    {
        if (!array_key_exists($name, $_POST)) {
            return $defaultValue;
        } else {
            return $_POST[$name];
        }
    }

    /**
     * Get valid json data. Throw ResponseException in error case.
     *
     * @param $string
     * @param bool $returnArray
     * @return mixed
     * @throws ResponseException
     */
    public static function getValidJsonData($string, $returnArray = false)
    {

        $data = json_decode($string, $returnArray);
        switch (json_last_error()) {
            case JSON_ERROR_NONE:
                $error = '';
                break;
            case JSON_ERROR_DEPTH:
                $error = 'The maximum stack depth has been exceeded.';
                break;
            case JSON_ERROR_STATE_MISMATCH:
                $error = 'Invalid or malformed JSON.';
                break;
            case JSON_ERROR_CTRL_CHAR:
                $error = 'Control character error, possibly incorrectly encoded.';
                break;
            case JSON_ERROR_SYNTAX:
                $error = 'Syntax error, malformed JSON.';
                break;
            case JSON_ERROR_UTF8:
                $error = 'Malformed UTF-8 characters, possibly incorrectly encoded.';
                break;
            case JSON_ERROR_RECURSION:
                $error = 'One or more recursive references in the value to be encoded.';
                break;
            case JSON_ERROR_INF_OR_NAN:
                $error = 'One or more NAN or INF values in the value to be encoded.';
                break;
            case JSON_ERROR_UNSUPPORTED_TYPE:
                $error = 'A value of a type that cannot be encoded was given.';
                break;
            default:
                $error = 'Unknown JSON error.';
                break;
        }

        if ($error != "") {
            $error = "An error has been occurred during JSON parsing: " .  $error;
            syslog(LOG_DEBUG, 'Requesting - ' . $_SERVER['REQUEST_URI'] .' '. $error . ': ' . $string);
            throw new ResponseException($error, Response::BADREQUEST);
        }

        return $data;
    }

    public static function checkRequiredParameters($data = array(), $requiredParams = array())
    {


        foreach ($requiredParams as $key) {
            if (array_key_exists($key, $data) === false) {
                $message = $key . ' is a required parameter';
                throw new ResponseException($message, Response::BADREQUEST);
            }
        }
    }

    public static function checkValidQueryParams($queryParams)
    {
        $defaultParams = array(
            'from',
            'to',
            'count',
            'page'
        );


        foreach (array_keys($_GET) as $value) {
            if (!in_array($value, $queryParams, true) && !in_array($value, $defaultParams, true)) {
                throw new ResponseException($value . ' is not a valid query parameter', Response::BADREQUEST);
            }
        }
    }

    public static function checkInteger($var, $name)
    {
        if (!is_null($var)) {
            if (ctype_digit($var)) {
                return (int)$var;
            } else {
                throw new ResponseException("query parameter '" . $name . "' must be an integer", Response::BADREQUEST);
            }
        } else {
            return null;
        }
    }

    public static function checkBoolean($var, $name)
    {
        if (!is_null($var)) {
            if (strtolower($var) === "true") {
                return true;
            } elseif (strtolower($var) === "false") {
                return false;
            } else {
                throw new ResponseException("query parameter '" . $name . "' must be an true or false", Response::BADREQUEST);
            }
        } else {
            return null;
        }
    }

    public static function checkPromiseState($state)
    {
        if (!is_null($state) && !in_array($state, array(
            'kept',
            'notkept',
            'repaired'
        ))) {
            throw new ResponseException('Valid promise states are kept, notkept and repaired', Response::BADREQUEST);
        }
        return $state;
    }

    public static function prefixedHostKey($hostkey)
    {
        if ($hostkey && substr($hostkey, 0, 4) !== 'SHA=') {
            return 'SHA=' . $hostkey;
        }
        return $hostkey;
    }

    public static function packageResult($data, $pageNum, $total)
    {
        $packData = $data;
        $metadata = array(
            "page" => $pageNum,
            "count" => count($packData) ,
            "total" => $total,
            "timestamp" => time()
        );
        $output = array(
            'meta' => $metadata,
            'data' => $packData
        );
        return json_encode($output);
    }

    public static function CFAPIErrorCodeToHttpCode($cfapi_code)
    {


        switch ($cfapi_code) {
            case Utils::ERRID_SUCCESS:
                return 200;
            case Utils::ERRID_ITEM_NONEXISTING:
                return 404;
            case Utils::ERRID_ARGUMENT_MISSING:
            case Utils::ERRID_ARGUMENT_WRONG:
                return 400;
            case Utils::ERRID_ITEM_EXISTS:
                return 409;
            case Utils::ERRID_RBAC_ACCESS_DENIED:
            case Utils::ERRID_ACCESS_DENIED:
            case Utils::ERRID_ACCESS_DENIED_EXTERNAL:
                return 403;
            case Utils::ERRID_DATA_UNAVAILABLE:
            case Utils::ERRID_HOST_NOT_FOUND:
                return 404;
            default:
                return 500;
        }
    }

    public static function CFAPIErrorMessageByCode(int $cfapi_code, string $message): string
    {
        switch ($cfapi_code) {
            // do not display a raw error message that comes from database
            case Utils::ERRID_DB_OPERATION:
                syslog(LOG_ERR, 'Error while database querying: ' . $message);
                return 'An error occurred while database querying.';
            default:
                return $message;
        }
    }

    public static function deleteAll($directory, $empty = false)
    {
        if (substr($directory, -1) == "/") {
            $directory = substr($directory, 0, -1);
        }
        if (!file_exists($directory) || !is_dir($directory)) {

            return false;
        } elseif (!is_readable($directory)) {

            return false;
        } else {
            $directoryHandle = opendir($directory);


            while ($contents = readdir($directoryHandle)) {
                if ($contents != '.' && $contents != '..') {
                    $path = $directory . "/" . $contents;
                    if (is_dir($path)) {
                        self::deleteAll($path);
                    } else {
                        unlink($path);
                    }
                }
            }
            closedir($directoryHandle);
            if ($empty == false) {
                if (!rmdir($directory)) {

                    return false;
                }
            }

            return true;
        }
    }

    public static function delete_git_update_scripts($scriptsPath)
    {
        $deleteScripts = array(
            'params.sh'
        );


        foreach ($deleteScripts as $s) {
            @unlink($scriptsPath . DIRECTORY_SEPARATOR . $s);
        }
    }

    public static function get_rbac_settings(string $username, bool $getUserPermission)
    {
        $userJson = cfapi_user_get('admin', $username, $getUserPermission);
        $settingsJson = cfapi_settings_get('admin');
        $user = json_decode($userJson, true);
        $settings = json_decode($settingsJson, true);
        $rbacMode = (bool)$settings['data'][0]['rbacEnabled'];
        $roles = (array)$user['data'][0]['roles'];

        $includes = array();
        $excludes = array();


        foreach ($roles as $rolename) {
            $roleJson = cfapi_role_get('admin', $rolename);
            $roleDetails = json_decode($roleJson, true);
            if (isset($roleDetails['data'][0]['includeContext'])) {
                $includes[] = $roleDetails['data'][0]['includeContext'];
            }
            if (isset($roleDetails['data'][0]['excludeContext'])) {
                $excludes[] = $roleDetails['data'][0]['excludeContext'];
            }

        }

        $includes = array_filter($includes);
        $excludes = array_filter($excludes);

        $result = array(
            'rbac' => $rbacMode,
            'roles' => $roles,
            'includes' => $includes,
            'excludes' => $excludes
        );

        return $result;
    }

    public static function getQueryApiResultRows(array $data)
    {
        if (!isset($data['data']['0']['rows'])) {
            throw new Exception('Utils::getQueryApiResultRows: cfapi_query_post function has returned empty object');
        }
        return $data['data']['0']['rows'];
    }

    public static function isTimeZone(PDO $pdo, $timeZone)
    {
        $statement = $pdo->prepare(
            'SELECT 1 FROM pg_timezone_names 
                       WHERE LOWER(name) = LOWER(:tz) 
                       OR LOWER(abbrev) = LOWER(:tz)'
        );
        $statement->execute(['tz' => $timeZone]);
        $count = $statement->rowCount();
        return $count == 1;
    }

    public static function processQueryApiParams(array $params)
    {
        $params['sortDescending'] = $params['sortDescending'] ?? false;
        $params['sortColumn'] = $params['sortColumn'] ?? false;
        $params['skip'] = isset($params['skip']) ? intval($params['skip']) : -1;
        $params['limit'] = isset($params['limit']) ? intval($params['limit']) : -1;
        $params['hostContextInclude'] = $params['hostContextInclude'] ?? [];
        $params['hostContextExclude'] = $params['hostContextExclude'] ?? [];

        if (!is_array($params['hostContextInclude'])) {
            throw new \InvalidArgumentException("'hostContextInclude' field must be an array");
        }

        if (!is_array($params['hostContextExclude'])) {
            throw new \InvalidArgumentException("'hostContextExclude' field must be an array");
        }

        return $params;
    }
}
