<?php

/**
 * Class InventoryLib
 */
class InventoryLib
{
    /**
     * @var VariablesDictionaryModel
     */
    private $variableDictionaryModel;
    private $pdo;


    public function __construct(VariablesDictionaryModel $variablesDictionary)
    {
        $this->variableDictionaryModel = $variablesDictionary;
        $this->pdo = CfPdo::getInstance()->getConnection();
    }

    /**
     * @var int
     */
    private $limit = 1000;
    /**
     * @var int
     */
    private $maxLimit = 1000000000;

    const LESS_OPERATOR = '<';
    const GREATER_OPERATOR = '>';
    const EQUAL_OPERATOR = '=';
    const NOT_EQUAL_OPERATOR = '!=';
    const LESS_OR_EQUAL_OPERATOR = '<=';
    const GREATER_OR_EQUAL_OPERATOR = '>=';
    const MATCHES_OPERATOR = 'matches';
    const NOT_MATCH_OPERATOR = 'not_match';
    const CONTAINS_OPERATOR = 'contains';
    const NOT_CONTAIN_OPERATOR = 'not_contain';
    const REGEX_MATCHES_OPERATOR = 'regex_matches';
    const REGEX_NOT_MATCH_OPERATOR = 'regex_not_match';
    const IS_REPORTED_OPERATOR = 'is_reported';
    const IS_NOT_REPORTED_OPERATOR = 'is_not_reported';

    /**
     * @var array
     */
    private $operatorsMap = [
        //Numbers
        self::LESS_OPERATOR => ['operator' => '<', 'function' => 'cf_convertToNum'],
        self::GREATER_OPERATOR => ['operator' => '>', 'function' => 'cf_convertToNum'],
        self::EQUAL_OPERATOR => ['operator' => '=', 'function' => 'cf_convertToNum'],
        self::NOT_EQUAL_OPERATOR => ['operator' => '!=', 'function' => 'cf_convertToNum'],
        self::LESS_OR_EQUAL_OPERATOR => ['operator' => '<=', 'function' => 'cf_convertToNum'],
        self::GREATER_OR_EQUAL_OPERATOR => ['operator' => '>=', 'function' => 'cf_convertToNum'],
        //Strings
        self::MATCHES_OPERATOR => ['operator' => 'ILIKE'],
        self::NOT_MATCH_OPERATOR => ['operator' => 'NOT ILIKE'],
        self::CONTAINS_OPERATOR => ['operator' => 'LIKE'],
        self::NOT_CONTAIN_OPERATOR => ['operator' => 'NOT LIKE'],
        'slist' => [
            self::MATCHES_OPERATOR => ['operator' => '~'],
            self::NOT_MATCH_OPERATOR => ['operator' => '!~'],
        ],
        self::REGEX_MATCHES_OPERATOR => ['operator' => '~*'],
        self::REGEX_NOT_MATCH_OPERATOR => ['operator' => '!~*'],
        self::IS_REPORTED_OPERATOR => ['operator' => 'IS NOT NULL'],
        self::IS_NOT_REPORTED_OPERATOR => ['operator' => 'IS NULL']
    ];


    /**
     * @param $data
     * @return string
     * @throws Exception
     */
    public function applyFilter($data): string
    {
        $pdo = CfPdo::getInstance()->getConnection();

        $where = ['WHERE'];
        if (isset($data->filter) && (is_object($data->filter) || is_array($data->filter))) {
            $dataFilter = $data->filter;

            foreach ($dataFilter as $variable => $opVal) {
                foreach ($opVal as $operator => $value) {
                    $andOrOperator = 'AND';
                    // if operator starts with `or|` like `or|match` then we apply OR condition and use match operator
                    if (str_starts_with($operator, 'or|')) {
                        $andOrOperator = 'OR';
                        $operator = substr($operator, 3);
                    }
                    if (is_array($value)) {
                        foreach ($value as $v) {
                            $where[] = sizeof($where) > 1 ? $andOrOperator : '';
                            $where[] = $this->makeWhereCondition($operator, $variable, $v);
                        }
                    } else {
                        $where[] = sizeof($where) > 1 ? $andOrOperator : '';
                        $where[] = $this->makeWhereCondition($operator, $variable, $value);
                    }
                }
            }
        }

        if (
            (property_exists($data, 'hostContextInclude') && sizeof($data->hostContextInclude) > 0) ||
            (property_exists($data, 'hostContextExclude') && sizeof($data->hostContextExclude) > 0)
        ) {
            /**
            * contexts allow usage of "_" and ":" signs
            * "_" is treated by postgres text search parser as whitespace
            * ":" sign is not allowed in tsquery and postgreql throws an error
            * replace "_" and ":" signs with "." instead, as we do in the cache generation
            */
            $data->hostContextInclude = property_exists($data, 'hostContextInclude') ?
                str_replace(['_', ':'], '.', $data->hostContextInclude) :
                '';
            $data->hostContextExclude = property_exists($data, 'hostContextExclude') ?
                str_replace(['_', ':'], '.', $data->hostContextExclude) :
                '';

            $where[] = sizeof($where) === 1 ? '' : ' AND ';
            // here is safe to use user data in the query because the data passes to the Query API and not executing directly
            // and data is additionally quoted to prevent SQL injections
            $where[] = " inventory_new.hostkey IN  ( 
                        SELECT v_contextcache.hostkey FROM v_contextcache 
                        WHERE ContextVector @@ to_tsquery(
                        'simple', 
                        {$pdo->quote($this->makeContextsFilter($data->hostContextInclude,  $data->hostContextExclude))} 
                        ))";

        }

        if (isset($data->hostsFilter)) {
            $this->applyHostsFilter($where, $data->hostsFilter, $pdo);
        }

        //if $where contains only 1 element it's WHERE, skip it
        return sizeof($where) > 1 ? implode(' ', $where) : '';
    }

    const HOSTS_FILTER_HOSTKEY_TYPE = 'hostkey';

    private function applyHostsFilter(&$where, $hostsFilter, \PDO $pdo): void
    {
        $excludeSQL = '';

        // we support only hostkey type now but will be extended with inventory attribute or class later
        if (isset($hostsFilter->excludes->type) && $hostsFilter->excludes->type === self::HOSTS_FILTER_HOSTKEY_TYPE) {
            $where[] = sizeof($where) === 1 ? '' : ' AND ';
            $where[] = $excludeSQL = sprintf("inventory_new.hostkey NOT IN (%s)",
                implode(',', array_map(function ($item) use ($pdo) {
                    return $pdo->quote($item);
                }, $hostsFilter->excludes->data))
            );
        }

        if (
            isset($hostsFilter->includes->type) &&
            $hostsFilter->includes->type === self::HOSTS_FILTER_HOSTKEY_TYPE &&
            sizeof($where) > 1 // do not apply includes hosts if no filters were added because makes no difference
        ) {
            if (strlen($excludeSQL) > 0) {
                $excludeSQL = "AND $excludeSQL";
            }
            $where[] = sprintf("OR (inventory_new.hostkey IN (%s) %s)",
                implode(',', array_map(function ($item) use ($pdo) {
                    return $pdo->quote($item);
                }, $hostsFilter->includes->data)),
                $excludeSQL
            );
        }
    }

    private function makeContextsFilter($hostContextInclude, $hostContextExclude): string
    {
        $where = [];
        if ($hostContextInclude != null) {
            if (is_array($hostContextInclude)) {
                $where[] = "(" . implode(' & ', $hostContextInclude) . ")";
            } else {
                $where[] = "($hostContextInclude)";
            }
        }

        if ($hostContextExclude != null) {
            if (is_array($hostContextExclude)) {
                $where[] = "(!" . implode(' & !', $hostContextExclude) . ")";
            } else {
                $where[] = "(!$hostContextExclude)";
            }
        }

        return implode(' & ', $where);
    }


    /**
     * @param $data
     * @return string
     */
    public function applySorting($data)
    {
        if (empty($data->sort)) {
            return '';
        }
        $sort = $data->sort;
        if ($sort[0] == '-')  //check first symbol
        {
            $variable = substr($sort, 1); //if first symbol is `-`, get variable without `-`
            $direction = 'DESC';
        } else {
            $variable = $sort;
            $direction = 'ASC';
        }

        $variableEntity = $this->variableDictionaryModel->getByName($variable);

        if ($variableEntity && in_array($variableEntity['type'], ['int', 'real'])) {
            return " ORDER BY cf_convertToNum(values->>{$this->pdo->quote($variable)}) $direction ";
        }

        return " ORDER BY values->>{$this->pdo->quote($variable)} $direction ";

    }


    /**
     * @param $data
     * @return string
     * @throws Exception
     */
    public function applySelect($data)
    {
        $variableModel = new VariablesDictionaryModel();
        if (!isset($data->select)) {
            throw new Exception('Select field is required');
        }

        if (!is_array($data->select)) {
            throw new Exception('Wrong Select data');
        }

        $selectArray = $data->select;

        $selectData = [];
        $functionsMap = $variableModel->getFunctionsMap();

        /**
         * var $aliases contains associative array where key is an alias and value is inventory attribute
         * if alias will be passed as select argument then API will return value from related inventory attribute
         */
        $aliases = (new CfInventoryAliases())->getAliases();

        foreach ($selectArray as $key => $select) {

            $selectAs = '"' . trim(
                    $this->pdo->quote(
                        str_replace('"', '""', $select) // escape double quotes
                    ), "'" // trims ' as we don't need them in AS statement
                ) . '"'; // surrounds AS value by "
            if (isset($aliases[strtolower($select)])) {
                $select = $aliases[strtolower($select)];
            }

            if($select == 'hostkey'){
                $selectData[] = " $select AS $selectAs";
            }
            elseif ($select == 'resultCount'){
                $selectData[] = " count(*) as count ";
            }
            elseif (isset($functionsMap[$select])) {
                $selectData[] = " {$functionsMap[$select]}(values->>{$this->pdo->quote($select)}) AS $selectAs";
            } else {
                $selectData[] = " values->>{$this->pdo->quote($select)} AS $selectAs";
            }


        }
        return implode(', ', $selectData);
    }


    /**
     * @param $literalOperator
     * @param $variable
     * @param $value
     * @return string
     * @throws Exception
     */
    private function makeWhereCondition($literalOperator, $variable, $value)
    {
        $variableEntity = $this->variableDictionaryModel->getByName($variable);
        $variableType = null;

        if (is_array($variableEntity) && isset($variableEntity['type'])) {
            $variableType = $variableEntity['type'];
        }

        if (!isset($this->operatorsMap[$literalOperator])) {
            throw new Exception('Wrong abbreviation of operator');
        }

        $conditionData = $this->operatorsMap[$variableType][$literalOperator] ?? $this->operatorsMap[$literalOperator];
        $value = $this->prepareValueToWhere($value, $literalOperator, $variableType);

        if ($variable == 'hostkey') {
            $whereCond = vsprintf("%s %s %s", [$variable, 'LIKE', $value]);
        } elseif (isset($conditionData['function']) && in_array($variableType, ['int', 'real'])) {
            $whereCond = vsprintf("%s(values->>%s) %s %s(%s)",
                [
                    $conditionData['function'],
                    $this->pdo->quote($variable),
                    $conditionData['operator'],
                    $conditionData['function'],
                    $value
                ]);
        } else {
            /**
             * If a variable is not reported, then a negative comparison will not return it.
             * For example, CFEngine roles != 'policy_server' in the case of regular hosts will not work as they're not reporting it.
             * To solve this case, we set not reported values to the ideographic space if operators are not is_reported or is_not_reported
             */
            $variablePattern =
                (in_array($literalOperator, [self::IS_REPORTED_OPERATOR, self::IS_NOT_REPORTED_OPERATOR]))
                    ? "values->>%s"
                    : "COALESCE (values->>%s, '　')";
            $whereCond = vsprintf("$variablePattern %s %s", [$this->pdo->quote($variable), $conditionData['operator'], $value]);
        }

        return $whereCond;
    }

    private function prepareValueToWhere($value, $literalOperator, $type)
    {
        $filterValueFormatter = FormatInventoryFilterFactory::getImplementation($type);
        return $filterValueFormatter->getValueForWhere($value, $literalOperator, $this->pdo);
    }


    /**
     * @param $data
     * @param $showSQL
     * @param HttpClientBase $httpClient
     * @return Response
     */
    public function sendRequestToQueryApi($data, HttpClientBase $httpClient, $showSQL = false)
    {
        $requestHeaders = getallheaders();
        $headers = ['Authorization: ' . $requestHeaders['Authorization']];
        return $this->changeResponse($httpClient->post('/query', $data, $headers), $showSQL);

    }

    /**
     * @param $data
     * @return string
     */
    private function changeResponse($data, $showSQL = false)
    {
        $data = json_decode($data);
        if (isset(reset($data->data)->query) AND !$showSQL) {
            unset(reset($data->data)->query);
        }
        return json_encode($data, JSON_PRETTY_PRINT);
    }

    /**
     * @param $limit
     * @return int
     */
    public function getLimit($limit = 1000)
    {
        //if the limit is null then set -1 as non-limit value
        $limit = $limit === null ? -1 : $limit;
        $limit = (isset($limit) && is_numeric($limit)) ? $limit : $this->limit;
        return $limit > $this->maxLimit ? $this->maxLimit : $limit;
    }
    
    public function extendSQLWithInventoryCTE(&$data): void
    {
        $inventoryLib = new InventoryLib(new VariablesDictionaryModel());
        $where = $inventoryLib->applyFilter($data);
        $cte = "WITH inventory_cte as (SELECT hostkey as filtered_hostkey FROM inventory_new $where) ";

        $re = '/^(\s+)*WITH/';
        preg_match($re, $data->query, $matches);
        // if user query already contain CTE then add inventory together with existing one
        if (sizeof($matches) > 0) {
            $data->query = preg_replace($re, $cte . ', ', $data->query);
        } else {
            $data->query = $cte . $data->query;
        }
    }
}