<?php
class Authentication_model extends Cf_Model
{

    protected $userResource = 'user';
    protected $rolesResource = 'role';
    protected $defaultRolesResource = 'default_role';
    const CF_PASSWORD_RESET_HEADER = 'Cf-Password-Reset';

    public function __construct()
    {
        parent::__construct();
        $this->load->config('ion_auth', true);
        $this->load->helper('cookie');
        $this->load->helper('date');
    }
    /**
     * For login operation
     * @param type $username
     * @param type $password
     * @param type $remmember
     * @return type
     */
    function login($identity, $password, $mode = 'oauth2')
    {
        $mpClient = config_item('MP_CLIENT_ID');
        $mpSecret = config_item('MP_CLIENT_SECRET');
        $grantType = config_item('MP_GRANT_TYPE');

        $data = array(
            'username' => $identity,
            'password' => $password,
            'client_id' => $mpClient,
            'grant_type' => $grantType
        );

        if (!empty($mpSecret))
        {
            $data[ 'client_secret'] = $mpSecret;
        }

        try
        {
            $body = $this->getRestClient()->post('/oauth2/token', $data);
            $data = $this->checkData($body);
            if (is_array($data) && $this->hasErrors() == 0)
            {
                if (!empty($data['access_token']))
                {
                    log_message('debug', 'Login successful with access token');
                    $this->getRestClient()->setupOauthAuth($data['access_token']);
                    return $data;
                }
                else
                {
                    throw new Exception('Invalid server response cannot access server token.');
                }
                delete_cookie('dark_mode');
            }
            throw new Exception($this->getErrorsString());
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'login ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }

    function unsetToken($token)
    {
        try
        {
            $tokens = is_array($token) ? $token : array($token);
            $data = array(
                'tokens' => $tokens
            );
            $response = $this->getRestClient()->post('/oauth2/token/revoke', $data);
            return $response;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'Cannot unset access token. Error:: ' . $e->getMessage());
            throw $e;
        }
    }

    function getActiveSession()
    {
        try
        {
            $response = $this->getRestClient()->get('/oauth2/token/');
            return json_decode($response,true);
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'Cannot get active sessions from API. Error:: ' . $e->getMessage());
            throw $e;
        }
    }
    /**
     * Get Roles for a users
     */
    function getRolesForUser($username)
    {
        try
        {

            $user = $this->getUserDetails($username);
            if (key_exists('roles', $user))
            {
                return $user['roles'];
            }
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), "getRolesForUser , Error grabbing roles for user");
            return false;
        }
    }
    /**
     * Get all roles
     */
    function getAllRoles()
    {
        try
        {
            $body = $this->getRestClient()->get('/' . $this->rolesResource);
            $data = $this->checkData($body);
            if (is_array($data) && $this->hasErrors() == 0)
            {
                return $data['data'];
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'getAllRoles ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }

    /**
     * Get default role
     */
    function getDefaultRole()
    {
        try
        {
            $body = $this->getRestClient()->get('/' . $this->defaultRolesResource);
            $data = $this->checkData($body);
            if ($this->hasErrors() == 0)
            {
                return $data;
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'getDefaultRole ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }



    function setDefaultRole($name)
    {
        try
        {
            $body = $this->getRestClient()->post('/' . $this->defaultRolesResource, ['name' => $name]);
            $data = $this->checkData($body);
            if ( $this->hasErrors() == 0)
            {
                return $data;
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'setDefaultRole ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }

    /**
     * Get the details of a role
     * @param type $rolename
     */
    function getRoleDetails($rolename)
    {
        try
        {
            $body = $this->getRestClient()->get('/' . $this->rolesResource . '/' . $rolename);
            $data = $this->checkData($body);
            if (is_array($data) && $this->hasErrors() == 0)
            {
                return $data['data'][0];
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'getRoleDetails ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * For creating users
     * @param type $data
     */
    function createRole($data)
    {
        try
        {
            $this->getRestClient()->put('/' . $this->rolesResource . '/' . $data['name'], $data);
            if ($this->getRestClient()->lastStatus() == 201)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'createRole ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     *
     * @param type $username
     * @param type $data
     * @return type boolean
     */
    function updateRole($rolename, $data)
    {
        try
        {
            $this->getRestClient()->post('/' . $this->rolesResource . '/' . $rolename, $data);
            if ($this->getRestClient()->lastStatus() == 204)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'updateRole ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * Deletes a role
     * @param type $rolename
     */
    function deleteRole($rolename)
    {
        try
        {
            $this->getRestClient()->delete('/' . $this->rolesResource . '/' . $rolename);
            if ($this->getRestClient()->lastStatus() == 204)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'deleteRole ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * Create a user
     * @param type $data
     * @return type
     */
    function createUser($data)
    {
        try
        {
            $this->getRestClient()->put('/' . $this->userResource . '/' . $data['id'], $data);
            if ($this->getRestClient()->lastStatus() == 201)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'createUser ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     *
     * @return type array data
     */
    function getAllUsers($page = 1, $count = 50, $userType = null, $returnWithMeta = false)
    {
        try
        {

            if ($userType == 'external')
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?external=true&page=' . $page . '&count=' . $count);
            }
            elseif ($userType == 'internal')
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?external=false&page=' . $page . '&count=' . $count);
            }
            else
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?page=' . $page . '&count=' . $count);
            }

            $data = $this->checkData($body);

            if (is_array($data) && $this->hasErrors() == 0)
            {
                if ($returnWithMeta)
                {
                    return $data;
                }
                else
                {
                    return $data['data'];
                }
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'getAllUsers ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /*get a user list for searched users*/

    function searchUser($userType, $usernameExpression, $returnWithMeta = false)
    {
        try
        {

            if ($userType == 'external')
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?external=true&id=' . $usernameExpression);
            }
            elseif ($userType == 'internal')
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?external=false&id=' . $usernameExpression);
            }
            else
            {
                $body = $this->getRestClient()->get('/' . $this->userResource . '?id=' . $usernameExpression);
            }

            $data = $this->checkData($body);
            if (is_array($data) && $this->hasErrors() == 0)
            {
                if ($returnWithMeta)
                {
                    return $data;
                }
                else
                {
                    return $data['data'];
                }
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'getAllUsers ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * Generate the details for given object i.e user
     * @param type $username
     * @param type $password
     * @param type $object
     * @return type
     */
    function getUserDetails($username)
    {

        try
        {

            $body = $this->getRestClient()->get('/' . $this->userResource . '/' . $username);

            $data = $this->checkData($body);
            if (is_array($data) && $this->hasErrors() == 0)
            {
                return $data['data'][0];
            }
            else
            {
                throw new Exception($this->getErrorsString());
            }
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), "getUserDetails  " . get_class($e) . " " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     *
     * @param type $id
     * @param type $data
     */
    function update_user($id, $data, $headers = [])
    {

        try
        {
            $this->getRestClient()->post('/' . $this->userResource . '/' . $id, $data, $headers);
            if ($this->getRestClient()->lastStatus() == 202)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            $this->setError($e->getMessage());
            log_message(log_level_for_exception($e), 'update_user ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * Delete the supplied user
     * @param type $username
     */
    function deleteUser($id)
    {
        try
        {
            $this->getRestClient()->delete('/' . $this->userResource . '/' . $id);
            if ($this->getRestClient()->lastStatus() == 202)
            {
                return true;
            }
            return false;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'deleteUser ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
    /**
     * Change user passwdor
     *
     * @param type $id - user id
     * @param type $old - old passwd
     * @param type $new - new passwd
     * @return boolean
     * @throws Exception
     */

    function change_password($id, $old, $new)
    {
        if (!$this->isPasswordValid($id, $old)) {
            return  false;
        }

        try
        {
            return $this->update_user($id, ['password' => $new]);
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'change_password ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }

    /**
     * Validates the provided password by making a request to the API using basic authentication.
     *
     * @param string $username The username to authenticate.
     * @param string $password The password to validate.
     * @return bool Returns true if the password is valid, false if the authentication fails.
     */
    public function isPasswordValid(string $username, string $password) : bool
    {
        try {
            $restClient = new Http_client(['base_url' => $this->config->item('rest_server')]);
            $restClient->setupAuth($username, $password);
            $restClient->get('/');
            return true;
        } catch (HttpClient_Unauthorized $exception) {
            return false;
        }
    }

    function resetPassword($id)
    {
        $this->load->model('ProfileModel');
        $this->load->helper('string');
        $generatedPassword = random_string(len: 14);
        try
        {
            $this->update_user($id, ['password' => $generatedPassword], [self::CF_PASSWORD_RESET_HEADER => 'true']);
            if ($this->getRestClient()->lastStatus() == 202)
            {
                $this->ProfileModel->setAdditionalData($id, ProfileModel::FIRST_lOGIN_AFTER_RESET_KEY, data: true);
                return $generatedPassword;
            }
            return false;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), 'change_password ' . get_class($e) . "  " . $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }
}
