<?php

/**
 * @uri /user/([^\/]+)/subscription/query/([^\/]+) 3
 */
class UserSubscriptionQuery extends CfProtectedResource
{
    /**
     * @param $request
     * @param $username
     * @param $subscription_id
     *
     * @rbacName View report
     * @rbacGroup Scheduled reports
     * @rbacAlias userSubscriptionQuery.get
     * @rbacAllowedByDefault
     *
     * @return Response
     */
    function get($request, string $username, string $subscription_id)
    {
        $user = $this->username;

        $response = new Response($request);

        $payload = cfapi_user_subscription_query_get($user, $username, $subscription_id);
        if ($payload)
        {
            $response->code = Response::OK;
            $response->body = $payload;
        }
        else
        {
            $response->code = Response::NOTFOUND;
        }

        return $response;
    }

    /**
     * @param $request
     * @param $username
     * @param $subscription_id
     *
     * @rbacName Create report
     * @rbacGroup Scheduled reports
     * @rbacAlias userSubscriptionQuery.put
     * @rbacAllowedByDefault
     *
     * @return Response
     * @throws ResponseException
     */
    function put($request, $username, $subscription_id)
    {
        $user = $this->username;

        $data = Utils::getValidJsonData($request->data);
        $output_types = $data->outputTypes;
        if (!$output_types)
        {
            $output_types = array('csv');
        }

        if ($data->hostContextInclude)
        {
            if (!is_array($data->hostContextInclude))
            {
                $response = new Response($request);
                $response->code = Response::BADREQUEST;
                $response->body = "'hostContextInclude' field must be an array";
                return $response;
            }
        }
        else
        {
            $data->hostContextInclude = array();
        }

        if ($data->hostContextExclude)
        {
            if (!is_array($data->hostContextExclude))
            {
                $response = new Response($request);
                $response->code = Response::BADREQUEST;
                $response->body = "'hostContextExclude' field must be an array";
                return $response;
            }
        }
        else
        {
            $data->hostContextExclude = array();
        }

        $response = new Response($request);

        if (cfapi_user_subscription_query_put($user,
                                              $username,
                                              $subscription_id,
                                              $data->to ?? '',
                                              $data->enabled,
                                              $data->query ?? '',
                                              $data->schedule ?? '',
                                              $data->title ?? '',
                                              $data->description ?? '',
                                              $output_types,
                                              $data->hostContextInclude ?? [],
                                              $data->hostContextExclude ?? []))
        {
            $response->code = Response::CREATED;
        }
        else
        {
            $response->code = Response::INTERNALSERVERERROR;
        }

        return $response;
    }

    /**
     * @param $request
     * @param $username
     * @param $subscription_id
     *
     * @rbacName Delete report
     * @rbacGroup Scheduled reports
     * @rbacAlias userSubscriptionQuery.delete
     * @rbacAllowedByDefault
     *
     * @return Response
     */
    function delete($request, string $username, string $subscription_id)
    {
        $user = $this->username;

        $response = new Response($request);

        if (cfapi_user_subscription_query_delete($user, $username, $subscription_id))
        {
            $response->code = Response::ACCEPTED;
        }
        else
        {
            $response->code = Response::INTERNALSERVERERROR;
        }

        return $response;
    }
}

/**
 * @uri /user/(.+)/subscription/query 2
 */
class UserSubscriptionQueryList extends CfProtectedResource
{
    /**
     * @param $request
     * @param $username
     *
     * @rbacName View report list
     * @rbacGroup Scheduled reports
     * @rbacAlias userSubscriptionQueryList.get
     * @rbacAllowedByDefault
     *
     * @return Response
     */
    function get($request, string $username)
    {
        $user = $this->username;

        $response = new Response($request);
        $payload = cfapi_user_subscription_query_list($user, $username);
        if ($payload)
        {
            $response->code = Response::OK;
            $response->body = $payload;
        }
        else
        {
            $response->code = Response::NOTFOUND;
        }

        return $response;
    }
}


/**
 * @uri /user/:username 1
 */
class User extends CfProtectedResource
{
    const CF_PASSWORD_RESET_HEADER = 'Cf-Password-Reset';
    /**
     * @var CfSettings 
     */
    private $cfSettingsService;
    public function __construct($parameters)
    {
        $this->cfSettingsService = CfSettings::getInstance();
        parent::__construct($parameters);
    }

    /**
     * @param $request
     * @param $username
     *
     * @rbacName View users info
     * @rbacGroup User
     * @rbacAlias user.get
     * @rbacAllowedByDefault
     * @rbacSkipControllerCheck
     *
     * @return Response
     */
    function get($request, string $username)
    {
        $user = $this->username;

        $permissions = (new CfRBAC())->getPermissionsByRoles((new CfUsers($user))->getUserRoles());
        $getUserPermission = RbacAccessService::isActionAllowed('user.get', $permissions);

        $response = new Response($request);
        $payload = cfapi_user_get($user, $username, $getUserPermission);
        if ($payload)
        {
            $response->code = Response::OK;
            $response->body = $payload;
        }
        else
        {
            $response->code = Response::NOTFOUND;
        }

        return $response;
    }

    /**
     * @param $request
     * @param $username
     *
     * @rbacName Create user
     * @rbacGroup User
     * @rbacAlias user.put
     *
     * @return Response
     * @throws ResponseException
     */
    function put($request, string $username)
    {
        $user = $this->username;

        $data = Utils::getValidJsonData($request->data);

        $response = new Response($request);

        $this->validate($data);

        if (cfapi_user_put(
            $user,
            $username,
            $data->password ?? '',
            $data->name ?? '',
            $data->email ?? '',
            $data->time_zone ?? '',
            isset($data->roles) && is_array($data->roles) ? array_unique($data->roles) : '')
        ) {
            $response->code = Response::CREATED;
        } else {
            $response->code = Response::INTERNALSERVERERROR;
        }

        return $response;
    }

    /**
     * @param $request
     * @param $username
     *
     * @rbacName Update user
     * @rbacGroup User
     * @rbacAlias user.post
     * @rbacAllowedByDefault
     * @rbacSkipControllerCheck
     *
     * @return Response
     * @throws ResponseException
     */
    function post($request, string $username)
    {
        $user = $this->username;
        $headers = $this->_getallheaders();

        $data = Utils::getValidJsonData($request->data);

        $response = new Response($request);

        $this->validate($data);
        
        $isPasswordChanged = property_exists($data, 'password');

        if ($isPasswordChanged && CFAuthService::checkPasswordValid($username,$data->password)){
            $response->code = Response::UNPROCESSABLE_ENTITY;
            $response->body = "New password should be different from previous password";
            return $response;
        }

        $permissions = (new CfRBAC())->getPermissionsByRoles((new CfUsers($user))->getUserRoles());
        $updatePermission = RbacAccessService::isActionAllowed('user.post', $permissions);
        $passwordExpiration = $this->cfSettingsService->getSetting(SettingsHelper::PASSWORD_EXPIRATION_AFTER_RESET_HOURS);

        if (cfapi_user_post(
            $user,
            $username,
            $data->password ?? '',
            $data->name ?? '',
            $data->email ?? '',
            $data->time_zone ?? '',
            isset($data->roles) && is_array($data->roles) ? array_unique($data->roles) : '',
            $updatePermission))
        {
            $response->code = Response::ACCEPTED;
            if ($isPasswordChanged) {
                $isPasswordReset = isset($headers[self::CF_PASSWORD_RESET_HEADER]) && $headers[self::CF_PASSWORD_RESET_HEADER] === 'true';
                $updatedCfUser = new CfUsers($username);
                // if password reset then set expiration time now + RESET_PASSWORD_EXPIRE_AFTER_HOURS
                // if password updated but not reset then set it to null
                $hoursOffset = $isPasswordReset ? $passwordExpiration : null;
                $updatedCfUser->setPasswordExpirationTime($hoursOffset);
            }
        }
        else
        {
            $response->code = Response::INTERNALSERVERERROR;
        }

        return $response;
    }

    /**
     * @param $request
     * @param $username
     *
     * @rbacName Delete user
     * @rbacGroup User
     * @rbacAlias user.delete
     *
     * @return Response
     */
    function delete($request, string $username)
    {
        $user = $this->username;

        $response = new Response($request);

        if (cfapi_user_delete($user, $username))
        {
            $response->code = Response::ACCEPTED;
        }
        else
        {
            $response->code = Response::INTERNALSERVERERROR;
        }

        return $response;
    }


    /**
     * @param $data
     * @throws InvalidArgumentException
     * @return void
     * 
     * Throws InvalidArgumentException in case of not valid input data
     */
    private function validate($data): void
    {
        if (isset($data->time_zone)) 
        {
            Validator::timezone($data->time_zone);
        }

        if (property_exists($data, 'password')) {
            Validator::password($data->password);
        }

        if (isset($data->roles) && is_array($data->roles)) {
            Validator::rolesExists($data->roles);
        }
    }
}

/**
 * @uri /user 0
 */
class UserList extends CfProtectedResource
{
    /**
     * @param $request
     *
     * @rbacName User list
     * @rbacGroup User
     * @rbacAlias userList.get
     * @rbacAllowedByDefault
     *
     * @return Response
     * @throws ResponseException
     */
    function get($request)
    {

        $this->accessRequired();
        $user = $this->username;

        $username_rx = Utils::queryParam('id');
        $external = Utils::checkBoolean(Utils::queryParam('external'), 'external');
        $userModel = new UsersList();

        $response = new Response($request);

        $userListRequest = new UsersListRequestStructure($user,  $username_rx,  $external ? 'external' : 'internal');
        $response->body = $userModel->getUsersList($userListRequest);
        $response->code = Response::OK;

        return $response;
    }
}
