import { createAsyncThunk } from '@reduxjs/toolkit';

import Api from '@northern.tech/store/api/general-api';
import { LLMConfig } from './index';

const api_url = '/settingsApi/llmConfig';
const api_preferences_url = '/settingsApi/preferences';

export const getConfig = createAsyncThunk(
  'llm/getConfig',
  async () => {
    const { data } = await Api.get(api_url);
    return data;
  }
);

export const createConfig = createAsyncThunk(
  'llm/createConfig',
  async (config: LLMConfig, { rejectWithValue }) => {
    try {
      const { model, token, provider, base_url } = config;
      await Api.post(api_url, { model, token, provider, base_url });
      notify.success('LLM config successfully saved.');
      return config;
    } catch (error) {
      notify.error('Failed to save LLM config. Please check the application logs.');
      return rejectWithValue(error);
    }
  }
);

export const updateConfig = createAsyncThunk(
  'llm/updateConfig',
  async ({ id, config }: { id : number, config: LLMConfig }, { rejectWithValue }) => {
    try {
      const { model, token, provider, base_url } = config;
      await Api.put(`${api_url}/${id}/`, { model, token, provider, base_url });
      notify.success('LLM config successfully saved.');
      return config;
    } catch (error) {
      notify.error('Failed to save LLM config. Please check the application logs.');
      return rejectWithValue(error);
    }
  }
);

export const deleteConfig = createAsyncThunk(
  'llm/deleteConfig',
  async (id: number) => {
    await Api.delete(`${api_url}/${id}/`);
    notify.success('LLM config successfully deleted.');
    return { id };
  }
);

export const updatePreferences = createAsyncThunk(
  'llm/updatePreferences',
  async (preferences, { rejectWithValue }) => {
    try {
      await Api.put(api_preferences_url, preferences);
      notify.success('Preferences successfully updated.');
      return preferences;
    } catch (error) {
      notify.error('Failed to update preferences. Please check the application logs.');
      return rejectWithValue(error);
    }
  }
);

export const getPreferences = createAsyncThunk(
  'llm/getPreferences',
  async () => {
    const { data } = await Api.get(api_preferences_url);
    return data;
  }
);
