<?php

class SetupConstants
{
    public const SESSION_ID_HEADER = 'Cf-Setup-Session-Id';
    public const ERROR_INVALID_SESSION = 'Invalid session ID. Please use a new setup code.';
    public const ERROR_INTERNAL = 'Internal server error.';
    public const ADMIN_ROLES = ['admin'];
}

class Setup extends CfBaseResource
{
    /**
     * @var CfSetup
     */
    protected $cfSetup;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cfSetup = new CfSetup(\CfSettings::getInstance()->getConnection());
    }
}

/**
 * @uri /setup/complete
 */
class CompleteSetup extends Setup
{
    /**
     * @param $request
     * @return Response
     * @throws ResponseException
     */
    public function post($request)
    {
        $requestHeaders = getallheaders();
        $sessionId = $requestHeaders[SetupConstants::SESSION_ID_HEADER] ?? null;
        $response = new Response($request);

        if (!$this->cfSetup->isSessionValid($sessionId)) {
            throw new ResponseException(SetupConstants::ERROR_INVALID_SESSION, Response::NOTACCEPTABLE);
        }

        $data = Utils::getValidJsonData($request->data);
        Validator::required($data, ['password', 'username', 'email']);
        Validator::password($data->password);
        Validator::email($data->email);
        Validator::alphanumeric($data->username, 'username');

        if (cfapi_user_put(
            $data->username,
            $data->username,
            $data->password,
            $name = '',
            $data->email,
            $timezone = '',
            SetupConstants::ADMIN_ROLES
        )) {
            $this->cfSetup->completeSetup();
            $response->code = Response::CREATED;
            AuditLogService::register([
                AuditLogFields::ACTOR => $data->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::USER,
                AuditLogFields::OBJECT_ID => $data->username,
                AuditLogFields::OBJECT_NAME => $data->username,
                AuditLogFields::ACTION => AuditLogActions::CREATE,
                AuditLogFields::DETAILS => ["Created user `{$data->username}`."]
            ]);
        } else {
            $response->code = Response::INTERNALSERVERERROR;
            $response->body = SetupConstants::ERROR_INTERNAL;
        }

        return $response;
    }
}

/**
 * @uri /setup/code/validate
 */
class ValidateSetup extends Setup
{
    /**
     * @param $request
     * @return Response
     * @throws ResponseException
     */
    public function post($request)
    {
        $data = Utils::getValidJsonData($request->data);
        Validator::required($data, 'code');

        $result = $this->cfSetup->validateCode($data->code);
        $response = new Response($request);
        $response->body = json_encode($result);
        $response->code = Response::OK;

        return $response;
    }
}

/**
 * @uri /setup/status
 */
class SetupStatus extends CfBaseResource
{
    public function get($request)
    {
        $cfSetup = new CfSetup(\CfSettings::getInstance()->getConnection());
        $response = new Response($request);
        $response->body = json_encode([
            'is_setup_complete' => $cfSetup->isSetupComplete(),
        ]);
        $response->code = Response::OK;

        return $response;
    }
}
