import React, { useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigate, useParams } from 'react-router-dom';

import { Link } from '@mui/material';
import { makeStyles } from 'tss-react/mui';

import { TwoColumns } from '@northern.tech/common-ui/ConfigurationObject';
import { ActionDescriptor, TimeWrapper, TypeDescriptor } from '@northern.tech/common/auditlogs/ColumnComponents';

import { ChangeDescriptor, UserDescriptor } from './ColumnComponents';
import EventDetailsFallbackComponent from './EventDetailsFallbackComponent';
import { getAuditLogSelection } from './store/organizationSlice/selectors';
import { getAuditLog } from './store/organizationSlice/thunks';

const actionTitlesMap = {
  'Module input updated': 'Input data of Build project module updated',
  'Module added': 'Module added to Build project'
};
const titleProcessor = ({ type, action }) => (actionTitlesMap.hasOwnProperty(action) ? actionTitlesMap[action] : `${type} ${action.toLowerCase()}`);

const useStyles = makeStyles()(theme => ({
  wrapper: {
    maxWidth: 750
  },
  title: {
    marginTop: 24
  },
  header: {
    fontWeight: 500
  },
  twoColumns: {
    marginTop: 16,
    'div': {
      marginBottom: 7
    },
    'b': {
      color: theme.palette.gray[600],
      fontWeight: 500
    }
  },
  detailsBlock: {
    padding: '16px 20px',
    border: `1px solid ${theme.palette.primary.border}`,
    backgroundColor: theme.palette.background.code,
    borderRadius: 0,
    fontWeight: 500
  },
  goBack: {
    fontSize: 16,
    fontWeight: 500,
    textDecoration: 'none',
    color: `${theme.palette.text.primary} !important`,
    '&:hover': {
      color: 'inherit'
    },
    'i': {
      marginRight: 12
    }
  }
}));

export const AuditLogEvent = () => {
  const { id } = useParams();
  const navigate = useNavigate();
  const dispatch = useDispatch();
  const selectedLog = useSelector(getAuditLogSelection);
  const { classes } = useStyles();
  const [columnsData, setColumnsData] = useState({});

  useEffect(() => {
    dispatch(getAuditLog(id));
  }, [dispatch, id]);

  useEffect(() => {
    if (selectedLog) {
      setColumnsData({
        'Date:': TimeWrapper(selectedLog, 0),
        'Performed by:': UserDescriptor(selectedLog, 1),
        'Action:': ActionDescriptor(selectedLog, 2),
        'Type:': TypeDescriptor(selectedLog, 3),
        'Affected:': ChangeDescriptor(selectedLog, 4)
      });
    }
  }, [selectedLog]);

  if (!selectedLog) return null;

  const {
    action = '',
    object: { type = '' }
  } = selectedLog;
  const title = titleProcessor({ type, action });
  return (
    <div className={classes.wrapper}>
      <Link className={classes.goBack} variant="text" onClick={() => navigate(-1)}>
        <i className="bi bi-chevron-left" /> Go back
      </Link>
      <h1 className="capitalized-start">{title}</h1>
      <div style={{ marginTop: 20 }} className={classes.header}>
        Activity
      </div>
      <TwoColumns compact chipLikeKey={false} items={columnsData} className={classes.twoColumns} style={{ maxWidth: 400 }} />
      <div style={{ margin: '10px 0' }} className={classes.header}>
        Details
      </div>
      <EventDetailsFallbackComponent className={classes.detailsBlock} item={selectedLog} onClose={() => {}} />
    </div>
  );
};

export default AuditLogEvent;
