<?php

namespace Build\Controllers;

use Build\Entities\ModuleEntity;
use Build\Models\ModuleModel;
use Build\Models\ModulesListModel;

/**
 * @uri /build/modules
 */
class Modules extends \CfProtectedResource
{
    /**
     * @var ModulesListModel
     */
    private $modulesList;

    function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->modulesList = new ModulesListModel();
    }

    /**
     * @rbacName Get modules list
     * @rbacGroup Modules List API
     * @rbacAlias build.modules-list.get
     * @rbacAllowedByDefault
     */
    function get($request): \Response
    {
        $response = new \Response($request);
        $args = $_GET;
        if (
            isset($args['sortColumn']) &&
            (!property_exists(ModuleEntity::class, $args['sortColumn']) && $args['sortColumn'] != ModulesListModel::RELEVANCE_SORT_NAME)
        ) {
            throw new \InvalidArgumentException("Column `{$args['sortColumn']}` does not exist");
        }

        if (isset($args['sortColumn']) && $args['sortColumn'] == ModulesListModel::RELEVANCE_SORT_NAME && (!isset($args['searchQuery']) || empty($args['searchQuery']))) {
            throw new \InvalidArgumentException("Relevance sorting cannot be applied without search query");
        }

        $modules = $this->modulesList->get($args);
        $response->code = \Response::OK;
        $response->body = json_encode($modules);

        return $response;
    }

    /**
     * @rbacName Update modules list
     * @rbacGroup Modules List API
     * @rbacAlias build.modules-list.update
     * @rbacAllowedByDefault
     */
    function post($request): \Response
    {
        $response = new \Response($request);
        $this->modulesList->update(BUILD_MODULES_URL);
        $response->code = \Response::NOCONTENT;
        return $response;
    }
}

/**
 * @uri /build/modules/:name
 */
class Module extends \CfProtectedResource
{
    /**
     * @var ModulesModel
     */
    private $moduleModel;

    function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->moduleModel = new ModuleModel();
    }

    /**
     * @rbacName Get module by name
     * @rbacGroup Modules List API
     * @rbacAlias build.module.get
     * @rbacAllowedByDefault
     */
    function get($request, $name): \Response
    {
        $response = new \Response($request);

        $module = $this->moduleModel->get($name);

        if (!$module) {
            $response->code = \Response::NOTFOUND;
            return $response;
        }

        $response->code = \Response::OK;
        $response->body = json_encode($module);

        return $response;
    }
}

/**
 * @uri /build/modules/:name/:version
 */
class VersionedModule extends \CfProtectedResource
{
    /**
     * @var ModulesModel
     */
    private $moduleModel;

    function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->moduleModel = new ModuleModel();
    }

    /**
     * @rbacAlias build.module.get
     */
    function get($request, $name, $version): \Response
    {
        $response = new \Response($request);

        $module = $this->moduleModel->get($name, $version);

        if (!$module) {
            $response->code = \Response::NOTFOUND;
            return $response;
        }

        $response->code = \Response::OK;
        $response->body = json_encode($module);

        return $response;
    }
}
