<?php

class CfRBAC
{
    /**
     * @var PDO
     */
    private $dbConnection;

    const RBAC_PERMISSIONS_TABLE = 'rbac_permissions';
    const RBAC_ROLE_PERMISSION_TABLE = 'rbac_role_permission';

    /**
     * CfRole constructor.
     */
    public function __construct()
    {
        $this->dbConnection = CfPdo::getInstance()->getConnection();
    }

    public function getPermissionsByRoles(array $roles)
    {
        // use default permissions if user has no roles
        if(sizeof($roles) == 1 && $roles[0] == '') {
            $stmt = $this->dbConnection->prepare(
                sprintf('SELECT * FROM %s WHERE allowed_by_default = true  ORDER BY "application", "group", "name"', self::RBAC_PERMISSIONS_TABLE));
            $stmt->execute();
        } else {
            $placeHolders = implode(',', array_fill(0, count($roles), '?'));
            $stmt = $this->dbConnection->prepare(
                sprintf('SELECT * FROM %s WHERE alias IN (SELECT permission_alias FROM %s WHERE role_id IN (%s)) ORDER BY "application", "group", "name"',
                    self::RBAC_PERMISSIONS_TABLE,
                    self::RBAC_ROLE_PERMISSION_TABLE,
                    $placeHolders)
            );
            $stmt->execute($roles);
        }

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function assignPermission(string $role, string $alias)
    {
        $stmt = $this->dbConnection->prepare(
            sprintf("INSERT INTO %s (role_id, permission_alias) VALUES (:role, :alias)  ON CONFLICT (role_id, permission_alias) DO NOTHING",
                self::RBAC_ROLE_PERMISSION_TABLE)
        );
        $stmt->execute([':role' => $role, ':alias' => $alias]);
    }

    public function revokeAllPermissions(string $role)
    {
        $stmt = $this->dbConnection->prepare(
            sprintf("DELETE FROM %s WHERE role_id = ?", self::RBAC_ROLE_PERMISSION_TABLE)
        );
        $stmt->execute([$role]);
    }

    public function revokePermission(string $role, string $alias)
    {
        $stmt = $this->dbConnection->prepare(
            sprintf("DELETE FROM %s WHERE role_id = :role AND permission_alias = :alias", self::RBAC_ROLE_PERMISSION_TABLE));
        $stmt->execute([':role' => $role, ':alias' => $alias]);
    }

    public function isRolePermissionExist(string $role, string $alias)
    {
        $stmt = $this->dbConnection->prepare(
            sprintf("SELECT * FROM %s WHERE role_id = :role AND permission_alias = :alias",
                self::RBAC_ROLE_PERMISSION_TABLE, $role, $alias));
        $stmt->execute([':role' => $role, ':alias' => $alias]);
        return $stmt->rowCount() > 0;
    }

    public function isPermissionExist(string $alias)
    {
        $stmt = $this->dbConnection->prepare(sprintf("SELECT * FROM %s WHERE alias = ?",
            self::RBAC_PERMISSIONS_TABLE));
        $stmt->execute([$alias]);
        return $stmt->rowCount() > 0;
    }

    public function getPermissions()
    {
        return $this->dbConnection->query(sprintf('SELECT * FROM %s ORDER BY "application", "group", "name"',
            self::RBAC_PERMISSIONS_TABLE))->fetchAll(PDO::FETCH_ASSOC);
    }

}
