<?php

/**
 * CfFailedAttempts class is responsible for logging failed actions attempts (e.g.: authentication, password reset) and
 * returning time out before the next available attempt.
 * IP address is used as a user identifier. That means that a pause between attempts
 * will be applied to the IP not the real user.
 */
class CfFailedAttempts
{
    // fallback ip address. in case if the real IP address cannot be received for unknown reason
    public const FALLBACK_IP = '0.0.0.0';
    // reset attempts if more than 1 hour has passed since the last attempt
    public const RESET_ATTEMPTS_AFTER_SEC = 3600;
    // 30 seconds, 1 minute, 2 minutes, 5 minutes, 10 minutes.
    public const DEFAULT_DELAYS_IN_SECONDS = [30, 60, 120, 300, 600];
    // apply delays when the threshold is reached
    public const ATTEMPTS_THRESHOLD = 3;

    public const AUTHENTICATION_ACTION = 'authentication';
    public const PASSWORD_RESET_ACTION = 'password_reset';
    public const TWO_FA_ACTION = 'two_fa_verification';
    public const ALLOWED_ACTIONS = [self::AUTHENTICATION_ACTION, self::PASSWORD_RESET_ACTION, self::TWO_FA_ACTION];

    protected PDO $db;
    protected string $ipAddress;
    protected string $action;
    protected array $delayInSeconds;
    protected int $attemptsThreshold;
    protected int $resetAttemptsAfterSec;
    private string $table = 'failed_attempts';

    public function __construct(
        $ipAddress,
        string $action,
        array $delayInSeconds = self::DEFAULT_DELAYS_IN_SECONDS,
        int $attemptsThreshold = self::ATTEMPTS_THRESHOLD,
        int $resetAttemptsAfterSec = self::RESET_ATTEMPTS_AFTER_SEC
    ) {
        $this->db = CfSettings::getInstance()->getConnection();
        $this->ipAddress = $ipAddress ?: self::FALLBACK_IP;
        $this->action = $action;
        $this->delayInSeconds = $delayInSeconds;
        $this->attemptsThreshold = $attemptsThreshold;
        $this->resetAttemptsAfterSec = $resetAttemptsAfterSec;

        if (!in_array($action, self::ALLOWED_ACTIONS)) {
            throw new Exception('Invalid failed attempt action: ' . $action);
        }
    }

    /**
     * @return int|null
     *
     * Return delay in seconds if time out until time for the given IP address is greater than the current time.
     */
    public function timeOutBeforeNextAttempt(): ?int
    {
        $currentTimestamp = time();
        $fa = $this->getFailedAttempt();

        return ($fa && ($currentTimestamp < $fa->timeoutUntil)) ? $fa->timeoutSeconds : null;
    }

    public function logFailedAttempt(): void
    {
        $fa = $this->getFailedAttempt();

        if ($fa === null) {
            $this->createFailedAttempt();
        } else {
            $currentTimestamp = time();

            // set attempts to 1 when RESET_ATTEMPTS_AFTER_SEC passed since the last failed attempt
            $attempts = (($currentTimestamp - $fa->lastAttempt) > $this->resetAttemptsAfterSec) ?
                1 :
                $fa->attempts + 1;

            $timeOutSeconds = $attempts >= $this->attemptsThreshold ? $this->getNextDelay($fa->timeoutSeconds) : 0;
            $timeOutUntil = date('Y-m-d H:i:s', $currentTimestamp + $timeOutSeconds);

            $this->updateFailedAttempt($attempts, $timeOutUntil, $timeOutSeconds);
        }
    }

    private function getFailedAttempt(): ?FailedAttemptDTO
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE ip_address = ? AND action = ?");
        $stmt->execute([$this->ipAddress, $this->action]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? FailedAttemptDTO::fromArray($row) : null;
    }

    private function createFailedAttempt(): void
    {
        $statement = $this->db->prepare("INSERT INTO {$this->table} (ip_address, attempts, last_attempt, action) VALUES (?, 1, NOW(), ?)");
        $statement->execute([$this->ipAddress, $this->action]);
    }

    private function updateFailedAttempt(int $attempts, string $timeOutUntil, int $timeOutSeconds): void
    {
        $statement = $this->db->prepare(
            "UPDATE {$this->table}
                   SET attempts = :attempts, last_attempt = NOW(), timeout_until = :timeout_until, timeout_seconds = :timeout_seconds 
                   WHERE ip_address = :ip_address AND action = :action"
        );

        $statement->execute([
            'ip_address' => $this->ipAddress,
            'attempts' => $attempts,
            'timeout_until' => $timeOutUntil,
            'timeout_seconds' => $timeOutSeconds,
            'action' => $this->action,
        ]);
    }

    /**
     * @param int $currentDelay
     * @return int
     *
     * Returns the first delay that is greater than the current one
     * or returns the last delay from DELAYS_IN_SECONDS
     * what happens when the delay reaches its maximum.
     */
    private function getNextDelay(int $currentDelay): int
    {
        $delaysGreaterThanCurrent = array_filter($this->delayInSeconds, fn ($delay) => $delay > $currentDelay);
        $firstDelay = reset($delaysGreaterThanCurrent);
        return $firstDelay ?: $this->delayInSeconds[array_key_last($this->delayInSeconds)];
    }
}

readonly class FailedAttemptDTO
{
    public function __construct(
        public string $ipAddress,
        public string $action,
        public int $attempts,
        public int $lastAttempt,
        public int $timeoutUntil,
        public int $timeoutSeconds
    ) {
    }

    public static function fromArray(array $data): self
    {
        return new self(
            $data['ip_address'] ?? '',
            $data['action'] ?? '',
            $data['attempts'] ?? 0,
            strtotime($data['last_attempt'] ?? '') ?: 0,
            strtotime($data['timeout_until'] ?? '') ?: 0,
            $data['timeout_seconds'] ?? 0
        );
    }
}

class CfFailedAttemptsIdentifier
{
    public const CF_MP_SECRET_HEADER_KEY = 'Cf-Mp-Secret';
    public const CF_REAL_IP_HEADER_KEY = 'Cf-Real-Ip';

    /**
     *
     * Get Failed attempts identifier
     *
     * If there is Cf-Mp-Secret header and this header value equals to MP_CLIENT_SECRET
     * then we can use header Cf-Real-Ip that we set in Mission Portal.
     *
     * This is needed because when Mission Portal calls API we have bad value
     * inside $server['REMOTE_ADDR'] as it have Mission Portal server's IP.
     *
     * @param array $headers
     * @param array $server
     * @return string
     */
    public static function get(array $headers, array $server): string
    {
        $identifier = $server['REMOTE_ADDR'];
        if (
            isset($headers[self::CF_MP_SECRET_HEADER_KEY]) &&
            defined(constant_name: 'MP_CLIENT_SECRET') &&
            $headers[self::CF_MP_SECRET_HEADER_KEY] === MP_CLIENT_SECRET
        ) {
            $identifier = $headers[self::CF_REAL_IP_HEADER_KEY] ?? $server['REMOTE_ADDR'];
        }
        return $identifier;
    }
}
