<?php

require_once APPPATH . 'models/Entities/CF_Search.php';

class Search_save_model extends CI_Model
{

    var $tableName = 'report';
    var $scheduleTableName = 'report_schedule';
    var $errors = [];
    var $currentUserName = '';
    var $defaultFilter = '';


    function __construct()
    {
        //$this->output->enable_profiler(TRUE);
        parent::__construct();
    }

    /* we need this functions to filter schedules by user */
    function setCurrentUserName($username)
    {
        $this->currentUserName = $this->db->escape_str($username);
    }

    function getCurrentUserName()
    {
        return $this->currentUserName;
    }

    // we set default search in controller, but we need it in model for validation()
    function setDefaultFilter($filter)
    {
        $this->defaultFilter = $filter;
    }

    function validate($data, $overwrite=false, $is_admin = FALSE)
    {
        $valid = true;
        if (trim($data['label']) == '')
        {
            $valid = false;
            $this->setError('Provide a name to this query');
            return false;
        }

        if (trim($data['url']) == '')
        {
            $valid = false;
            $this->setError('URL is empty');
            return false;
        }

        if (trim($data['username']) == '')
        {
            $valid = false;
            $this->setError('username is empty');
            return false;
        }

        if ($this->_checkReadOnly($data) == TRUE)
        {
            $this->setError('This query is read only. Please provide another name.');
            return false;
        }

        $nameResult = $this->_validateNameQuery($data);

        if ($overwrite === false) {
            $item = $this->_nameAlreadyExist($nameResult);

            if(!empty($item)) {
                $valid = false;
                $this->setError($this->lang->line('advancedreports_query_exists'), 422); //422 Unprocessable Entity
            }
        }


        //check if user admin or owner of this report, otherwise - do not allow to save
        if ($this->_nameAlreadyExist($nameResult) === TRUE
                && ( ($is_admin === FALSE) && ($this->_checkReportOwner($nameResult,$data) === FALSE) ))
        {
            $this->clearErrors();
            $this->setError('You are not admin and not owner of this query. Please provide new name.', 422); //422 Unprocessable Entity
            $valid = false;
        }

        return $valid;
    }

    function _validateNameQuery($data)
    {
         $alreadyExist = false;
         $filter = $this->defaultFilter;

         if (isset($data['export_id'])) {
             $tmpFilter = sprintf("export_id = '%s' AND username = '%s'", $this->db->escape_str($data['export_id']), $this->db->escape_str($data['username']));
         } else {
             $tmpFilter = sprintf("label = '%s'", $this->db->escape_str($data['label']));
         }

         $filter = !empty($filter) ? $filter.' AND '.$tmpFilter : $tmpFilter;

         return $this->get_all_search($filter);
    }

    function _nameAlreadyExist($result)
    {
       return !empty($result);
    }

    function _checkReportOwner($result,$data)
    {
        $isOwner = false;

        if (!empty($result))
        {
            $reportOwner = $result[0]->getUsername();

            if ($reportOwner === $data['username'])
            {
                $isOwner = true;
            }
        }

        return $isOwner;
    }

    function _checkReadOnly($data)
    {
        $isReadOnly = false;
        $filter['label']  = $this->db->escape_str($data['label']);
        $filter['readonly'] = 1;
        $currentItem = $this->get_all_search($filter);

        if (!empty($currentItem))
        {
            $isReadOnly = true;
        }

        return $isReadOnly;
    }

    public function reportsListFilter(ReportCriteria $reportCriteria)
    {
        $filter = $this->_getExcludeOwnFilter($reportCriteria->isReportsAdmin(), $reportCriteria->getRoles());

        switch ($reportCriteria->getTab()) {
            case 'my':
                $filter = "username LIKE '{$reportCriteria->getUsername()}'";
                $favourites = $this->favourite_reports_model->list($reportCriteria->getUsername());
                if (sizeof($favourites) > 0) {
                    $filter = "( $filter OR id IN (" . implode(',', $favourites) . "))";
                }
                break;
            case 'ootb':
                $filter .= " AND meta_data @> '{\"source\": \"from CFEngine team\"}'::jsonb";
                break;
            case 'others':
                $filter .= " AND NOT (meta_data @> '{\"source\": \"from CFEngine team\"}'::jsonb OR meta_data @> '{\"source\": \"build module\"}'::jsonb)";
                break;
            case 'build':
                $filter .= " AND meta_data @> '{\"source\": \"build module\"}'::jsonb";
                break;
            default:
                $filter = "(username LIKE '{$reportCriteria->getUsername()}' OR is_public = 1)";
                break;
        }

        if (!empty($reportCriteria->getReportCategory())) {
            if (!empty($filter)) {
                $filter .= ' AND ';
            }
            $filter .= " reportCategory IN ('" . implode("','", escapeArrayValues($reportCriteria->getReportCategory())) . "')";
        }

        if (!empty($reportCriteria->getType())) {
            $filter .= (!empty($filter) ? 'AND' : '') . " type = '" . $this->db->escape_like_str($reportCriteria->getType()) . "' ";
        }

        if ($reportCriteria->isOwnReports()) {
            $filter .= (!empty($filter) ? 'AND' : '') . " username = '" . $this->db->escape_like_str($reportCriteria->getUsername()) . "' ";
        }

        return  $filter;
    }

    function insert($data, $overwrite = false, $is_admin = false)
    {
        if (!is_array($data))
        {
            $this->setError('Invalid parameter passed to function');
            return false;
        }

        if ($this->validate($data, $overwrite, $is_admin))
        {
            $id = '';
            $filter = $this->defaultFilter;

            // if id exists then filter by ID, otherwise by name
            if (array_key_exists('id', $data) && !empty($data['id'])) {
                $additionalFilter = 'id = ' . (int) $data['id'];
            } elseif (array_key_exists('export_id', $data) && !empty($data['export_id'])) {
                $additionalFilter = sprintf("export_id = '%s' AND username = '%s'", $this->db->escape_str($data['export_id']), $this->db->escape_str($data['username']));;
            } else {
                $additionalFilter = sprintf("label = '%s'", $this->db->escape_str($data['label']));
            }

            if (!isset($data['meta_data'])) {
                $data['meta_data'] = '{"source": "Create report UI"}';
            }

            $filter = !empty($filter) ? "$filter AND $additionalFilter" : $additionalFilter;
            $currentItem = $this->get_all_search($filter);

            if (!empty($currentItem))
            {
                $id = $currentItem[0]->getId();
                $reportOwner = $currentItem[0]->getUsername();

                // check report owner. if this report created by user and get's overwritten by admin, we will try to keep original owner
                // $data['username'] - user who created report, in most cases current logged user
                if ( ($reportOwner !== $data['username']) && $is_admin === TRUE)
                {
                    $data['username'] = $reportOwner;
                }

                // check permissions and remove users from shared list if they do not have permissions to see this report anymore
                $data['sharedBy'] = $this->_updateSharedByList($data, $currentItem);
            }

            $insertData = $this->_map_for_insert($data);
            if ($id == '')
            { // not exist - insert it will return inserted item
                $result = $this->db->insert($this->tableName, $insertData);
                if ($result)
                {
                    $id = $this->db->insert_id();
                }
            }
            else
            {
                $this->db->where('id', $id);
                $this->db->update($this->tableName, $insertData);
            }

            // get  object from DB  with ALL schedules
            //TODO: REFACTOR



            $offset = '';
            $limit = 1;
            $order = array();
            $showSchedules = 'all';
            $filter = array('id'=>$id);
            $searchObj = $this->get_all_search($filter, $offset, $limit, $order, $showSchedules, $is_admin);

            if ($searchObj)
            {
                return $searchObj[0];
            }
        }

        return false;
    }

    /**
     * Returns the data for inserting into PGSQL
     * @param  [type] $data [description]
     * @return [type]       [description]
     */
    function _map_for_insert($data)
    {
        unset($data['id']);
        $permissions = !empty($data['sharedPermission']) ? implode(",", $data['sharedPermission']) : '';
        $sharedBy = !empty($data['sharedBy']) ? implode(",", $data['sharedBy']) : '';
        $permissionSqlArray = '{' . $permissions . '}';
        $sharedBySqlArray = '{' . $sharedBy . '}';

        $mappedArray = array_change_key_case($data, CASE_LOWER);
        $mappedArray['advancedreportsdata'] = !empty($mappedArray['advancedreportsdata']) ? json_encode($mappedArray['advancedreportsdata']) : '{}';
        $mappedArray['sharedpermission'] = $permissionSqlArray;
        $mappedArray['sharedby'] = $sharedBySqlArray;

        $mappedArray['date'] = !empty($mappedArray['date']) ? date("Y-m-d H:i:s", $mappedArray['date']) : date('Y-m-d H:i:s');
        return $mappedArray;
    }

    /**
     * Return list of users who can still share reports
     * we need this in case if report roles were changed, to remove users who can not see report
     *
     * @param <array> $newReport
     * @param <CF_Search> $oldReport
     */
    function _updateSharedByList($newReport, $oldReport)
    {
        $oldReportSharedByList = $oldReport[0]->getSharedBy();

        if (empty($oldReportSharedByList))
        {
            return array();
        }

        // if report is public, or curent shared permission equal to the old one
        if (isset($newReport['is_public']) && $newReport['is_public'] == 1)
        {
            return $oldReportSharedByList;
        }

        // SHARING ISSUE
        //
        // In case new sharing options are not equal to old one we must remove users which can not see this report and their schedules.
        // To do this we need intersect between new sharedPermissions and users which subscribed to this report.
        //
        // BUT because only user with admin role can get information about users and roles via API we can not get this info as normal user and can not remove users and schedules
        // which do not have access to the report anymore
        //
        // So, for now, we will keep information about shared users in sharedBy column, we will also keep schedules and they will be active.

        // permission is different compare to old one, so we must remove users who can not see report.
        // also, ideally remove all schedules asigned to that user and this report
        if (isset($newReport['sharedPermission']) && ($newReport['sharedPermission'] !== $oldReport[0]->getSharedPermission()))
        {
            // we should remove users and schedules here, but we can't because of the API limitation.
            // so return list of old users for now
            return $oldReportSharedByList;
        }

        // if premissions are the same - return list as it is
        return $oldReportSharedByList;
    }

    /**
     * * Return record count according to supplied filter
     *
     * @param type $filter
     * @return type
     */
    function get_count_search($filter = array())
    {
        $result = $this->db->where($filter)->count_all_results($this->tableName);
        return $result;
    }


    /*
     * $showSchedules mode: user, all, check_only
     *
     * user - return only user schedules  and all schedules if admin
     * all- return all schedules
     * check_only - unset schedules and return hasSchedules = true if report has schedule
     */

    function get_all_search($filter = array(), $offset = '', $limit = '', $order = array(), $showSchedules = 'user', $is_admin = false)
    {

        if (!empty($limit) && $limit != '')
        {
            $this->db->limit($limit,$offset);
        }

        if (!empty($order))
        {
            if (is_array($order)) {
                $this->db->order_by( $order[0], $order[1]);
            } elseif (is_string($order)) {
                $this->db->order_by($order);
            }
        }

        $resultDb = $this->db->where($filter)->get($this->tableName);

        $result = array();

        foreach ((array) $resultDb->result_array() as $searchResult)
        {

            $searchResult = new CF_Search($searchResult);

            $scheduleFilter = array('reportId' => $searchResult->id);

            // filter schedules by username, user with admin role must see everything
            if ($is_admin !== TRUE  && $showSchedules == 'user')
            {
              $scheduleFilter['userid'] =  $this->currentUserName;
            }

            $querySchedules  = $this->db->where($scheduleFilter)->get($this->scheduleTableName);

            $searchResult->schedules = array();

            foreach ((array)$querySchedules->result_array() as $row)
            {
                $searchResult->schedules[] = $this->_formatScheduleDataForReportOutput($row);
            }

            if (!empty($searchResult->schedules))
            {
                foreach($searchResult->schedules as $item => $schedule)
                {
                    if ($schedule['userId'] == $this->currentUserName)
                    {
                        $searchResult->schedules[$item]['can_edit']   = 1;
                        $searchResult->schedules[$item]['can_delete'] = 1;
                    }
                    elseif($is_admin == TRUE)
                    {
                        $searchResult->schedules[$item]['can_delete'] = 1;
                        $searchResult->schedules[$item]['can_view']   = 1;
                    }
                }
            }


            // owner can do everything with own report
            // admin can edit/delete any report
            if ($is_admin === TRUE || $this->currentUserName === $searchResult->username)
            {
                $searchResult->can_edit   = TRUE;
                $searchResult->can_delete = TRUE;
            }
            else
            {
                $searchResult->can_edit   = FALSE;
                $searchResult->can_delete = FALSE;
            }

            // set flag if this is "own report" and set subsctibe flag (hide subscribe link (mostly for the is_public=1 reports))
            if ($searchResult->username == $this->currentUserName)
            {
                $searchResult->is_own = TRUE;
            }
            else
            {
                $searchResult->is_own = FALSE;
            }


            if (($searchResult->username !== $this->currentUserName) ) {
                $searchResult->can_subscribe = TRUE;
            }
            else
            {
                $searchResult->can_subscribe = FALSE;
            }

            // mark report as subscribed
            $searchResult->is_subscribed = FALSE;



            if (!empty($searchResult->sharedBy))
            {
                if (in_array($this->currentUserName, $searchResult->sharedBy))
                {
                    $searchResult->is_subscribed = TRUE;
                }
            }


            /*If report has schedules, and we interesting only if it has them or not -
             * unset schedules and set flaf hasSchedules
             *
             * this will reduce size of result JSON
             *
             */
            if ($showSchedules == 'check_only')
            {

                $searchResult->hasSchedules = FALSE;
                $searchResult->showScheduleStatistic = FALSE;
                if (!empty($searchResult->schedules))
                {
                    //calculate how many schedules this report has (own/total) available only for admin
                    $searchResult->ownSchedules   = count($this->_getSchedulesByUser($searchResult->schedules, $this->currentUserName));

                    if ($is_admin === TRUE)
                    {
                        $searchResult->showScheduleStatistic = TRUE;
                        $searchResult->totalSchedules = count($searchResult->schedules);
                        $searchResult->hasSchedules = TRUE;
                    }
                    else
                    {
                        // for normal user show how many schedules it has, and if =0 do not show schedules at all
                        if ($searchResult->ownSchedules >0)
                        {
                            $searchResult->hasSchedules = TRUE;
                        }
                    }

                    unset($searchResult->schedules);
                }
            }

            $result[] = $searchResult;
            //$obj = new CF_Search($searchResult);
            //$result[] = $obj;
        }
        return $result;
    }

    /**
     * Gets report schedule by id
     * @param  [type] $queryId [description]
     * @return [type]          [description]
     */
    function get_schedule_by_id($id)
    {
        $table = 'report_schedule';
        $filter['id'] = $id;
        $query = $this->db->where($filter)->get($table);
        return $query->first_row('array');
    }

    function delete($filter = array())
    {
        $return = $this->db->where($filter)->delete($this->tableName);
        if (!$return)
        {
            $msg = $this->db->_error_message();
            $num = $this->db->_error_number();
            $this->setError($msg,$num);

        }
        return $return;
    }

    function setError($errorText, $errorCode = 406)
    {
        $i = sizeof($this->errors);
        $this->errors[$i]['text']      = $errorText;
        $this->errors[$i]['errorCode'] = $errorCode;
    }

    function getErrors()
    {
        return $this->errors;
    }

    function clearErrors()
    {
        $this->errors = array();
    }

    /**
     * Formats the data from database
     * @param  [type] $scheduleData [description]
     * @return [type]               [description]
     */
    function _formatScheduleDataForReportOutput($scheduleData)
    {
        $scheduleData['scheduleData'] = json_decode($scheduleData['scheduledata'],true);
        $scheduleData['hostcontexts'] = json_decode($scheduleData['hostcontexts']);
        $scheduleData['outputTypes']  = PgsqlArrayToPhpArray($scheduleData['outputtypes']);
        $scheduleData['scheduleHumanReadableTime'] = $scheduleData['schedulehumanreadabletime'];
        $scheduleData['scheduleName'] = $scheduleData['schedulename'];
        $scheduleData['userId']       = $scheduleData['userid'];


        unset($scheduleData['scheduledata']);
        unset($scheduleData['outputtypes']);
        unset($scheduleData['schedulename']);
        unset($scheduleData['schedulehumanreadabletime']);
        unset($scheduleData['userid']);


        return $scheduleData;

    }

    function _getSchedulesByUser($data, $user)
    {
        $userSched = array();
        if(!empty($data))
        {
            foreach($data as $item) {
                if ($item['userId'] == $user)
                {
                    $userSched[] = $item;
                }
            }
        }
       return $userSched;
    }

    /**
     * Allow to subscribe only if
     *   - user is not owner
     *   - this is public report OR visible by roles limitation
     *
     *  user with admin role can subscribe to any report where he is not an owner
     *
     *  if public report created by this username - not allow to subscribe
     *
     * @param type $reportId
     * @param type $username
     * @param type $roles
     * @param type $is_admin
     * @return boolean
     */
    function subscribeToReport($reportId, $username, $roles, $is_admin)
    {
        if (empty($reportId) || empty($username))
        {
            $this->setError($this->lang->line('advancedreports_report_id_empty'));
            return false;
        }

        $filter = $this->_getExcludeOwnFilter($is_admin, $roles);
        $idFilter = sprintf('id = %d', (int) $reportId);

        $filter = !empty($filter) ? $filter.' AND '.$idFilter : $idFilter;


        $offset = '';
        $limit = 1;
        $order = array();
        $showSchedules = 'check_only';

        $searchObj = $this->get_all_search($filter, $offset, $limit, $order, $showSchedules, $is_admin);



        // check if report exist
        if (empty($searchObj) || ($searchObj[0]->getId() != $reportId))
        {
            $this->setError($this->lang->line('advancedreports_report_not_found'));
            return false;
        }

        $sharedBy = '';

        if (!empty($searchObj[0]->sharedBy)) {
            // check if report already has subscribers - if so, add username to array and convert to string
            $sharedBy = $searchObj[0]->sharedBy;
            $sharedBy[] = $username;
            $sharedBy = implode(",", $sharedBy);
        }
        else {
            $sharedBy = $username;
        }

        $updateData = array('sharedBy' => '{'.$sharedBy.'}');
        $result     = $this->db->where($filter)->update($this->tableName, $updateData);

        return (bool)$this->db->affected_rows();
    }


    function unSubscribeFromReport($reportId, $username, $is_admin)
    {
        if (empty($reportId) || empty($username))
        {
            $this->setError($this->lang->line('advancedreports_report_id_empty'));
            return false;
        }

        $filter = sprintf("id = %s and '{%s}' <@ sharedBy", $this->db->escape_str($reportId), $this->db->escape_str($username));


        $offset = '';
        $limit = 1;
        $order = array();
        $showSchedules = 'all';

        $searchObj = $this->get_all_search($filter, $offset, $limit, $order, $showSchedules, $is_admin);

        // check if report exist
        if (empty($searchObj) || ($searchObj[0]->getId() !== $reportId))
        {
            $this->setError($this->lang->line('advancedreports_report_not_found'));
            return false;
        }


        // check if already unSubscribed
        if (!in_array($username, $searchObj[0]->getSharedBy()))
        {
            $this->setError($this->lang->line('advancedreports_unable_to_unsubscribe'));
            return false;
        }

        // check if this user has schedules for this report and delete them from phpcfenfine and cfreport DB
        $schedules = $this->_getSchedulesByUser($searchObj[0]->getSchedules(), $this->currentUserName);

        if (!empty($schedules))
        {
            $this->load->model('advancedreports_model');

            foreach ($schedules as $item)
            {
                $scheduledData = array();
                $scheduledData['userId'] = $this->currentUserName;
                $scheduledData['id']     = $item['id'];

                    // delete schedule from cfreports
                    try
                    {
                        $obj = $this->advancedreports_model->deleteScheduledReport_REST($scheduledData);

                        // delete schedule from phpcfengine
                        $obj = $this->advancedreports_model->deleteSchedule($reportId, $item['id']);
                    }
                    catch (Exception $e)
                    {
                        $this->search_save_model->setError($this->lang->line('advancedreports_schedule_delete_error'), 406);
                        return false;
                    }
            }
        }


        $sql = 'UPDATE report SET sharedBy = array_remove(sharedBy,?) where id=?';
        return $this->db->query($sql,array($username,$reportId));
    }



    function copyReport($data, $username, $roles, $is_admin, $overwrite)
    {
        if (empty($data['reportId']) || empty($data['reportLabel']) || empty($username) || empty($data['reportCategory']))
        {
            $this->setError($this->lang->line('advancedreports_save_not_all_parameters_set'));
            return false;
        }

        $filter = $this->_getExcludeOwnFilter($is_admin, $roles);
        $idFilter = sprintf('id = %s',$data['reportId']);
        $filter = !empty($filter) ? $filter.' AND '.$idFilter : $idFilter;




        $offset = '';
        $limit = 1;
        $order = array();
        $showSchedules = 'check_only';

        $itemLoaded = $this->get_all_search($filter, $offset, $limit, $order, $showSchedules, $is_admin);

        // check if report exist
        if (empty($itemLoaded) || ($itemLoaded[0]->getId() != $data['reportId']))
        {
            $this->setError($this->lang->line('advancedreports_report_not_found'));
            return false;
        }

        // create new object
        $newReport = array(
            'username'   => $username,
            'url'        => $itemLoaded[0]->getUrl(),
            'reportType' => $itemLoaded[0]->getReportType(),
            'type'       => $itemLoaded[0]->getType(),
            'label'      => $data['reportLabel'],
            'reportCategory'   => $data['reportCategory'],
            'date'       => time(),
            'params'     => $itemLoaded[0]->getSQLString(),
            'sharedPermission'    => $data['sharedPermission'],
            'is_public'  => (bool)$data['is_public'] ? 1: 0,
            'AdvancedReportsData' => $itemLoaded[0]->getAdvancedReportsData()
        );

        $obj = $this->search_save_model->insert($newReport, $overwrite, $is_admin);

        return $obj;
    }


    /**
     * Return userbased filter - display reports created or subscribed by user if he has enough roles
     *
     * @return array
     */
    function _getUserbasedFilter($is_admin, $roles = array())
    {

        $sqlCondition = '';

        $filter = array();

        $usernameArrayForSql = '{' . $this->currentUserName.'}';
        $rolesArrayForSql = $this->getEscapedRolesArrayForSql($roles);

        if (empty($this->currentUserName))
        {
            $this->setError('Unable to set filter. Username is not set');
            return false;
        }


        if ($is_admin == TRUE) // don't care about roles
        {
            /*$filter = array('$or' => array(
                                            array('username' => $this->currentUserName),
                                            array('sharedBy' => array('$in' => array($this->currentUserName)),
                                          )
            ));*/
            $sqlCondition = sprintf("(username = '%s' OR  '%s' <@ sharedby)",$this->currentUserName,$usernameArrayForSql);
        }
        else
        {
            /*$filter = array('$or' => array(
                                            array('username' => $this->currentUserName),
                                            array('$or' =>
                                                array(
                                                        array('$and' => array
                                                                        (
                                                                            array('sharedBy' => array('$in' => array($this->currentUserName))),
                                                                            array('sharedPermission' => array('$in' => $roles))
                                                                        )
                                                            ),
                                                        array('$and' => array
                                                                        (
                                                                            array('sharedBy' => array('$in' => array($this->currentUserName))),
                                                                            array('is_public'=> "1")
                                                                        )
                                                            )
                                                    )
                                                )
                                        )
                );*/
            $sqlCondition = sprintf("(username = '%s' OR (('%s' <@sharedby ) AND (is_public = 1 OR '%s' && sharedPermission)))",$this->currentUserName,$usernameArrayForSql,$rolesArrayForSql);
        }

        return $sqlCondition;
    }


    /**
     * filter to display all reports except reports created by user if they are not public
     *
     *  - if user doesn't have any roles - only public where he is not author
     *  - if user has roles - show public + reports available for him by roles where he is not author
     *  - if admin - all reports where he is not author
     *
     *
     * @return array
     */
    function _getExcludeOwnFilter($is_admin, $roles)
    {
        $sqlCondition = '';
        $rolesArrayForSql = $this->getEscapedRolesArrayForSql($roles);

        if (empty($this->currentUserName))
        {
            $this->setError('Unable to set filter. Username is not set.');
            return FALSE;
        }

        //not admin and don't have roles - only public
        if ($is_admin == false && empty($roles))
        {
            $filter['is_public']   = '1';
            $filter['username'] = array('$ne' => $this->currentUserName);

            $sqlCondition = sprintf("(is_public = 1 and username <> '%s')",$this->currentUserName);

            return $sqlCondition;
        }

        if ($is_admin === FALSE)
        {
            $filter = array('$or' =>
                              array(
                                  // public and not author
                                    array(
                                          '$and' =>
                                                 array(
                                                        array('is_public'   => "1"),
                                                        array('username' => array('$ne' => $this->currentUserName))
                                                      )
                                         ),
                                  // not public, available by role not author
                                    array(
                                          '$and' =>
                                                 array(
                                                        array('is_public'   => array('$ne' => "1")),
                                                        array('username' => array('$ne' => $this->currentUserName)),
                                                        array('sharedPermission' => array('$in' => $roles))
                                                      )
                                        )
                                   )
                       );

         $sqlCondition = sprintf("(username <> '%s' AND ( is_public = 1 OR (is_public <> 1  AND '%s' && sharedPermission)))", $this->currentUserName, $rolesArrayForSql);
        }

        // admin must see everything ecxept own reports, so we don't care about roles
        if ($is_admin == TRUE)
        {
            $filter['username'] = array('$ne' => $this->currentUserName);
            $sqlCondition = sprintf("(username <> '%s')", $this->currentUserName);
        }

        return $sqlCondition;
    }

    /**
     * This filter return all repors available to user.
     * for example in load()user should be able to run any report - own, subscribed and allowed to see
     * but in case allowed reports - button set in view must be different
     *
     *
     * @param type $is_admin
     * @param type $roles
     */

    public function _getAllAvailableReportsFilter($is_admin, $roles = [])
    {

        $sqlCondition = '';
        $rolesArrayForSql = $this->getEscapedRolesArrayForSql($roles);

        if (empty($this->currentUserName)) {
            $this->setError('Unable to set filter. Username is not set.');
            return FALSE;
        }

        if ($is_admin === FALSE) {
            $sqlCondition = empty($roles) ?
                sprintf("(is_public = 1 OR username = '%s')", $this->currentUserName) :
                sprintf("(is_public = 1  OR username = '%s' OR  sharedPermission && '%s')", $this->currentUserName, $rolesArrayForSql);
        }

        return $sqlCondition;
    }

    public function generateExportId(CF_Search $report, $username)
    {
        $newExportId = strtolower(str_replace(' ', '-', $report->label));

        $reportWithSameId = $this
            ->get_all_search(
                $filter = ['export_id' => $newExportId, 'username' => $username],
                $offset = 0,
                $limit = 1,
                $order = ['export_id', 'DESC']
            );

        if ($reportWithSameId) {
            $newExportId = $newExportId .'-'. (intval(preg_replace('/[^0-9]/', '', $reportWithSameId[0]->export_id)) + 1);
        }

        $sql = 'UPDATE report SET export_id = ? where id = ?';
        $this->db->query($sql, [$newExportId, $report->id]);
        return $newExportId;
    }

    private function getEscapedRolesArrayForSql($roles)
    {
      return  '{'.implode(',', escapeArrayValues($roles)).'}';
    }
}

class ReportCriteria
{
    private $username;
    private $roles;
    private $id;
    private $isReportsAdmin;
    private $tab;
    private $type;
    private $reportCategory;
    private $ownReports;

    public function __construct(
        $username = null,
        $roles = [],
        $isReportsAdmin = false,
        $id = null,
        $tab = null,
        $type = null,
        $reportCategory = [],
        $ownReports = false
    )
    {
        $this->username = $username;
        $this->roles = $roles;
        $this->id = $id;
        $this->isReportsAdmin = $isReportsAdmin;
        $this->tab = $tab;
        $this->type = $type;
        $this->reportCategory = $reportCategory;
        $this->ownReports = $ownReports;
    }

    public function getUsername()
    {
        return $this->username;
    }

    public function setUsername(string $username)
    {
        $this->username = $username;
        return $this;
    }

    public function getRoles()
    {
        return $this->roles;
    }

    public function setRoles($roles)
    {
        $this->roles = $roles;
        return $this;
    }

    public function getId()
    {
        return $this->id;
    }

    public function setId($id)
    {
        $this->id = $id;
        return $this;
    }

    public function isReportsAdmin()
    {
        return $this->isReportsAdmin;
    }

    public function setIsReportsAdmin($isReportsAdmin)
    {
        $this->isReportsAdmin = $isReportsAdmin;
        return $this;
    }

    public function getTab()
    {
        return $this->tab;
    }

    public function setTab($tab)
    {
        $this->tab = $tab;
        return $this;
    }

    public function getType()
    {
        return $this->type;
    }

    public function setType($type)
    {
        $this->type = $type;
        return $this;
    }

    public function getReportCategory()
    {
        return $this->reportCategory;
    }

    public function setReportCategory($reportCategory)
    {
        $this->reportCategory = $reportCategory;
        return $this;
    }

    public function isOwnReports()
    {
        return $this->ownReports;
    }

    public function setOwnReports($ownReports)
    {
        $this->ownReports = $ownReports;
        return $this;
    }
}
