import { createSlice } from '@reduxjs/toolkit';
import { getRoles, getProfileData, getAdditionalData, updateAdditionalData } from './thunks';

export interface Role {
    id: string;
    description: string;
}

export interface ProfileData {
  email: string;
  external: boolean;
  id: string;
  name: string;
  roles: string[];
  time_zone: string;
  two_factor_enabled: boolean;
  locked: boolean;
  is2FaVerificationNeeded: boolean;
}
export interface AdditionalData {
  build_app_opened: boolean;
  groups_app_opened: boolean;
  first_login_after_reset: boolean;
  is_ai_agent_opened: boolean;
}

interface UsersState {
    roles: Role[];
    isLoading: boolean;
    error: string|null;
    profileData: ProfileData;
    additionalData: AdditionalData;
}

const initialState: UsersState = {
    roles: [],
    isLoading: false,
    error: null,
    profileData: {
        email: '',
        external: false,
        id: '',
        name: '',
        roles: [],
        time_zone: '',
        two_factor_enabled: false,
        locked: false,
        is2FaVerificationNeeded: false,
    },
    additionalData: {
        build_app_opened: true,
        groups_app_opened: true,
        first_login_after_reset: false,
        is_ai_agent_opened: true
    }
};

const aiChatSlice = createSlice({
    name: 'users',
    initialState,
    reducers: {},
    extraReducers: (builder) => {
        builder
            .addCase(getRoles.pending, (state) => {
                state.isLoading = true;
                state.error = null;
            })
            .addCase(getRoles.fulfilled, (state, action) => {
                state.isLoading = false;
                state.roles = action.payload;
            })
            .addCase(getRoles.rejected, (state, action) => {
                state.isLoading = false;
                state.error = action.error.message || 'Failed to fetch users roles.';
            });

        builder
            .addCase(getProfileData.pending, (state) => {
                state.isLoading = true;
                state.error = null;
            })
            .addCase(getProfileData.fulfilled, (state, action) => {
                state.isLoading = false;
                state.profileData = action.payload;
            })
            .addCase(getProfileData.rejected, (state, action) => {
                state.isLoading = false;
                state.error = action.error.message || 'Failed to fetch profile data.';
            });

        builder
            .addCase(getAdditionalData.pending, (state) => {
                state.isLoading = true;
                state.error = null;
            })
            .addCase(getAdditionalData.fulfilled, (state, action) => {
                const {key, value} = action.payload;
                state.isLoading = false;
                state.additionalData[key] = value;
            })
            .addCase(getAdditionalData.rejected, (state, action) => {
                state.isLoading = false;
                state.error = action.error.message || 'Failed to fetch profile data.';
            });

        builder
            .addCase(updateAdditionalData.pending, (state) => {
                state.isLoading = true;
                state.error = null;
            })
            .addCase(updateAdditionalData.fulfilled, (state, action) => {
                const {key, value} = action.payload;
                state.isLoading = false;
                state.additionalData[key] = value;
            })
            .addCase(updateAdditionalData.rejected, (state, action) => {
                state.isLoading = false;
                state.error = action.error.message || 'Failed to update profile data.';
            });
    },
});

export default aiChatSlice.reducer;
