import React from 'react';
import {
  Box,
  TextField,
  FormHelperText,
  Typography
} from '@mui/material';

interface PolicyEvaluationValue {
  files?: string;
  bundles?: string;
}
interface PolicyEvaluationFormProps {
  onChange: (value: PolicyEvaluationValue) => void;
  errors?: Record<string, string>;
  files?: string | string[];
  bundles?: string | string[];
  classes?: Record<string, string>;
}

interface PolicyEvaluationViewProps {
  files?: string | string[];
  bundles?: string | string[];
  classes?: Record<string, string>;
  onValueClick?: () => void;
}

export const PolicyEvaluationForm: React.FC<PolicyEvaluationFormProps> = ({ 
  onChange, 
  errors = {}, 
  files, 
  bundles, 
  classes = {} 
}) => {
  const filesValue = Array.isArray(files) ? files.join("\n") : (files || '');
  const bundlesValue = Array.isArray(bundles) ? bundles.join("\n") : (bundles || '');

  return (
    <Box sx={{ mt: 2, mb: 2 }}>
      <Box className={classes.formRow}>
        <Typography variant="body1" className={classes.label}>
          List of policy files:
        </Typography>
        <Box className={classes.fieldContainer}>
          <TextField
            fullWidth
            variant="outlined"
            multiline
            rows={4}
            size="small"
            placeholder="Policy files"
            value={filesValue}
            onChange={(e) => onChange({ bundles: bundlesValue, files: e.target.value })}
            error={!!errors.files}
            helperText={errors.files}
          />
          <FormHelperText>Press enter to add new value</FormHelperText>
          {errors?.value && (
            <FormHelperText error>
              {errors.value}
            </FormHelperText>
          )}
        </Box>
      </Box>
      <Box className={classes.formRow}>
        <Typography variant="body1" className={classes.label}>
          List of bundles:
        </Typography>
        <Box className={classes.fieldContainer}>
          <TextField
            fullWidth
            variant="outlined"
            multiline
            rows={4}
            size="small"
            placeholder="Policy bundles"
            value={bundlesValue}
            onChange={(e) => onChange({ files: filesValue, bundles: e.target.value })}
            error={!!errors.bundles}
            helperText={errors.bundles}
          />
          <FormHelperText>Press enter to add new value</FormHelperText>
        </Box>
      </Box>
    </Box>
  );
};

export const PolicyEvaluationView: React.FC<PolicyEvaluationViewProps> = ({ 
  files, 
  bundles, 
  classes = {},
  onValueClick
}) => {
  const filesValue = Array.isArray(files) ? files.join("\n") : (files || '');
  const bundlesValue = Array.isArray(bundles) ? bundles.join("\n") : (bundles || '');
  return (
    <Box>
      <Box className={classes.container}>
        <Typography className={classes.key}>
          List of policy files:
        </Typography>
        <Box onClick={onValueClick} className={classes.value}>
          <TextField
            fullWidth
            variant="outlined"
            multiline
            rows={4}
            size="small"
            placeholder="Policy files"
            value={filesValue}
          />
        </Box>
      </Box>
      <Box className={classes.container}>
        <Typography className={classes.key}>
          List of bundles:
        </Typography>
        <Box onClick={onValueClick} className={classes.value}>
          <TextField
            fullWidth
            variant="outlined"
            multiline
            rows={4}
            size="small"
            placeholder="Policy bundles"
            value={bundlesValue}
          />
        </Box>
      </Box>
    </Box>
  );
};