<?php

namespace Build\Controllers;

use AuditLogActions;
use AuditLogObjectTypes;
use AuditLogFields;
use AuditLogService;
use Build\Exceptions\CfbsNotFoundException;
use Build\Exceptions\InvalidCfbsFormatException;
use Build\Exceptions\InvalidCfbsRequestException;
use Build\Models\CfbsRequestsModel;
use Build\Models\ModuleModel;
use Build\Models\ModulesListModel;
use Build\Models\ProjectModel;
use Build\Models\ProjectModulesModel;

/**
 * @uri /build/projects/:id/modules
 */
class ProjectModules extends \CfProtectedResource
{
    /**
     * @var ProjectModulesModel
     */
    private $projectModules;

    function __construct($parameters)
    {
        parent::__construct($parameters);

        $this->projectModules = new ProjectModulesModel();
    }

    /**
     * @rbacName Get modules added to project
     * @rbacGroup Build Projects API
     * @rbacAlias build.project-modules.get
     */
    function get($request, $projectId): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        $project = (new ProjectModel($this->username))->get((int)$projectId);
        if (!$project) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        try {
            $modules = $this->projectModules->list((int)$projectId);
            $this->projectModules->markExternalModules($modules, new ModulesListModel());
            $response->body = json_encode($modules);
        } catch (CfbsNotFoundException $exception) {
            $response->code = \Response::NOTFOUND;
            $response->body = $exception->getMessage();
        } catch (InvalidCfbsFormatException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        }

        return $response;
    }
}


/**
 * @uri /build/projects/:id/modules/:module
 */
class ProjectModule extends \CfProtectedResource
{
    /**
     * @var ProjectModulesModel
     */
    private $projectModules;

    /**
     * @var ProjectModel
     */
    private $projectModel;

    function __construct($parameters)
    {
        parent::__construct($parameters);

        $this->projectModules = new ProjectModulesModel();
        $this->projectModel = new ProjectModel($this->username);
        $this->cfbsRequestsModel = new CfbsRequestsModel();
    }

    /**
     * @rbacName Add module to project
     * @rbacGroup Build Projects API
     * @rbacAlias build.project-modules.post
     */
    function post($request, $projectId, $moduleName): \Response
    {
        if (isset($_GET['encodedName']) && $_GET['encodedName'] == 'true') {
            $moduleName = base64_decode($moduleName);
        }

        $response = new \Response($request);
        $response->code = \Response::CREATED;
        $requestData = \Utils::getValidJsonData($request->data);

        if (!isset($requestData->version) || empty($requestData->version)) {
            $response->code = \Response::BADREQUEST;
            $response->body = 'Version is required.';
            return $response;
        }

        if (!($project = $this->projectModel->get((int)$projectId))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::ADD_MODULES_REQUEST,
            [
                'project_id' => $projectId,
                'modules' => [['name' => $moduleName, 'version' => $requestData->version]],
                'git' => getGitDataFromRequest($requestData)
            ]
        );

        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
            $this->projectModel->updateIsDeployedLocally((int)$projectId, false);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $projectId,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::MODULE_ADDED,
                AuditLogFields::DETAILS => ["Added module `$moduleName` to project #$projectId."]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while adding module to build project.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }

    /**
     * @rbacName Remove module from project
     * @rbacGroup Build Projects API
     * @rbacAlias build.project-modules.delete
     */
    function delete($request, $projectId, $moduleName): \Response
    {
        if (isset($_GET['encodedName']) && $_GET['encodedName'] == 'true') {
            $moduleName = base64_decode($moduleName);
        }
        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;
        $requestData =  (isset($request->data) && !empty($request->data)) ? \Utils::getValidJsonData($request->data) : [];

        if (!($project = $this->projectModel->get((int)$projectId))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::REMOVE_MODULES_REQUEST,
            [
                'project_id' => $projectId,
                'modules' => [['name' => $moduleName]],
                'git' => getGitDataFromRequest($requestData)
            ]
        );

        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
            $this->projectModel->updateIsDeployedLocally((int)$projectId, false);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $projectId,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::MODULE_DELETED,
                AuditLogFields::DETAILS => ["Deleted `$moduleName` from project #$projectId."]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while removing module.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }

    /**
     * @rbacName Update module version
     * @rbacGroup Build Projects API
     * @rbacAlias build.project-modules.patch
     */
    function patch($request, $projectId, $moduleName): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;
        $requestData = \Utils::getValidJsonData($request->data);

        if (!isset($requestData->version) || empty($requestData->version)) {
            $response->code = \Response::BADREQUEST;
            $response->body = 'Version is required.';
            return $response;
        }

        if (!($project = $this->projectModel->get((int)$projectId))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::UPDATE_MODULES_REQUEST,
            [
                'project_id' => $projectId,
                'modules' => [['name' => $moduleName, 'version' => $requestData->version]],
                'git' => getGitDataFromRequest($requestData)
            ]
        );

        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
            $this->projectModel->updateIsDeployedLocally((int)$projectId, false);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $projectId,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::MODULE_UPDATED,
                AuditLogFields::DETAILS => ["Updated the module `$moduleName` to version `{$requestData->version}` in project #$projectId."]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while updating module.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }
}

/**
 * @uri /build/projects/:id/modules/:module/input
 */
class ProjectModuleInput extends \CfProtectedResource
{
    private $moduleModel;
    private $projectModel;
    private $cfbsRequestsModel;

    function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->moduleModel = new ModuleModel();
        $this->projectModel = new ProjectModel($this->username);
        $this->cfbsRequestsModel = new CfbsRequestsModel();
    }

    /**
     * @rbacName Get module input
     * @rbacGroup Build Projects API
     * @rbacAlias build.module.input.get
     */
    function get($request, $projectId, $moduleName): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        $module = $this->moduleModel->get($moduleName);

        if (!$this->projectModel->get((int)$projectId)) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        if (!$module) {
            $response->body = 'Module not found';
            $response->code = \Response::NOTFOUND;
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::GET_MODULE_INPUT_REQUEST,
            [
                'project_id' => $projectId,
                'module' => ['name' => $moduleName]
            ]
        );

        try {
            $reply = $this->cfbsRequestsModel->processRequestResponse($requestID);
            $response->body = json_encode($reply);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while getting module input.';
            syslog(LOG_ERR, $response->body . '. Exception trace:' . $exception->getTraceAsString());
        }

        return $response;
    }

    /**
     * @rbacName Set module input
     * @rbacGroup Build Projects API
     * @rbacAlias build.module.input.set
     */
    function post($request, $projectId, $moduleName): \Response
    {
        $response = new \Response($request);
        $response->code = \Response::OK;
        $requestData = \Utils::getValidJsonData($request->data);

        $module = $this->moduleModel->get($moduleName);
        if(\Utils::getValidJsonData($request->data)) {
            $input = $request->data;
        }

        if (!($project = $this->projectModel->get((int)$projectId))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        if (!$module) {
            $response->body = 'Module not found';
            $response->code = \Response::NOTFOUND;
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::SET_MODULE_INPUT_REQUEST,
            [
                'project_id' => $projectId,
                'module' => $moduleName,
                'input' => json_decode($input),
                'git' => getGitDataFromRequest($requestData)
            ]
        );

        try {
            $reply = $this->cfbsRequestsModel->processRequestResponse($requestID);
            $this->projectModel->updateIsDeployedLocally((int)$projectId, false);
            $response->body = json_encode($reply);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $projectId,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::MODULE_INPUT_UPDATED,
                AuditLogFields::DETAILS => ["Changed input data for the `$moduleName` module in project #$projectId.", ['input' => json_decode($input)]]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while setting module input.';
            syslog(LOG_ERR, $response->body . '. Exception trace:' . $exception->getTraceAsString());
        }

        return $response;
    }
}


/**
 * @uri /build/projects/:id/modules-by-url
 */
class ProjectModuleUrl extends \CfProtectedResource
{
    /**
     * @var ProjectModulesModel
     */
    private $projectModules;

    /**
     * @var ProjectModel
     */
    private $projectModel;

    function __construct($parameters)
    {
        parent::__construct($parameters);

        $this->projectModules = new ProjectModulesModel();
        $this->projectModel = new ProjectModel($this->username);
        $this->cfbsRequestsModel = new CfbsRequestsModel();
    }

    /**
     * @rbacAlias build.project-modules.post
     */
    function post($request, $projectId): \Response
    {

        $response = new \Response($request);
        $response->code = \Response::CREATED;
        $requestData = \Utils::getValidJsonData($request->data);

        if (!isset($requestData->url) || empty($requestData->url)) {
            $response->code = \Response::BADREQUEST;
            $response->body = 'Url is required.';
            return $response;
        }

        if (strlen($requestData->url) > MAX_URL_LENGTH) {
            throw new InvalidArgumentException('Url is too long. Maximum allowed length is: ' . MAX_URL_LENGTH);
        }

        if (!($project = $this->projectModel->get((int)$projectId))) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'Project not found.';
            return $response;
        }

        $requestID = $this->cfbsRequestsModel->create(
            cfbsRequestsModel::ADD_MODULES_REQUEST,
            [
                'project_id' => $projectId,
                'modules' => [['url' => $requestData->url]],
                'git' => getGitDataFromRequest($requestData)
            ]
        );

        try {
            $this->cfbsRequestsModel->processRequestResponse($requestID);
            $this->projectModel->updateIsDeployedLocally((int)$projectId, false);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::PROJECT,
                AuditLogFields::OBJECT_ID => $projectId,
                AuditLogFields::OBJECT_NAME => $project['name'],
                AuditLogFields::ACTION => AuditLogActions::MODULE_ADDED,
                AuditLogFields::DETAILS => ["Added module `{$requestData->url}` to project #$projectId."]
            ]);
        } catch (InvalidCfbsRequestException $exception) {
            $response->code = \Response::UNPROCESSABLE_ENTITY;
            $response->body = $exception->getMessage();
        } catch (\Exception $exception) {
            $response->code = \Response::INTERNALSERVERERROR;
            $response->body = 'Internal sever error occurred while adding module to build project.';
            syslog(LOG_ERR, $response->body . '. Err.:' . $exception->getTraceAsString());
        }

        return $response;
    }
    
}
