#!/bin/bash
TYPE=$1
PACKAGE_RELEASE=$2
PACKAGE_BUILD=$3

# The base directory where to look for packages to install
LOCAL_PACKAGE_BASE="/vagrant/packages"

function detect_distro
{
    # If the host has /etc/release we should use that.
    if [ -e /etc/os-release ]; then
        echo "Using /etc/os-release for distro information"
        source /etc/os-release
    else
        echo "Could not find /etc/os-release, moving on"
        if [ -e /etc/centos-release ]; then
            echo "Found /etc/centos-release"
            ID="centos"
        elif [ -e /etc/redhat-release ]; then
            echo "Found /etc/redhat-release"
            ID="redhat"
        else
            echo "Unable to detect distro"
        fi
    fi
    case $ID in
        "redhat"|"rhel"|"centos")
            detect_redhat_and_centos
            ;;

        "debian")
            detect_deb_based_distros
            ;;
        *)
            echo "Unable to detect distro, will try quick install"
            ;;
    esac

    #echo ID=$ID
    #echo VERSION_ID=$VERSION_ID
}

function detect_deb_based_distros
{
    if [ -e /etc/os-release ]; then
        VERSION_ID_MAJOR=$(echo $VERSION_ID|cut -d"." -f "1")
    else
        echo "Unable to detect deb based distro details"
    fi

    echo NAME=$NAME
    echo ID=$ID
    echo VERSION_ID=$VERSION_ID
    echo VERSION_ID_MAJOR=$VERSION_ID_MAJOR
}
function detect_redhat_and_centos
{
    if [ -e /etc/os-release ]; then
        VERSION_ID_MAJOR=$(echo $VERSION_ID|cut -d"." -f "1")
    elif [ -e /etc/centos-release ]; then
        NAME="CentOS Linux"
        ID="centos"
        VERSION_ID=$(grep -P "(\d.*)\s" --only-matching /etc/centos-release)
        VERSION_ID_MAJOR=$(echo $VERSION_ID|cut -d"." -f "1")
    elif [ -e /etc/redhat-release ]; then
        NAME="RedHat Linux"
        ID="centos"
        VERSION_ID=$(grep -P "(\d.*)\s" --only-matching /etc/redhat-release)
        VERSION_ID_MAJOR=$(echo $VERSION_ID|cut -d"." -f "1")
    else
        echo "Unable to detect redhat or centos details"
    fi

    echo NAME=$NAME
    echo ID=$ID
    echo VERSION_ID=$VERSION_ID
    echo VERSION_ID_MAJOR=$VERSION_ID_MAJOR
}


function firewalld_open_http_s
{
    if systemctl is-enabled firewalld --quiet; then
        echo "Opening firewall for HTTP(s)"
        firewall-cmd --quiet --permanent --zone=public --add-service=http
        firewall-cmd --quiet --permanent --zone=public --add-service=https
        firewall-cmd --quiet --reload
    fi
}

function firewalld_open_cfengine
{
    if systemctl is-enabled firewalld --quiet; then
      echo "Opening firewall for CFEngine"
      firewall-cmd --quiet --permanent --zone=public --add-service=cfengine
      firewall-cmd --quiet --reload
    fi
}

function set_deb_package_details
{
    ARCH=$(uname -m)
    case $ARCH in
        "x86_64")
            echo "x86_64 detected on Debian, overriding ARCH as amd64."
            ARCH="amd64"
            ;;
        *)
            echo "Unsupported ARCH $ARCH detected on Debian"
            exit 1
            ;;
    esac

    case $ID in
        "debian")

            if [ "$VERSION_ID_MAJOR" -eq 12 ]; then

                if [ "$TYPE" == "agent" ]; then
                    PKG_NAME=cfengine-nova
                elif [ "$TYPE" == "hub" ]; then
                    PKG_NAME=cfengine-nova-hub
                fi

                if [ "$TYPE" == "agent" ]; then
                    # PACKAGES_x86_64_linux_debian_12/
                    PKG_DIR=PACKAGES_x86_64_linux_debian_${VERSION_ID_MAJOR}
                else
                    # PACKAGES_HUB_x86_64_linux_debian_12
                    PKG_DIR=PACKAGES_HUB_x86_64_linux_debian_${VERSION_ID_MAJOR}
                fi
            else
                echo "Only Debian 12 packages are shipped in the vagrant quickstart environment."
                echo "Error: Unable to select PKG_DIR for $ID $VERSION_ID_MAJOR $TYPE"
                exit 1
            fi
            PKG=${PKG_NAME}_${PACKAGE_RELEASE}-${PACKAGE_BUILD}.debian${VERSION_ID_MAJOR}_${ARCH}.deb
            ;;

        *)
            echo "Only Debian packages are shipped with the vagrant environment."
            exit 1
            ;;
    esac

    echo PKG_DIR="$PKG_DIR"
    echo PKG="$PKG"
}
function set_rpm_package_details
{
    ARCH=$(uname -i)

    if [ "$TYPE" == "agent" ]; then
        PKG_NAME=cfengine-nova
    elif [ "$TYPE" == "hub" ]; then
        PKG_NAME=cfengine-nova-hub
    fi

    case $ID in
        "redhat"|"rhel"|"centos")

            # Firewalld often blocks CFEngine by default in el8+
            if [ "$VERSION_ID_MAJOR" -ge 7 ]; then
                firewalld_open_cfengine
            fi

            # Firewalld often blocks http(s) by default in el8+, hubs need it
            # for Mission Portal
            if [ "$TYPE" == "hub" ] && [ "$VERSION_ID_MAJOR" -ge 7 ]; then
                firewalld_open_http_s
            fi

            if [ "$TYPE" == "agent" ]; then
                PKG_DIR=agent_rhel${VERSION_ID_MAJOR}_x86_64
            else
                PKG_DIR=redhat_${VERSION_ID_MAJOR}_x86_64
            fi

            if [ "$VERSION_ID_MAJOR" -le 6 ] && [ "$TYPE" == "hub" ]; then
                echo "Error: Unable to select PKG_DIR for $ID $VERSION_ID_MAJOR $TYPE"
                echo "Enterprise Hub is not supported on this platform"
                exit 1
            elif [ "$VERSION_ID_MAJOR" -le 5 ]; then
                echo "Sorry, we don't maintain packages for Enterprise Linux versions lower than 6."
                exit 1
            fi
            PKG=$PKG_NAME-$PACKAGE_RELEASE-$PACKAGE_BUILD.el${VERSION_ID_MAJOR}.$ARCH.rpm
            ;;
        "sles")
            echo "SLES SUPPORT NOT IMPLEMENTED"
            exit 1
            ;;
    esac

    echo PKG_DIR="$PKG_DIR"
    echo PKG="$PKG"
}

function check_pkg_presence
{
    _LOCAL_PACKAGE_BASE=$1
    _PKG_DIR=$2
    _PKG=$3
    _PACKAGE_RELEASE=$4
    _TYPE=$5

    if [ ! -e $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG ]; then
        echo "Could not find $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG"
        mkdir -p $_LOCAL_PACKAGE_BASE/$_PKG_DIR
        echo "Fetching $_PKG"
        if command -v wget; then
            echo using wget
            echo wget --quiet http://cfengine-package-repos.s3.amazonaws.com/enterprise/Enterprise-$_PACKAGE_RELEASE/$_TYPE/$_PKG_DIR/$_PKG -O $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG
            wget --quiet http://cfengine-package-repos.s3.amazonaws.com/enterprise/Enterprise-$_PACKAGE_RELEASE/$_TYPE/$_PKG_DIR/$_PKG -O $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG
            if [ "$?" != "0" ]; then
                echo "Failed to retrieve package. Try another PLATFORM and/or PACKAGE_RELEASE if you are using something other than what is included."
                exit 1
            fi
        else
            echo wget not found, falling back to curl
            echo curl -s http://cfengine-package-repos.s3.amazonaws.com/enterprise/Enterprise-$_PACKAGE_RELEASE/$_TYPE/$_PKG_DIR/$_PKG -o $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG
            curl -s http://cfengine-package-repos.s3.amazonaws.com/enterprise/Enterprise-$_PACKAGE_RELEASE/$_TYPE/$_PKG_DIR/$_PKG -o $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG
            if [ "$?" != "0" ]; then
                echo "Failed to retrieve package. Try another PLATFORM and/or PACKAGE_RELEASE if you are using something other than what is included."
                exit 1
            fi
        fi
    else
        echo "Using $_LOCAL_PACKAGE_BASE/$_PKG_DIR/$_PKG"
    fi
}

function install_rpm_pkg
{
    if ! /bin/rpm -q $PKG_NAME ; then
        echo "Installing CFEngine Enterprise $TYPE"
        yum -y install $LOCAL_PACKAGE_BASE/$PKG_DIR/$PKG
    else
        echo "$PKG_NAME already installed, skipping installation"
    fi
}

function install_deb_pkg
{
    if ! /bin/dpkg -l $PKG_NAME 2>/dev/null | grep -q '^ii'; then
        echo "Installing CFEngine Enterprise $TYPE"
        cp $LOCAL_PACKAGE_BASE/$PKG_DIR/$PKG /
        DEBIAN_FRONTEND=noninteractive apt-get -y install /$PKG
        rm -f /$PKG
    else
        echo "$PKG_NAME already installed, skipping installation"
    fi
}
function select_package
{
    case $ID in
        "redhat"|"rhel"|"centos")
            set_rpm_package_details
            check_pkg_presence $LOCAL_PACKAGE_BASE $PKG_DIR $PKG $PACKAGE_RELEASE $TYPE
            install_rpm_pkg
            ;;
        "debian")
            set_deb_package_details
            check_pkg_presence $LOCAL_PACKAGE_BASE $PKG_DIR $PKG $PACKAGE_RELEASE $TYPE
            install_deb_pkg
            ;;
        *)
            echo "Only Debian is built-in to vagrant quickstart, attempting quick install script"
            if which wget > /dev/null; then
                wget https://s3.amazonaws.com/cfengine.packages/quick-install-cfengine-enterprise.sh
            elif which curl > /dev/null; then
                curl https://s3.amazonaws.com/cfengine.packages/quick-install-cfengine-enterprise.sh -O
            else
                echo "Apparently no wget or curl present, cannot download quickinstall script"
                exit 1
            fi
            CFEngine_Enterprise_Package_Version=$PACKAGE_RELEASE bash quick-install-cfengine-enterprise.sh $TYPE
            ;;
    esac
}

detect_distro
select_package
