import React, { useEffect, useRef, useState } from 'react';
import { useSelector, useDispatch } from 'react-redux';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  IconButton,
  Box,
  Tooltip,
  DialogActions,
  Link,
} from '@mui/material';
import {
  Close as CloseIcon,
  Remove as MinimizeIcon,
  History as ClearHistoryIcon,
} from '@mui/icons-material';
import { makeStyles } from 'tss-react/mui';
import { RootState, AppDispatch, selectProfileAdditionalData, updateAdditionalData } from '../store';
import { addMessage, Message } from '../store/chatSlice';
import { getConfig, getRoles } from '../store';
import { sendMessage } from '../store/chatSlice/thunks';
import { selectConfig, selectConfigLoading } from '../store/chatSlice/selectors';
import Spinner from '../../common/Spinner';
import Messages from './Messages';
import AiAgentHeader from './AiAgentHeader';
import NotConfigured from './NotConfigured';
import InputArea from './InputArea';
import InitialScreen from './InitialScreen';

const useStyles = makeStyles()(theme => ({
  dialogContent: {
    display: 'flex',
    flexDirection: 'column',
    height: '80vh',
    '& table th, & table .td, & table td': {
      padding: '0 16px !important',
    },
    '> div': {
      width: 700,
      margin: '0 auto',
      position: 'relative',
      paddingBottom: 100,
      minHeight: '50vh'
    }
  },
  dialogActions: {
    border: 'none',
    backgroundColor: theme.palette.background.default
  },
  disclaimer: {
    fontSize: 12,
    color: theme.palette.text.secondary,
    fontWeight: 400,
    textAlign: 'center',
    letterSpacing: '0.4px',
    'a': {
      color: theme.palette.text.secondary
    }
  }
}));

const HISTORY_ITEMS_NUMBER = 8;

const getMessagesForHistory = (number: number, messages: Message[]) =>
  messages
    .slice(-HISTORY_ITEMS_NUMBER)
    .map((message: Message) => {
      const payload = message.payload.hasOwnProperty('sql')
        ? { sql: message.payload.sql, explanation: message.payload?.explanation }
        : message.payload;
      return { ...message, payload };
    });

interface AiChatProps {
  onClose: () => void;
  onMinimize: () => void;
  onHistoryClear: () => void;
}

const AiChat: React.FC<AiChatProps> = ({ onClose, onMinimize, onHistoryClear }) => {
  const { classes } = useStyles();
  const dispatch = useDispatch<AppDispatch>();
  const { messages = [], isLoading } = useSelector((state: RootState) => state.chat);
  const config = useSelector(selectConfig);
  const configLoading = useSelector(selectConfigLoading);
  const isOpenedBefore = useSelector(selectProfileAdditionalData).is_ai_agent_opened;
  const currentRequestRef = useRef<any>(null);
  const [inputValue, setInputValue] = useState('');
  const dialogContainerRef = useRef(undefined);
  const hasMessages = !!messages.length;

  useEffect(() => {
    setTimeout(() => {
      if (dialogContainerRef.current?.scrollHeight) {
        dialogContainerRef.current.scrollTop = dialogContainerRef.current?.scrollHeight;
      }
    }, 50);
  }, [messages, dialogContainerRef?.current, configLoading]);

  useEffect(() => {
    dispatch(getConfig());
  }, []);

  useEffect(() => {
    if (!isOpenedBefore) {
      dispatch(updateAdditionalData({ key: 'is_ai_agent_opened', value: true }));
    }
  }, [isOpenedBefore, updateAdditionalData, dispatch]);

  useEffect(() => {
    dispatch(getRoles());
  }, []);

  const handleSendMessage = async (message: string) => {
    if (message.trim() === '') {
      return;
    }
    dispatch(addMessage({ sender: 'user', payload: message }));
    setInputValue('');
    currentRequestRef.current = dispatch(sendMessage({
      message,
      history: getMessagesForHistory(HISTORY_ITEMS_NUMBER, messages)
    }));
  };

  const handleCancelRequest = () => {
    if (currentRequestRef.current) {
      currentRequestRef.current.abort();
      currentRequestRef.current = null;
    }
  };

  const renderDialogContent = () => {
    if (configLoading) return <Spinner />;
    if (!config) return <NotConfigured />;
    if (hasMessages) return <Messages messages={messages} isLoading={isLoading} minimize={onMinimize} />;
    return <InitialScreen
      inputValue={inputValue}
      isLoading={isLoading}
      onSendMessage={handleSendMessage}
      onCancelRequest={handleCancelRequest}
      onInputChange={setInputValue} />
  };

  return (
    <Box>
      <Dialog
        open={true}
        onClose={onClose}
        fullWidth
        maxWidth="lg"
        keepMounted
      >
        <DialogTitle
          sx={{
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'space-between'
          }}
        >
          <AiAgentHeader fontSize={21} />
          <Box>
            <IconButton onClick={onHistoryClear}>
              <Tooltip title="Clear history">
                <ClearHistoryIcon />
              </Tooltip>
            </IconButton>
            <IconButton onClick={onMinimize}>
              <Tooltip title="Minimize">
                <MinimizeIcon />
              </Tooltip>
            </IconButton>
            <IconButton aria-label="close" onClick={onClose}>
              <Tooltip title="Close">
                <CloseIcon />
              </Tooltip>
            </IconButton>
          </Box>
        </DialogTitle>
        <DialogContent ref={dialogContainerRef} className={classes.dialogContent}>
          <Box>
            {renderDialogContent()}
          </Box>
        </DialogContent>
        {hasMessages && 
        <DialogActions className={classes.dialogActions}>
          <Box sx={{ margin: '0 auto' }} >
            <InputArea
              inputValue={inputValue}
              isLoading={isLoading}
              onSendMessage={handleSendMessage}
              onCancelRequest={handleCancelRequest}
              onInputChange={setInputValue}
            />
            <Box className={classes.disclaimer}>
              AI can make mistakes, so don't forget to double-check the responses.&nbsp;
              <Link href="/settings/ai">AI settings</Link>
            </Box>
          </Box>
        </DialogActions>}
      </Dialog>
    </Box>
  );
};

export default AiChat;
