<?php


namespace Build\Models;

use Build\Entities\ProjectEntity;

class ProjectModel
{
    private \PDO $dbConnection;
    private $table = 'build_projects';
    private $username;

    public function __construct(string $username)
    {
        $this->username = $username;
        $this->dbConnection = \CfSettings::getInstance()->getConnection();
    }


    public function create(ProjectEntity $entity): string
    {
        $sql = "INSERT INTO {$this->table} 
                (repository_url, branch, name, authentication_type, username, password, ssh_private_key, is_local, is_deployed_locally, ssh_key_id) 
                VALUES (:repository_url, :branch, :name, :authentication_type, :username, :password, :ssh_private_key, :is_local, :is_deployed_locally, :ssh_key_id)";

        $this->dbConnection->prepare($sql)->execute($entity->toArray());
        return $this->dbConnection->lastInsertId();
    }

    public function update(int $id, ProjectEntity $entity)
    {
        $set = [];
        $data = $entity->changedItemsToArray();

        if (sizeof($data) == 0) {
            return;
        }

        foreach ($data as $key => $value) {
            $set[] = "$key = :$key";
        }

        $sql = "UPDATE {$this->table} SET ". implode(', ', $set) ." WHERE id = :id";
        $this->dbConnection->prepare($sql)->execute(array_merge($data, ['id' => $id]));
    }

    public function get(int $id)
    {
        $stmt = $this->dbConnection->prepare(
            "SELECT id, repository_url, branch, name, authentication_type, username, is_local, is_deployed_locally, created_at, pushed_at, ssh_key_id, action,
             (CASE WHEN password IS NULL THEN 'not set' ELSE 'set' END) AS password,
             (CASE WHEN ssh_private_key IS NULL THEN 'not set' ELSE 'set' END) AS ssh_private_key 
             FROM {$this->table}
             WHERE id = :id"
        );
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }

    public function getWithSensitiveData(int $id)
    {
        $stmt = $this->dbConnection->prepare(
            "SELECT {$this->table}.id, repository_url, branch, name, authentication_type, username, is_local, is_deployed_locally, created_at, pushed_at, password,  action,
             COALESCE(ssh_keys.private_key, build_projects.ssh_private_key) AS ssh_private_key, ssh_key_id
             FROM {$this->table} LEFT JOIN ssh_keys ON ssh_keys.id = ssh_key_id
             WHERE {$this->table}.id = :id"
        );

        $stmt->execute(['id' => $id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC);
    }

    public function list($limit = 10, $offset = 0, $filter = [])
    {
        $where = sizeof($filter) > 0 ? ' WHERE ' . implode(' AND ', $filter) : '';
        $stmt = $this->dbConnection->prepare(
            "SELECT id, repository_url, branch, name, authentication_type, username, is_local, is_deployed_locally, created_at, pushed_at, ssh_key_id, action,
             (CASE WHEN password IS NULL THEN 'not set' ELSE 'set' END) AS password,
             (CASE WHEN ssh_private_key IS NULL THEN 'not set' ELSE 'set' END) AS ssh_private_key 
             FROM {$this->table} 
             $where
             ORDER BY id
             LIMIT :limit OFFSET :offset"
        );
        $stmt->execute(['limit' => $limit, 'offset' => $offset]);

        $projects = $stmt->fetchAll(\PDO::FETCH_ASSOC);

        return [
            'data' => $projects,
            'meta' => [
                'count' => sizeof($projects),
                'page' => $offset + 1,
                'timestamp' => time(),
                'total' => $this->getTotalProjectsCount()
            ]
        ];
    }

    public function getTotalProjectsCount()
    {
        $data = $this->dbConnection->query("SELECT count(*) as count FROM {$this->table}")->fetch(\PDO::FETCH_ASSOC);;
        return $data['count'];
    }

    public function delete(int $id)
    {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $this->dbConnection->prepare($sql)->execute(['id' => $id]);
    }

    public function updatePushedAt(int $id)
    {
        $sql = "UPDATE {$this->table} SET pushed_at = now() WHERE id = :id";
        $this->dbConnection->prepare($sql)->execute(['id' => $id]);
    }

    public function updateIsDeployedLocally(int $id, bool $isDeployedLocally): void
    {
        $sql = "UPDATE {$this->table} SET is_deployed_locally = :is_deployed_locally WHERE id = :id";
        $this->dbConnection->prepare($sql)->execute(['id' => $id, 'is_deployed_locally' => intval($isDeployedLocally)]);
    }

    public function getNextAvailableLocalProjectName($defaultName = 'Local project'): string
    {
        $stmt = $this->dbConnection->prepare("SELECT name FROM {$this->table} WHERE name ~ :nameregex ORDER BY name DESC LIMIT 1");
        $stmt->execute(['nameregex' => "$defaultName($|.*\d+)"]);
        $project = $stmt->fetch(\PDO::FETCH_ASSOC);

        return empty($project) ?
            $defaultName :
            "$defaultName " . (intval(preg_replace('/[^0-9]/', '', $project['name'])) + 1);
    }

    public function setDefaultAction(int $id, string $action): void
    {
        $sql = "UPDATE {$this->table} SET action = :action WHERE id = :id";
        $this->dbConnection->prepare($sql)->execute(['id' => $id, 'action' => $action]);
    }
}
