<?php

class SharedGroupCmdbItemEntity
{
    private $groupId;
    private $classesValue;
    private $variablesValue;

    public static function fromRequest($obj): SharedGroupCmdbItemEntity
    {
        $item = new SharedGroupCmdbItemEntity();

        if (property_exists($obj, 'groupId') && is_string($obj->groupId)) {
            $item->setGroupId($obj->groupId);
        } else {
            throw new \InvalidArgumentException('Group Id is missing');
        }

        if (property_exists($obj, 'classes') && is_object($obj->classes)) {
            $item->setClassesValue($obj->classes);
        }

        if (property_exists($obj, 'variables') && is_object($obj->variables)) {
            $item->setVariablesValue($obj->variables);
        }

        return $item;
    }

    public static function fromDB(array $data): SharedGroupCmdbItemEntity
    {
        $item = new SharedGroupCmdbItemEntity();
        $item->setGroupId($data['group_id']);

        if (isset($data['value'])) {
            $value = $data['value'];

            if (property_exists($value, 'classes') && is_array($value->classes)) {
                $item->setClassesValue($value->classes);
            }

            if (property_exists($value, 'variables') && is_array($value->variables)) {
                $item->setVariablesValue($value->variables);
            }
        }

        return $item;
    }

    public static function setOldEntityValuesFromNew(SharedGroupCmdbItemEntity $new, SharedGroupCmdbItemEntity $old): SharedGroupCmdbItemEntity
    {
        $entity = new SharedGroupCmdbItemEntity();
        $entity->setClassesValue(($new->getClassesValue() !== null) ? $new->getClassesValue() : $old->getClassesValue());
        $entity->setVariablesValue(($new->getVariablesValue() !== null) ? $new->getVariablesValue() : $old->getVariablesValue());
        $entity->setGroupId($new->getGroupId());
        return $entity;
    }

    public function getClassesValue()
    {
        return $this->classesValue;
    }

    public function setClassesValue($classesValue)
    {
        $this->validateClasses($classesValue);
        $this->classesValue = $classesValue;
    }

    public function getVariablesValue()
    {
        return $this->variablesValue;
    }

    public function setVariablesValue($variablesValue)
    {
        $this->validateVariables($variablesValue);
        $this->variablesValue = $variablesValue;
    }

    /**
     * @return string
     */
    public function getGroupId(): string
    {
        return $this->groupId;
    }

    /**
     * @param string $groupId
     */
    public function setGroupId(string $groupId)
    {
        $this->groupId = $groupId;
    }

    public function valuesToJson()
    {
        return json_encode(
            [
                'classes' => $this->getClassesValue(),
                'variables' => $this->getVariablesValue(),
            ]
        );
    }

    private function validateClasses($classes)
    {
        foreach ($classes as $name => $value) {
            if (!is_object($value)) {
                throw new \InvalidArgumentException(
                    "Wrong `$name` class value: " . var_export($value, $return = true) . '. Objects are only allowed'
                );
            }
        }
    }

    private function validateVariables($variables)
    {
        foreach ($variables as $name => $value) {
            if (!is_object($value)) {
                throw new \InvalidArgumentException(
                    "Wrong `$name` variable value: " . var_export($value, $return = true) .
                    '. Objects are only allowed'
                );
            } elseif (!property_exists($value, 'value')) {
                throw new \InvalidArgumentException(
                    "Variable `$name` should contain `value` property : " . var_export($value, $return = true)
                );
            }
        }
    }
}
