<?php
/**
 * some report properties
 *
 * readonly  -only read access, but can schedule
 * is_public - everyone can see
 *
 *
 */
class advancedreports_model extends Cf_Model
{
    const INVENTORY_SEARCH_QUERY_ATTRIBUTE = 'cfHostSearchQuery';

    var $changeUrl = 'v2/changes/policy';
    var $countUrl = 'v2/changes/policy/count';
    var $collectionName = 'custom_search';
    var $scheduleTableName = 'report_schedule';
    var $reportTableName = 'report';
    var $errors;
        // DistinctVariables - all inventory variables
        // UnnestData - unnest meta array
        // final query - remove inventory records - because unnest produce one record which has inventory as metatag
    var $inventory_variables_query = "SELECT DISTINCT metaParsed, keyname, type
                                      FROM (SELECT keyname,type, unnest(metatags) AS metaParsed
                                                 FROM (
                                                         SELECT DISTINCT keyname, type, metatags
                                                         FROM inventory
                                                         WHERE NOT ('attribute_name=none' = ANY (metatags))
                                                       ) as uniqueAttr
                                           ) as InventoryVariablesList
                                           WHERE  metaParsed != 'inventory'";
    private $inventoryLib = null;
    
    public function __construct() {
        parent::__construct();

        $inventoryBridgePath = FCPATH . '/api/modules/inventory/autoload-no-controllers.php';
        if (
            file_exists($inventoryBridgePath) &&
            (include_once $inventoryBridgePath) &&
            class_exists('InventoryLib')
        ) {
            $this->inventoryLib = new InventoryLib(new VariablesDictionaryModel());
        }
    }

    /**
     * This function will return json as result of query
-    * query example
-    * $SQLString="SELECT Software.Name,Contexts.Name FROM Contexts INNER JOIN Software ON Software.HostKey = Contexts.HostKey LIMIT 10";
-    *
     *
     * @param <string> $username
     * @param <string> $SQLString
     * @param <string> $sortColumn
     * @param <string> $sortDescending
     * @param <string> $skip   if skip  = 0 - do not skip enything
     * @param <string> $limit  if limit = 0 - return all
     * @param <array>  $includes
     * @param <array>  $excludes
     * @param <array>  $inventoryFilter
     * @param string  $url
     * @return <json>
     * @throws Exception
     */
    /* NOTE:  this function is also used in console report generation. */
    function runQuery(
        string $username,
        string $SQLString,
        string $sortColumn = '',
        string $sortDescending = '',
        int $skip = 0,
        int $limit = 0,
        array $includes = array(),
        array $excludes = array(),
        bool|array $inventoryFilter = [],
        string $url = '/query',
        bool $fetchDataViaAPI = true
    )
    {
        try
        {
            if (ENVIRONMENT == 'development')
            {
                $startQueryTime = microtime(true);
            }

            $SQLString = str_replace('\n', ' ', trim($SQLString));

            $jsonPayload = array(
                'query'          => $SQLString,
                'sortColumn'     => $sortColumn,
                'sortDescending' => $sortDescending,
                'skip'           => $skip,
                'limit'          => $limit,
                'hostContextInclude' => $includes,
                'hostContextExclude' => $excludes,
                'filter' => $inventoryFilter,
            );
            
            $isInvFilterApplied = is_array($jsonPayload['filter']) && sizeof($jsonPayload['filter']) > 0;
            if (CLI === FALSE && $fetchDataViaAPI)
            {
                $jsonBody = $isInvFilterApplied ?
                    $this->getRestClient()->post('/inventory-filtered-query/', $jsonPayload) :
                    $this->getRestClient()->post($url, $jsonPayload);
            }
            else
            {
                $params = (object) $jsonPayload;
                if ($isInvFilterApplied && $this->inventoryLib) {
                    $this->inventoryLib->extendSQLWithInventoryCTE($params);
                }
                $jsonBody = cfapi_query_post($username, $params->query, $sortColumn, $sortDescending, $skip, $limit, $includes, $excludes);
            }
            
            $jsonObj = $this->checkdata($jsonBody);


            // log slow queries in development mode
            if (ENVIRONMENT == 'development')
            {
                $this->_logQueryTime($startQueryTime, $SQLString, $limit, $includes, $excludes);
            }

            // ignore meta and other values for now
            return $jsonObj['data'][0];
        }
        catch (Exception $e)
        {
            throw $e;
        }
    }

    function saveScheduledReport_REST($data)
    {
        if (is_array($data) && !empty($data))
        {
            try
            {
                $this->getRestClient()->put('/user/' . $data['userId'] . '/subscription/query/' . $data['id'], $data);
            }
            catch (Exception $e)
            {
                log_message(log_level_for_exception($e), $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
                throw $e;
            }
        }
    }

    /**
     *
     */

    function runChangeQuery($data)
    {
        try
        {
            $contextParameters['include'] = is_array($data['includes']) ? $data['includes'] : [];
            $contextParameters['exclude'] = is_array($data['excludes']) ? $data['excludes'] : [];

            $url = sprintf("/%s?%s&from=%d&to=%d&%s&sort=%s&count=%d&page=%d",$this->changeUrl,$data['cleanFilters'],$data['from'],$data['to'], http_build_query($contextParameters), $data['sort'],$data['count'],$data['page']);
            $result = $this->getRestClient()->get($url);
            $changeResult = json_decode($result, true);
            return $changeResult;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), $e->getMessage() . " File: " . $e->getFile() . ' line:' . $e->getLine());
            throw $e;
        }
    }


    function runChangeCount($data)
    {
        if(!isset($data['changeDaysNumber']) && empty($data['changeDaysNumber'])){
            log_message('error', "runChangeCount() : changeDaysNumber is empty.");
            throw new Exception("runChangeCount() : changeDaysNumber is empty.");
        }

        if(isset($data['timezone'])){
            date_default_timezone_set($data['timezone']);
        }

        try
        {
            $changeResult = [];
            for ($i = $data['changeDaysNumber']; $i >= 0; $i--){
                $unixTimeFrom = mktime(0, 0, 0, date("m"), date("d")-$i, date("Y"));
                $unixTimeTo = $i==0 ? time() : mktime(23, 59, 59, date("m"), date("d")-$i, date("Y"));
                $url = sprintf("/%s?from=%d&to=%d",$this->countUrl, $unixTimeFrom, $unixTimeTo);
                $result = $this->getRestClient()->get($url);
                $changeResult[]  = array_merge(json_decode($result, true), ['date' => date("D F d",$unixTimeFrom)]);
            }

            return $changeResult;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), $e->getMessage() . " File: " . $e->getFile() . ' line:' . $e->getLine());
            throw $e;
        }
    }


    function getHostCount($data)
    {
        try
        {
            $result = $this->getRestClient()->post('/host-count',  $data);
            return   $result;
        }
        catch(Exception $e)
        {
            log_message(log_level_for_exception($e), $e->getMessage() . " File: " . $e->getFile() . ' line:' . $e->getLine());
            throw $e;
        }
    }

    function deleteScheduledReport_REST($data)
    {
        if (is_array($data) && !empty($data))
        {
            try
            {
                $this->getRestClient()->delete('/user/' . $data['userId'] . '/subscription/query/' . $data['id']);
            }
            catch (Exception $e)
            {
                log_message(log_level_for_exception($e), $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
                throw $e;
            }
        }
    }


    function saveScheduleForReport($reportId,  $scheduleData, $overwrite=false)
    {
        try
        {
            # For inserting in postgresql
            $scheduleDataToInsert = $this->_map_for_insert_schedule($reportId, $scheduleData);

            if ($overwrite === true)
            {
                return $this->db->where(array('reportid' => $reportId, 'id' => $scheduleDataToInsert['id']))->update($this->scheduleTableName, $scheduleDataToInsert);
            }
            else
            {
               return $this->db->insert($this->scheduleTableName, $scheduleDataToInsert);
            }

        }
        catch (Exception $e)
        {
            log_message(log_level_for_exception($e), $e->getMessage() . " " . $e->getFile() . " line:" . $e->getLine());
            throw $e;
        }
    }


    /**
     * Prepare data before insert into report_schedule table
     *
     * @param <int> $reportId
     * @param <array> $scheduleData
     * @return <array>
     */

    private function _map_for_insert_schedule($reportId, $scheduleData)
    {
        $mapped_data = array();

        $mapped_data['reportid']    = (int)$reportId;

        $mapped_data['id']          = $scheduleData['id'];
        $mapped_data['userid']      = $scheduleData['userId'];
        $mapped_data['title']       = isset($scheduleData['title'])          ? $scheduleData['title'] : '';
        $mapped_data['description'] = isset($scheduleData['description'])    ? $scheduleData['description'] : '';
        $mapped_data['emailfrom']   = isset($scheduleData['emailfrom'])      ? $scheduleData['emailfrom'] : '';
        $mapped_data['emailto']     = isset($scheduleData['emailto'])        ? $scheduleData['emailto'] : '';
        $mapped_data['enabled']     = isset($scheduleData['enabled'])        ? $scheduleData['enabled'] : '';
        $mapped_data['query']       = isset($scheduleData['query'])          ? $scheduleData['query'] : '';
        $mapped_data['skipMailing']       = isset($scheduleData['skipMailing'])          ? 'true' : 'false';

        $mapped_data['outputtypes'] = PhpArrayToPgsql($scheduleData['outputTypes']);

        $mapped_data['schedule']                  = isset($scheduleData['schedule'])                   ? $scheduleData['schedule'] : '';
        $mapped_data['schedulehumanreadabletime'] = isset($scheduleData['scheduleHumanReadableTime'])  ? $scheduleData['scheduleHumanReadableTime'] : '';
        $mapped_data['schedulename']              = isset($scheduleData['scheduleName'])               ? $scheduleData['scheduleName'] : '';
        $mapped_data['site_url']                  = isset($scheduleData['site_url'])                   ? $scheduleData['site_url'] : '';
        $mapped_data['hostcontextsprofileid']     = isset($scheduleData['hostcontextsprofileid'])      ? $scheduleData['hostcontextsprofileid'] : '';
        // Note: hostcontextspath already json so do not transform
        $mapped_data['hostcontextspath']          = isset($scheduleData['hostcontextspath'])           ? $scheduleData['hostcontextspath'] : '';
        $mapped_data['hostcontexts']              = json_encode($scheduleData['hostcontexts']);
        $mapped_data['scheduleData']              = json_encode($scheduleData['scheduleData']);
        $mapped_data['excludedhosts']             = !empty($scheduleData['excludedHosts']) ? json_encode($scheduleData['excludedHosts']) : '{}';

        return $mapped_data;
    }



    /**
     * Query REST API and receive an array with [data] contain 2 fields
     *
     *   id    - job id, it will aslo create csv file with the same name inside api/static folder
     *   query - SQL string
     *
     * @param type $username
     * @param type $SQLString
     * @return <string> json
     * @throws Exception
     */
    function createReportJob($username, $SQLString, $params)
    {
        try
        {
            $SQLString = str_replace('\n', ' ', trim($SQLString));
            if(isset($params['sqlSortColumn']) && $params['sqlSortColumn']){
                $SQLString = $this->processSorting($params, $SQLString);
            }

            $jsonPayload = array(
                'query'               => $SQLString,
                'outputType'          => 'csv',
                'hostContextInclude'  => (isset($params['includes']) ? $params['includes'] : array()),
                'hostContextExclude'  => (isset($params['excludes']) ? $params['excludes'] : array())
                );

            $jsonBody = $this->getRestClient()->post('/query/async/', $jsonPayload);
            $jsonObj = $this->checkdata($jsonBody);

            return $jsonObj['data'][0];
        }
        catch (Exception $e)
        {
	        log_message(log_level_for_exception($e), $e->getMessage());
            throw $e;
        }
    }

    /**
     * Process order by column for sql string
     * @param array $params
     * @param string $SQLString
     * @return string
     */
    private function processSorting(array $params, $SQLString){
        $orderDirection = (boolean)$params['sqlSortDescending']? 'DESC' : 'ASC';
        $oderString = 'ORDER BY "'.$params['sqlSortColumn'].'" '. $orderDirection;
        $SQLString =  sprintf('WITH user_query AS (%s) SELECT * FROM user_query %s', $SQLString, $oderString);

        return $SQLString;
    }

    function getReportJobStatus($username, $jobId)
    {
        try
        {
            $jsonBody = $this->getRestClient()->get('/query/async/' . $jobId);
            $jsonObj = $this->checkdata($jsonBody);
            return $jsonObj['data'][0];
        }
        catch (Exception $e)
        {
            throw $e;
        }
    }

    function cancelReportJob($username, $jobId)
    {
        try
        {

            $jsonBody = $this->getRestClient()->delete('/query/async/' . $jobId);
            return true; // ???
            /*
             * $jsonObj = $this->checkdata($jsonBody);
              return $jsonObj['data'][0]; */
        }
        catch (Exception $e)
        {
            throw $e;
        }
    }

    /**
     *  return false if no schedule with this name found
     *
     *  return scheduleId if it is found
     */

    function checkIfScheduleExist($reportId, $userId, $scheduleName)
    {
        $filter = array('reportid' => $reportId, 'userid' => $userId,  'schedulename' => $scheduleName);

        $result = $this->db->where($filter)->get($this->scheduleTableName);


        if ($result->num_rows() > 0 )
        {
            return $result->first_row('array');
        }

        return false;
    }


    function deleteSchedule($reportId, $scheduleId)
    {
      $filter = array('id' => $scheduleId,'reportid'=>$reportId);

      $this->db->where($filter)->delete($this->scheduleTableName);
    }

    /**
     * Return autocomplete list
     *
     *
     * @param type $username
     * @param type $fieldName   - field name which will be used in SELECT and WHERE. NOTE: fielname should be exactly as it is in DB  (postgree do not support aliases in WHERE)
     * @param type $value       - string to search
     * @param type $parentType  - if autocomplete depends on 2 fields - this is parent field name
     * @param type $parentValue - if autocomplete depends on 2 fields - this is paret field value
     * @return type
     */

    function getAutocompleteFieldData($username, $fieldName, $value, $parentFieldName, $parentValue)
    {
        // TODO
        // Figure out if we are going to support include/exclude and promise filters during autocomplete

        $SQL = '';
        $skip= 0;
        $rows= 15;
        $includes = array();
        $excludes = array();

        $allowedFields = [
            'bundlename', 'promiser', 'promisehandle', 'promisees',
            'patchname', 'patchversion', 'softwarename', 'softwarearchitecture'
        ];

        if (!in_array($fieldName, $allowedFields)) {
            throw new Exception("Invalid field name for autocomplete: " . $fieldName);
        }

        $allowedParentFields = [
            'bundlename', 'promiser', 'promisehandle', 'promisees',
            'patchname', 'patchversion', 'softwarename', 'softwarearchitecture',
            'hostkey', 'hostname'
        ];

        if (!empty($parentFieldName) && !in_array($parentFieldName, $allowedParentFields)) {
            throw new Exception("Invalid parent field name for autocomplete: " . $parentFieldName);
        }

        $WHERE = '';

        if (!empty($value))
        {
            $escapedValue = $this->db->escape_str($value);
            $WHERE = "WHERE ". $fieldName ." ILIKE '%". $escapedValue ."%'";
        }


        if (!empty($parentValue)) {
            if (empty($WHERE))
            {
                $WHERE = "WHERE ";
            }
            else {
                $WHERE .= " AND ";
            }

            $escapedParentValue = $this->db->escape_str($parentValue);
            $WHERE .= $parentFieldName. " = '". $escapedParentValue."'";
        }

        switch($fieldName)
        {
            // all this from promiseexecutions table
            case "bundlename"   :
            case "promiser"     :
            case "promisehandle":  $SQL = 'SELECT DISTINCT '.$fieldName.' AS "autocompleteFieldData" FROM promiseexecutions '. $WHERE .' LIMIT '. $rows.''; break;

            case "promisees"    :  $SQL = 'SELECT promisees
                                                FROM
                                                    (SELECT DISTINCT unnest(promisees) as promisees
                                                            FROM promiseexecutions) X ' .
                                                $WHERE .' LIMIT '. $rows.''; break;


//          case "hosts"        :  $SQL = 'SELECT Hosts.HostName AS "autocompleteFieldData" FROM Hosts '. $WHERE .' LIMIT '. $rows.''; break;
//          case "class"        :  $SQL = 'SELECT DISTINCT Contexts.ContextName AS "autocompleteFieldData" FROM Contexts '. $WHERE .' LIMIT '. $rows.'' ; break;

            // SoftwareUpdates table
            // use software table for autocomplete
            // patchversion should be prepopulated

            case "patchname"    :
            case "patchversion" :  $SQL = 'SELECT DISTINCT '. $fieldName .' AS "autocompleteFieldData" FROM SoftwareUpdates '. $WHERE .' LIMIT '. $rows.'' ; break;



            case "softwarename":
            case "softwarearchitecture": $SQL = 'SELECT DISTINCT '. $fieldName .' AS "autocompleteFieldData" FROM Software '. $WHERE .' LIMIT '. $rows.'' ; break;
        }
        if ($SQL == '') {
            throw new Exception("getAutocompleteFieldData(): Empty SQL string for autocomplete. Field: ".$fieldName);
        }

        try
        {
            $data = $this->runQuery($username, $SQL, '', '', $skip, $rows, $includes, $excludes);
        }
        catch (Exception $e)
        {
            throw $e;
        }
        if (!empty($data['rows'])) {
            $result = call_user_func_array('array_merge',  $data['rows']);
        }
        else
        {
            $result = array();
        }

        return $result;
    }

    function getInventoryFields ($username)
    {
        $SQL = $this->inventory_variables_query;
        $data = $this->runQuery($username, $SQL);



        if (empty($data['rows'])) {
            $this->setError($this->lang->line('inventory_is_empty'));
            return;
        }

        $tmpItems = array();

        // get unique items by keyname, make sure that we don't have duplicates because of source= or any other unnested metas
        foreach ($data['rows'] as $item => $value)
        {
            $attribute_name = $value[0];
            $keyname        = $value[1];
            $type           = $value[2];


            if (!isset($tmpItems[$keyname]['attribute_name']))
            {
                if (strpos($attribute_name, 'attribute_name=') !== false)
                {
                    $tmpItems[$keyname]['attribute_name'] = str_replace("'","''",$attribute_name);
                    $tmpItems[$keyname]['label'] = substr($attribute_name, 15);
                }
                else
                {
                    $tmpItems[$keyname]['attribute_name'] = $keyname;
                    $tmpItems[$keyname]['label'] = $keyname; // default value
                }
            }
            else
            {   //replace attribute and lavel only if we found attribute_name
                if (strpos($value[0], 'attribute_name=') !== false)
                {
                    $tmpItems[$keyname]['attribute_name'] = str_replace("'","''",$attribute_name);
                    $tmpItems[$keyname]['label'] = substr($attribute_name, 15);
                }
            }


            $tmpItems[$keyname]['type'] = $type;
            $tmpItems[$keyname]['keyname'] = $keyname;
        }


        $inventoryItems = array();
        // revert array, we need attribute_name as key field
        foreach ($tmpItems as $item => $value)
        {
            $inventoryItems[$value['attribute_name']] = $value;
        }


        // merge result with user defined variables types from variables_dictionary
        $this->load->model("variables_dictionary_model");

        $dictionary = $this->variables_dictionary_model->getVariables();

        if (!empty($dictionary))
        {
            foreach ($dictionary as $item => $variable)
            {
                if (isset($inventoryItems[$variable['attribute_name']]))
                {
                    $inventoryItems[$variable['attribute_name']]['type'] = $variable['type'];

                    if (isset($variable['category']))
                    {
                        $inventoryItems[$variable['attribute_name']]['category'] = $variable['category'];
                    }

                    if (isset($variable['convert_function']))
                    {
                        $inventoryItems[$variable['attribute_name']]['convert_function'] = $variable['convert_function'];
                    }

                    if (isset($variable['readonly']))
                    {
                        $inventoryItems[$variable['attribute_name']]['readonly'] = $variable['readonly'];
                    }
                    else
                    {
                        $inventoryItems[$variable['attribute_name']]['readonly'] = "0";
                    }
                }
            }
        }

        return $inventoryItems;
    }


    private function _logQueryTime($startQueryTime, $SQLString, $limit=0, $includes=array(), $excludes=array())
    {
        $endQueryTime = microtime(TRUE);
        $totalQueryTime = $endQueryTime - $startQueryTime;

        if ($totalQueryTime >= $this->config->item('API_QUERY_MAX_EXECUTION_TIME'))
        {

            $filepath = ($this->config->item('log_path') != '') ? $this->config->item('log_path') : APPPATH . 'logs/';
            $filepath .= 'query-log-' . date('Y-m-d') . '.php';

            $message = '';

            if (!file_exists($filepath))
            {
                $message .= "<" . "?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed'); ?" . ">\n\n";
            }



            $message .= "\n---- " . date("Y-m-d H:i:s") . " ----";
            $message .= "\nApplication: " . getCurrentApplication();
            $message .= "\nQuery time: " . $totalQueryTime;
            $message .= "\n" . "Query: \n";
            $message .= $SQLString;
            $message .= "\nIncludes: " . json_encode($includes);
            $message .= "\nExcludes: " . json_encode($excludes);
            $message .= "\nLimit: " . $limit;
            $message .= "\n :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::: \n\n";

            if (!$fp = @fopen($filepath, FOPEN_WRITE_CREATE))
            {
                $msg = "Unable to create query log.";
                throw new Exception($msg);
            }

            flock($fp, LOCK_EX);
            fwrite($fp, $message);
            flock($fp, LOCK_UN);
            fclose($fp);

            @chmod($filepath, FILE_WRITE_MODE);
        }
    }

    public function getBootstrappedHostReportId(){
        return $this->db->select('id')->where(['label' => 'Hosts were bootstrapped to the hub'])->get($this->reportTableName)->first_row('array');
    }

    public function getNotKeptReportId()
    {
        return $this->db->select('id')->where(['label' => 'Promises not kept (failed) that have not been kept or repaired'])->get($this->reportTableName)->first_row('array');
    }


    public function getClassesByHostkey($username, $hostKey, $filter = [])
    {
        try {
            $escapedHostKey = $this->db->escape_str($hostKey);
            $sql = "SELECT contextname as \"Class name\", metatags as \"Meta tags\" FROM contexts WHERE hostkey LIKE '$escapedHostKey'";
            if (sizeof($filter) > 0) {
                $escapedFilter = array_map([$this->db, 'escape_str'], $filter);
                $sql .= " AND contextname IN ('" . implode("','", $escapedFilter) . "')";
            }
            $sql .= " ORDER BY contextname";
            return $this->runQuery($username, $sql, '', '', 0, -1);
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), $e->getMessage());
            return null;
        }
    }

    public function getVariablesByHostkey($username, $hostKey, $filter = [])
    {
        try {
            $escapedHostKey = $this->db->escape_str($hostKey);
            $sql = "SELECT comp as \"Namespace.Bundle.Variable\", variablevalue as \"Value\" FROM variables WHERE hostkey LIKE '$escapedHostKey'";
            if (sizeof($filter) > 0) {
                $escapedFilter = array_map([$this->db, 'escape_str'], $filter);
                $sql .= " AND comp IN ('" . implode("','", $escapedFilter) . "')";
            }
            $sql .= " ORDER BY comp";
            return $this->runQuery($username, $sql, '', '', 0, -1);
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), $e->getMessage());
            return null;
        }
    }

    public function getInventoriesByHostkey($username, $hostKey)
    {
        try {
            $escapedHostKey = $this->db->escape_str($hostKey);
            $sql = 'SELECT key AS "Attribute name", value AS "Value" FROM json_each_text(
                       (SELECT values::json FROM "inventory_new" WHERE hostkey LIKE \'' . $escapedHostKey . '\' LIMIT 1)
                    ) ORDER BY key';

            return $this->runQuery($username, $sql);
        } catch (Exception $e) {
            log_message(log_level_for_exception($e), $e->getMessage());
            return null;
        }
    }
}
