<?php

namespace CMDB\Entities;

class CmdbItemEntity
{
    private $hostKey;
    private $classesValue;
    private $variablesValue;

    public static function fromRequest($obj): CmdbItemEntity
    {
        $item = new CmdbItemEntity();

        if (property_exists($obj, 'hostkey') && is_string($obj->hostkey)) {
            $item->setHostKey($obj->hostkey);
        } else {
            throw new \InvalidArgumentException('Hostkey is missing');
        }

        if (property_exists($obj, 'classes') && is_object($obj->classes)) {
            $item->setClassesValue($obj->classes);
        }

        if (property_exists($obj, 'variables') && is_object($obj->variables)) {
            $item->setVariablesValue($obj->variables);
        }

        return $item;
    }

    public static function fromDB(array $data): CmdbItemEntity
    {
        $item = new CmdbItemEntity();

        if (isset($data['hostkey']) && is_string($data['hostkey'])) {
            $item->setHostKey($data['hostkey']);
        } else {
            throw new \InvalidArgumentException('Hostkey is missing');
        }

        if (isset($data['value'])) {
            $value = $data['value'];

            if (property_exists($value, 'classes') && is_array($value->classes)) {
                $item->setClassesValue($value->classes);
            }

            if (property_exists($value, 'variables') && is_array($value->variables)) {
                $item->setVariablesValue($value->variables);
            }
        }

        return $item;
    }

    public static function merge(CmdbItemEntity $new, CmdbItemEntity $old): CmdbItemEntity
    {
        $merged = new CmdbItemEntity();
        $merged->setClassesValue(($new->getClassesValue() !== null) ? $new->getClassesValue() : $old->getClassesValue());
        $merged->setVariablesValue(($new->getVariablesValue() !== null) ? $new->getVariablesValue() : $old->getVariablesValue());
        $merged->setHostKey($new->getHostKey());
        return $merged;
    }

    public function getClassesValue()
    {
        return $this->classesValue;
    }

    public function setClassesValue($classesValue)
    {
        $this->validateClasses($classesValue);
        $this->classesValue = $classesValue;
    }

    public function getVariablesValue()
    {
        return $this->variablesValue;
    }

    public function setVariablesValue($variablesValue)
    {
        $this->validateVariables($variablesValue);
        $this->variablesValue = $variablesValue;
    }

    /**
     * @return string
     */
    public function getHostKey(): string
    {
        return $this->hostKey;
    }

    /**
     * @param string $hostKey
     */
    public function setHostKey(string $hostKey)
    {
        $this->hostKey = $hostKey;
    }

    public function valuesToJson()
    {
        return json_encode(
            [
                'classes' => $this->getClassesValue(),
                'variables' => $this->getVariablesValue(),
            ]
        );
    }

    private function validateClasses($classes)
    {
        foreach ($classes as $name => $value) {
            if (!is_object($value)) {
                throw new \InvalidArgumentException(
                    "Wrong `$name` class value: " . var_export($value, $return = true) .
                    '. Objects are only allowed'
                );
            }
        }
    }

    private function validateVariables($variables)
    {
        foreach ($variables as $name => $value) {
            if (!is_object($value)) {
                throw new \InvalidArgumentException(
                    "Wrong `$name` variable value: " . var_export($value, $return = true) .
                    '. Objects are only allowed'
                );
            } elseif (!property_exists($value, 'value')) {
                throw new \InvalidArgumentException(
                    "Variable `$name` should contain `value` property : " . var_export($value, $return = true)
                );
            }
        }
    }

}
