<?php

use CMDB\V2\CMDBResource;
use CMDB\V2\Validators\CmdbRequestValidator;

trait SharedGroupVerification
{
    public function isSharedGroupExists(int $id): bool
    {
        return boolval($this->sharedHostGroupsModel->get(intval($id)));
    }
}

/**
 * @uri /host-groups/v2/shared/:id/cmdb 0
 */
class SharedGroupCMDBItemV2 extends CMDBResource
{
    use SharedGroupVerification;
    /**
     * @var SharedGroupCmdbModel
     */
    private $cmdbItemModel;
    private $sharedHostGroupsModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new SharedGroupCmdbModel($this->username);

        $this->sharedHostGroupsModel = new SharedHostGroupsModel($this->username);

        if (!isActionAllowed($this->username, 'host-groups-shared.get')) {
            throw new AccessDenyException("Action is not allowed. Permission to view shared group is missing.");
        }
    }

    /**
     * @rbacName Get data
     * @rbacGroup Shared host group configuration management data
     * @rbacAlias shared-group-cmdb.get
     * @rbacAllowedByDefault
     */
    public function get($request, $groupId)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!$this->isSharedGroupExists($groupId)) {
            $response->code = Response::NOTFOUND;
            return $response;
        }
        $params = $this->sanitizeQueryParams(['sortColumn' => 'created_at', 'sortDescending' => true, 'limit' => 10, 'skip' => 0]);
        $items = json_encode($this->cmdbItemModel->list($groupId, \Utils::processQueryApiParams($params)));
        $response->body = $items;

        return $response;
    }


    /**
     * @rbacName Create data
     * @rbacGroup Shared host group configuration management data
     * @rbacAlias shared-group-cmdb.create
     */
    public function post($request, $groupId)
    {

        $requestData = json_decode($request->data, associative: false);
        $requestArrayData = json_decode($request->data, associative: true);
        CmdbRequestValidator::validateCreateUpdateRequest($requestArrayData);
        /**
         * @var SharedGroupCmdbEntry
         */
        $cmdbEntry = SharedGroupCmdbEntryFactory::fromRequest($requestData, $groupId);
        $response = new \Response($request);
        $response->code = \Response::CREATED;

        if (!$this->isSharedGroupExists($groupId)) {
            $response->code = Response::NOTFOUND;
            return $response;
        }
        $group = $this->sharedHostGroupsModel->get(intval($groupId));
        $sharedGroupsEntity = new SharedHostGroupEntity(...$group);

        try {
            $insertedId = $this->cmdbItemModel->create($cmdbEntry);
            $this->sharedHostGroupsModel->sendCmdbRefreshPsqlNotification();
            $response->body = json_encode(['id' => $insertedId]);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::GROUP,
                AuditLogFields::OBJECT_ID => $groupId,
                AuditLogFields::OBJECT_NAME => $sharedGroupsEntity->getName(),
                AuditLogFields::ACTION => AuditLogActions::CMDB_UPDATED,
                AuditLogFields::DETAILS => ["Created {$cmdbEntry->name} {$cmdbEntry->type} data in group `{$sharedGroupsEntity->getName()}`."]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        } catch (\PDOException $exception) {
            DatabaseHelper::transformUniqueViolationToResponseException($exception, 'Subentry with the same name and type already exists.');
        }

        return $response;
    }
}

/**
 * @uri /host-groups/v2/shared/cmdb/:entry_id
 */
class SharedGroupCMDBSpecificItemV2 extends CMDBResource
{
    use SharedGroupVerification;
    /**
     * @var SharedGroupCmdbModel
     */
    private $cmdbItemModel;
    private $sharedHostGroupsModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new SharedGroupCmdbModel($this->username);
        $this->sharedHostGroupsModel = new SharedHostGroupsModel($this->username);

        if (!isActionAllowed($this->username, 'host-groups-shared.get')) {
            throw new AccessDenyException("Action is not allowed. Permission to view shared group is missing.");
        }
    }

    /**
     * @rbacName Get data
     * @rbacGroup Shared host group configuration management data
     * @rbacAlias shared-group-cmdb.get
     * @rbacAllowedByDefault
     */
    public function get($request, $entry_id)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        try {
            $data = $this->cmdbItemModel->getById($entry_id);
            $response->body = json_encode($data);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        }

        if ($data === null) {
            $response->code = \Response::NOTFOUND;
        }

        return $response;
    }


    /**
     * @rbacName Update data
     * @rbacGroup Shared host group configuration management data
     * @rbacAlias shared-group-cmdb.update
     */
    public function put($request, $entry_id)
    {
        $requestData = json_decode($request->data, associative: false);
        $requestArrayData = json_decode($request->data, associative: true);
        CmdbRequestValidator::validateCreateUpdateRequest($requestArrayData);
        $response = new \Response($request);
        $response->code = \Response::OK;

        $existingConfig = $this->cmdbItemModel->getById($entry_id);

        if ($existingConfig === null) {
            throw new ResponseException('Not found', Response::NOTFOUND);
        }
        $groupId = $existingConfig['group_id'];
        /**
         * @var SharedGroupCmdbEntry
         */
        $cmdbEntry = SharedGroupCmdbEntryFactory::fromRequest($requestData, $groupId);
        $sharedGroupsEntity = $this->getGroupEntity($groupId);

        try {
            $this->cmdbItemModel->update($entry_id, $cmdbEntry);
            $this->sharedHostGroupsModel->sendCmdbRefreshPsqlNotification();
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::GROUP,
                AuditLogFields::OBJECT_ID => $groupId,
                AuditLogFields::OBJECT_NAME => $sharedGroupsEntity->getName(),
                AuditLogFields::ACTION => AuditLogActions::CMDB_UPDATED,
                AuditLogFields::DETAILS => ["Updated {$cmdbEntry->name} {$cmdbEntry->type} data in group `{$sharedGroupsEntity->getName()}`.", ['data' => $requestData]]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        } catch (\PDOException $exception) {
            DatabaseHelper::transformUniqueViolationToResponseException($exception, 'Subentry with the same name and type already exists.');
        }

        return $response;
    }

    /**
     * @rbacName Delete data
     * @rbacGroup Shared host group configuration management data
     * @rbacAlias shared-group-cmdb.delete
     */
    public function delete($request, $entry_id)
    {
        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;

        $existingConfig = $this->cmdbItemModel->getById($entry_id);

        if ($existingConfig === null) {
            throw new ResponseException('Not found', Response::NOTFOUND);
        }
        $groupId = $existingConfig['group_id'];

        $sharedGroupsEntity = $this->getGroupEntity($groupId);
        try {
            $this->cmdbItemModel->delete($entry_id);
            $this->sharedHostGroupsModel->sendCmdbRefreshPsqlNotification();
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::GROUP,
                AuditLogFields::OBJECT_ID => $groupId,
                AuditLogFields::OBJECT_NAME => $sharedGroupsEntity->getName(),
                AuditLogFields::ACTION => AuditLogActions::CMDB_DELETED,
                AuditLogFields::DETAILS => ["Deleted group data entry #$entry_id in group `{$sharedGroupsEntity->getName()}`."]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        }

        return $response;
    }

    private function getGroupEntity($groupId): SharedHostGroupEntity
    {
        $group = $this->sharedHostGroupsModel->get(intval($groupId));
        return new SharedHostGroupEntity(...$group);
    }
}

/**
 * @uri /host-groups/v2/shared/:id/cmdb/subentry/:type/:name
 */
class SharedGroupCMDBSubentryController extends CMDBResource
{
    use SharedGroupVerification;
    /**
     * @var SharedGroupCmdbModel
     */
    private $cmdbItemModel;
    private $sharedHostGroupsModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new SharedGroupCmdbModel($this->username);
        $this->sharedHostGroupsModel = new SharedHostGroupsModel($this->username);
    }

    /**
     * @rbacAlias shared-group-cmdb.get
     */
    public function get($request, $id, $type, $name)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!$this->isSharedGroupExists($id)) {
            $response->code = Response::NOTFOUND;
            return $response;
        }

        $subentry = $this->cmdbItemModel->getSubEntry($type, $name, $id);
        if (empty($subentry)) {
            throw new ResponseException('Not found.', Response::NOTFOUND);
        }
        $response->body = json_encode($subentry);
        return $response;
    }
}


/**
 * @uri /host-groups/v2/shared/:id/cmdb/policy-configuration-ids
 */
class SharedGroupCMDBPolicyConfigurationIds extends CMDBResource
{
    use SharedGroupVerification;
    /**
     * @var SharedGroupCmdbModel
     */
    private $cmdbItemModel;
    private $sharedHostGroupsModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new SharedGroupCmdbModel($this->username);
        $this->sharedHostGroupsModel = new SharedHostGroupsModel($this->username);
    }

    /**
     * @rbacAlias cmdb.get
     */
    public function get($request, string $groupId)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        if (!$this->isSharedGroupExists($groupId)) {
            $response->code = Response::NOTFOUND;
            return $response;
        }

        $response->body = json_encode($this->cmdbItemModel->getPolicyConfigurationIds($groupId));
        return $response;
    }
}
