import { useEffect, useState, forwardRef, useImperativeHandle } from 'react';
import {
  Box,
  Typography,
  TextField
} from '@mui/material';
import { makeStyles } from 'tss-react/mui';

import useValidation, { MAX_NAME_LENGTH, ATTRIBUTE_NAME_PATTERN } from '../../../hooks/useValidation';
import { CmdbItem } from '../../../store/cmdbSlice/thunks';
import { CMDB_ITEM_TYPES } from '../../../store/constants';
import { getFirstEntryName, getFirstEntryValue, getFirstEntryId, getFirstEntryHostkey } from '../../../utils';

interface InventoryAttributeFormProps {
  identifier: string;
  setCmdbItem: (item: CmdbItem) => void;
  item?: CmdbItem;
}

interface InventoryAttributeFormRef {
  validate: () => Promise<boolean>;
}

const useStyles = makeStyles()((theme) => ({
  formRow: {
    display: 'flex',
    alignItems: 'center',
    '&:not(:last-of-type)': {
      marginBottom: theme.spacing(2.5),
    },
  },
  descriptionRow: {
    display: 'flex',
    alignItems: 'flex-start',
  },
  label: {
    width: 142,
    flexShrink: 0,
    marginRight: theme.spacing(3),
    color: theme.palette.text.primary,
    fontWeight: 500,
  },
  fieldContainer: {
    flex: 1,
  },
}));


const InventoryAttributeForm = forwardRef<InventoryAttributeFormRef, InventoryAttributeFormProps>(
  ({ identifier, setCmdbItem, item }, ref) => {
    const { classes } = useStyles();
    const [name, setName] = useState<string>(getFirstEntryName(item));
    const [attributeName, setAttributeName] = useState<string | null>(item?.name || null);
    const [value, setValue] = useState<string>(getFirstEntryValue(item) || '');
    const [description, setDescription] = useState<string>(item?.description || '');
    const [tags, setTags] = useState<string[]>(item?.tags || []);

    const { errors, validate } = useValidation({
      customRules: {
        attributeName: {
          required: true,
          maxLength: MAX_NAME_LENGTH,
          pattern: ATTRIBUTE_NAME_PATTERN,
          patternMessage: "Name can only contain letters, numbers, underscores and spaces.",
        }
      }
    });

    const validateForm = async (): Promise<boolean> => {
      const formItem: CmdbItem = {
        type: CMDB_ITEM_TYPES.INVENTORY,
        name: attributeName,
        attributeName,
        description,
        tags,
        entries: [{
          item_name: name,
          item_value: value,
          entry_id: item?.id,
          item_type: CMDB_ITEM_TYPES.VARIABLE,
          id: getFirstEntryId(item),
          hostkey: getFirstEntryHostkey(item)
        }]
      };

      return await validate({ item: formItem, identifier });
    };

    useImperativeHandle(ref, () => ({
      validate: validateForm,
    }), [validate, name, attributeName, value, description, tags, item, identifier]);

    useEffect(() => {
      const formItem: CmdbItem = {
        type: CMDB_ITEM_TYPES.INVENTORY,
        name: attributeName,
        description,
        tags,
        entries: [{
          item_name: name,
          item_value: value,
          item_type: "variable",
          id: getFirstEntryId(item),
          hostkey: getFirstEntryHostkey(item),
          entry_id: item?.id
        }]
      };

      setCmdbItem(formItem);
    }, [name, description, tags, value, attributeName, setCmdbItem, item]);

    useEffect(() => {
      if (attributeName !== null) return;

      const extractedName = tags.reduce((acc, tag) => {
        if (tag.includes("attribute_name=")) {
          return tag.replace("attribute_name=", "");
        }
        return acc;
      }, "");

      if (extractedName) {
        setAttributeName(extractedName);
      }
    }, [tags, attributeName]);

    useEffect(() => {
      if (attributeName) {
        setTags([CMDB_ITEM_TYPES.INVENTORY, `attribute_name=${attributeName}`]);
        setName(`data:variables.${attributeName.toLowerCase().replace(/\s+/g, '_')}`);
      }
    }, [attributeName]);

    return (
      <>
        <Box className={classes.formRow}>
          <Typography variant="body1" className={classes.label}>
            Name:
          </Typography>
          <Box className={classes.fieldContainer}>
            <TextField
              fullWidth
              variant="outlined"
              size="small"
              placeholder="Attribute name"
              value={attributeName}
              onChange={(e) => setAttributeName(e.target.value)}
              error={!!errors.attributeName || !!errors.item_name}
              helperText={errors.attributeName || errors.item_name}
            />
          </Box>
        </Box>
        <Box className={classes.formRow}>
          <Typography variant="body1" className={classes.label}>
            Value:
          </Typography>
          <Box className={classes.fieldContainer}>
            <TextField
              fullWidth
              variant="outlined"
              size="small"
              placeholder="Value"
              value={value}
              onChange={(e) => setValue(e.target.value)}
              error={!!errors.item_value}
              helperText={errors.item_value}
            />
          </Box>
        </Box>
        <Box className={classes.descriptionRow}>
          <Typography variant="body1" className={classes.label}>
            Description:
          </Typography>
          <Box className={classes.fieldContainer}>
            <TextField
              variant="outlined"
              multiline
              fullWidth
              rows={3}
              placeholder="Description"
              value={description}
              onChange={(e) => setDescription(e.target.value)}
              error={!!errors.description}
              helperText={errors.description}
            />
          </Box>
        </Box>
      </>
    );
  });

export default InventoryAttributeForm;
