<?php

namespace CMDB\v2;

use AuditLogActions;
use AuditLogObjectTypes;
use AuditLogFields;
use AuditLogService;
use CMDB\v2\Entities\CmdbHostEntry;
use CMDB\v2\Entities\CmdbEntryFactory;
use CMDB\v2\Models\CmdbItemModel;
use CMDB\v2\Validators\CmdbRequestValidator;
use FR\V1_0\Entities\RemoteHub\RemoteHub;
use FR\V1_0\Models\SetupHubModel;
use Response;
use ResponseException;

class CMDBResource extends \CfProtectedResource
{
    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $setupHubModel = new SetupHubModel(\CfSettings::getInstance()->getConnection(), $this->username);
        if ($setupHubModel->getHubRole() == RemoteHub::SUPERHUB_ROLE && $setupHubModel->isConfigured()) {
            throw new \Exception('CMDB API is not allowed on the superhub.');
        }
    }

    protected function sanitizeQueryParams(array $defaults = []): array
    {
        $allowedParams = ['sortColumn', 'sortDescending', 'limit', 'skip'];
        $sanitizedRequest = array_intersect_key($_REQUEST, array_flip($allowedParams));

        if (isset($sanitizedRequest['limit'])) {
            $sanitizedRequest['limit'] = max(1, min(100, (int)$sanitizedRequest['limit']));
        }
        if (isset($sanitizedRequest['skip'])) {
            $sanitizedRequest['skip'] = max(0, (int)$sanitizedRequest['skip']);
        }
        if (isset($sanitizedRequest['sortDescending'])) {
            $sanitizedRequest['sortDescending'] = filter_var($sanitizedRequest['sortDescending'], FILTER_VALIDATE_BOOLEAN);
        }
        if (isset($sanitizedRequest['sortColumn'])) {
            $sanitizedRequest['sortColumn'] = preg_replace('/[^a-zA-Z0-9_]/', '', $sanitizedRequest['sortColumn']);
        }

        return array_merge($defaults, $sanitizedRequest);
    }
}


/**
 * @uri /cmdb/v2/:hostkey 0
 */
class CMDBItem extends CMDBResource
{
    /**
     * @var CmdbItemModel
     */
    private $cmdbItemModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new CmdbItemModel($this->username);
    }

    /**
     * @rbacName Get config
     * @rbacDescription
     * @rbacGroup CMDB
     * @rbacAlias cmdb.get
     * @rbacAllowedByDefault
     */
    public function get($request, $hostkey)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;

        $params = $this->sanitizeQueryParams(['sortColumn' => 'created_at', 'sortDescending' => true]);
        $items = json_encode($this->cmdbItemModel->list($hostkey, \Utils::processQueryApiParams($params)));

        $response->body = $items;

        return $response;
    }

    /**
     * @rbacName Create config
     * @rbacDescription
     * @rbacGroup CMDB
     * @rbacAlias cmdb.create
     */
    public function post($request, string $hostkey)
    {
        $hostname = $this->checkHostAccessAndReturnName($this->username, $hostkey);
        $requestData = json_decode($request->data, associative: false);
        $requestArrayData = json_decode($request->data, associative: true);
        CmdbRequestValidator::validateCreateUpdateRequest($requestArrayData);
        /**
         * @var CmdbHostEntry
         */
        $cmdbEntry = CmdbEntryFactory::fromRequest($requestData, $hostkey);
        $response = new \Response($request);
        $response->code = \Response::CREATED;

        try {
            $insertedId = $this->cmdbItemModel->create($cmdbEntry);
            $response->body = json_encode(['id' => $insertedId]);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::HOST,
                AuditLogFields::OBJECT_ID => $hostkey,
                AuditLogFields::OBJECT_NAME => $hostname,
                AuditLogFields::ACTION => AuditLogActions::CMDB_CREATED,
                AuditLogFields::DETAILS => ["Created CMDB {$cmdbEntry->name} {$cmdbEntry->type} for the $hostkey host.", ['requestData' => $cmdbEntry->toArray()]]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        } catch (\PDOException $exception) {
            \DatabaseHelper::transformUniqueViolationToResponseException($exception, 'Subentry with the same name and type already exists.');
        }

        return $response;
    }
}


/**
 * @uri /cmdb/v2/subentry/:hostkey/:type/:name
 */
class CMDBSubentry extends CMDBResource
{
    /**
     * @var CmdbItemModel
     */
    private $cmdbItemModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new CmdbItemModel($this->username);
    }

    /**
     * @rbacAlias cmdb.get
     */
    public function get($request, $hostkey, $type, $name)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;
        $subentry = $this->cmdbItemModel->getSubEntry($type, $name, $hostkey);
        if ($subentry === null) {
            throw new ResponseException('Not found.', Response::NOTFOUND);
        }
        $response->body = json_encode($subentry);
        return $response;
    }
}

/**
 * @uri /cmdb/v2/entry/:id
 */
class CMDBSpecificItem extends CMDBResource
{
    /**
     * @var CmdbItemModel
     */
    private $cmdbItemModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new CmdbItemModel($this->username);
    }

    /**
     * @rbacName Update config
     * @rbacDescription
     * @rbacGroup CMDB
     * @rbacAlias cmdb.update
     */
    public function put($request, int $id)
    {
        $item = $this->cmdbItemModel->getById($id);

        if (!$item) {
            throw new ResponseException('Not found', Response::NOTFOUND);
        }
        $hostkey = $item['hostkey'];
        $hostname = $this->checkHostAccessAndReturnName($this->username, $hostkey);
        $requestData = json_decode($request->data, associative: false);
        $requestArrayData = json_decode($request->data, associative: true);
        CmdbRequestValidator::validateCreateUpdateRequest($requestArrayData);

        /**
         * @var CmdbHostEntry
         */
        $cmdbEntry = CmdbEntryFactory::fromRequest($requestData, $hostkey);
        $response = new \Response($request);
        $response->code = \Response::OK;

        try {
            $this->cmdbItemModel->update($id, $cmdbEntry);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::HOST,
                AuditLogFields::OBJECT_ID => $hostkey,
                AuditLogFields::OBJECT_NAME => $hostname,
                AuditLogFields::ACTION => AuditLogActions::CMDB_UPDATED,
                AuditLogFields::DETAILS => ["Updated CMDB host entry {$cmdbEntry->name} {$cmdbEntry->type} for host '$hostkey'.", ['requestData' => $cmdbEntry->toArray()]]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        } catch (\PDOException $exception) {
            \DatabaseHelper::transformUniqueViolationToResponseException($exception, 'Subentry with the same name and type already exists.');
        }

        return $response;
    }

    /**
     * @rbacName Delete config
     * @rbacDescription
     * @rbacGroup CMDB
     * @rbacAlias cmdb.delete
     */
    public function delete($request, int $id)
    {
        $item = $this->cmdbItemModel->getById($id);

        if (!$item) {
            throw new ResponseException('Not found', Response::NOTFOUND);
        }
        $hostkey = $item['hostkey'];
        $hostname = $this->checkHostAccessAndReturnName($this->username, $hostkey);
        $response = new \Response($request);
        $response->code = \Response::NOCONTENT;

        try {
            $this->cmdbItemModel->delete($id);
            AuditLogService::register([
                AuditLogFields::ACTOR => $this->username,
                AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::HOST,
                AuditLogFields::OBJECT_ID => $hostkey,
                AuditLogFields::OBJECT_NAME => $hostname,
                AuditLogFields::ACTION => AuditLogActions::CMDB_DELETED,
                AuditLogFields::DETAILS => ["Deleted CMDB host entry #$id for the $hostkey host."]
            ]);
        } catch (\InvalidArgumentException $exception) {
            $response->body = $exception->getMessage();
            $response->code = \Response::BADREQUEST;
        }

        return $response;
    }
}

/**
 * @uri /cmdb/v2/:hostkey/policy-configuration-ids
 */
class CmdbPolicyConfigurationIds extends CMDBResource
{
    /**
     * @var CmdbItemModel
     */
    private $cmdbItemModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new CmdbItemModel($this->username);
    }

    /**
     * @rbacAlias cmdb.get
     */
    public function get($request, string $hostkey)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;
        cfapi_host_get($this->username, $hostkey);

        $ids = json_encode($this->cmdbItemModel->getPolicyConfigurationIds($hostkey, \Utils::processQueryApiParams([])));
        $response->body = $ids;
        return $response;
    }
}


/**
 * @uri /cmdb/v2/:hostkey/json 0
 */
class CMDBViewConfig extends CMDBResource
{
    /**
     * @var CmdbItemModel
     */
    private $cmdbItemModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cmdbItemModel = new CmdbItemModel($this->username);
    }

    /**
     * @rbacName View host specific rendered JSON config
     * @rbacDescription
     * @rbacGroup CMDB
     * @rbacAlias cmdb.view-host-json
     * @rbacAllowedByDefault
     */
    public function get($request, string $hostkey)
    {
        $response = new \Response($request);
        $response->code = \Response::OK;
        cfapi_host_get($this->username, $hostkey); // check if user has RBAC access to this host

        $filePath = realpath(CMDB_PATH . "/$hostkey/host_specific.json");

        // Check if the resulted file path is within the cmdb directory
        if (strpos($filePath, CMDB_PATH) !== 0) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'File not found or not readable.';
            return $response;
        }

        if (!file_exists($filePath)) {
            $response->code = \Response::NOTFOUND;
            $response->body = 'File not found or not readable.';
        } else {
            $response->body = file_get_contents($filePath);
        }

        return $response;
    }
}
