import { useCallback, useEffect, useState } from 'react';

import { makeStyles } from 'tss-react/mui';
import { useSelector, useDispatch } from 'react-redux';

import { Pagination } from '@northern.tech/common-ui';
import { isDarkMode } from '@northern.tech/store/utils';

import Spinner from '../common/Spinner';
import CmdbCreate from './components/CmdbCreate';
import CmdbEdit from './components/CmdbEdit';
import CmdbItemView from './components/CmdbItemView';
import ShowJson from './components/ShowRenderedJson';
import { setIdentifier, setCmdbServiceUrl, setPage } from './store/cmdbSlice';
import { getPermissions } from '../../store/usersSlice/thunks';
import { selectCmdbItems, selectCmdbLoading, selectPagination } from './store/cmdbSlice/selectors';
import { getItems, getPolicyConfigurationsIds } from './store/cmdbSlice/thunks';
import { TYPES_NAMES_MAP } from './store/constants';
import CmdbItemPreviewFormatter from './components/CmdbItemPreviewFormatter';

import {
  Accordion,
  AccordionDetails,
  AccordionSummary,
  Box,
  Button,
  Paper,
  Typography,
} from '@mui/material';
import {
  Add as AddIcon,
  ExpandMore as ExpandMoreIcon
} from '@mui/icons-material';
import { selectPermissions } from '../../store/usersSlice/selectors';
import { isCreateAllowed } from './utils';


interface CmdbProps {
  identifier: string;
  ishub?: boolean;
  displayShowJson?: boolean;
  cmdbServiceUrl?: string;
}

const useStyles = makeStyles()((theme) => ({
  accordion: {
    border: `1px solid ${theme.palette.primary.border}`,
    borderRadius: '4px',
    padding: 1,
    boxShadow: 'none',
    '&::before': {
      display: 'none',
    },
    '&:not(:last-of-type)': {
      marginBottom: theme.spacing(1),
    }
  },
  summary: {
    backgroundColor: theme.palette.background.default,
    '&.Mui-expanded': {
      minHeight: '48px',
      backgroundColor: theme.palette.background.summary
    },
    '& .MuiAccordionSummary-content': {
      display: 'flex',
      alignItems: 'center',
      margin: `${theme.spacing(1.5)} 0`,
      '&.Mui-expanded': {
        margin: `${theme.spacing(1.5)} 0`,
      }
    },
  },
  itemLabel: {
    fontWeight: 500,
    width: 200
  },
  itemValue: {
    fontWeight: 500,
    color: theme.palette.gray[600]
  },
  truncate: {
    whiteSpace: 'nowrap',
    overflow: 'hidden',
    textOverflow: 'ellipsis',
    maxWidth: 400
  },
  details: {
    backgroundColor: isDarkMode(theme.palette.mode) ? theme.palette.darkBackground[700] : theme.palette.background.default,
    padding: theme.spacing(2),
    borderTop: `1px solid ${theme.palette.divider}`,
  },
  buttonContainer: {
    display: 'flex',
    gap: theme.spacing(1),
  },
  pagination: {
    justifyContent: 'flex-start',
    ' .MuiToolbar-root': {
      paddingLeft: 0
    },
    ' .MuiTablePagination-select, .MuiTablePagination-selectLabel': {
      display: 'none'
    }
  }
}));

const Cmdb: React.FC<CmdbProps> = ({
  identifier,
  displayShowJson = false,
  cmdbServiceUrl = ''
}) => {
  const { classes } = useStyles();
  const dispatch = useDispatch();
  const items = useSelector(selectCmdbItems);
  const paginationState = useSelector(selectPagination);
  const permissions = useSelector(selectPermissions);
  const loading = useSelector(selectCmdbLoading);
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);
  const [actionInterrupted, setActionInterrupted] = useState(false);

  useEffect(()=> {
     dispatch(getPermissions());
  }, []);

  useEffect(() => {
    dispatch(setIdentifier(identifier));
    dispatch(setCmdbServiceUrl(cmdbServiceUrl));
    dispatch(getItems({ identifier, baseUrl: cmdbServiceUrl }));
    dispatch(getPolicyConfigurationsIds({ identifier, baseUrl: cmdbServiceUrl }));
  }, [dispatch, identifier, cmdbServiceUrl]);

  const changePagination = useCallback(
    (page: number) => {
      dispatch(setPage(page));
      dispatch(getItems({ identifier, page, baseUrl: cmdbServiceUrl }));
    },
    [dispatch, identifier, cmdbServiceUrl]);

  const [expanded, setExpanded] = useState<string | false>(false);
  const [addingNew, setAddingNew] = useState<boolean>(false);
  const [editing, setEditing] = useState<number | undefined>();

  const handleAddingNew = useCallback(() => {
    if (hasUnsavedChanges) {
      setActionInterrupted(true);
    } else {
      setAddingNew(true);
      setEditing(undefined);
      setExpanded(false);
    }
  }, [hasUnsavedChanges]);

  const handleChange = useCallback((panel: string) => (event: React.SyntheticEvent, isExpanded: boolean) => {
    setExpanded(isExpanded ? panel : false);
    setAddingNew(false);
  }, []);

  return (
    <Box className="relative">
      {loading && <Spinner />}
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', pb: 2 }}>
        <h4>Host specific data</h4>
        <Box>
          {displayShowJson && <ShowJson identifier={identifier}></ShowJson>}
          {isCreateAllowed(permissions) && <Button
            variant="outlined"
            color="secondary"
            startIcon={<AddIcon />}
            onClick={handleAddingNew}
            size="small"
            sx={{ ml: 1 }}
          >
            Add
          </Button>}
        </Box>
      </Box>
      <Box>
        {addingNew && (
          <CmdbCreate
            onCancel={() => setAddingNew(false)}
            identifier={identifier}
          />
        )}
        {items.map((item) => (
          <Accordion
            key={item.id}
            expanded={expanded === item.id}
            onChange={handleChange(item.id)}
            className={classes.accordion}
          >
            <AccordionSummary
              expandIcon={<ExpandMoreIcon />}
              className={classes.summary}
            >
              <Typography variant="body2" component={Box} className={classes.itemLabel}>
                {TYPES_NAMES_MAP[item.type] ?? ''}
              </Typography>
              {expanded !== item.id && (
                <Typography variant="body2" component={Box} className={classes.itemValue}>
                  <Box className={classes.truncate}>
                    {CmdbItemPreviewFormatter(item)}
                  </Box>
                </Typography>
              )}
            </AccordionSummary>
            <AccordionDetails className={classes.details}>
              {editing === item.id ? (
                <CmdbEdit
                  item={item}
                  setHasUnsavedChanges={setHasUnsavedChanges}
                  actionInterrupted={actionInterrupted}
                  identifier={identifier}
                  onCancel={() => {
                    setEditing(undefined);
                    setHasUnsavedChanges(false);
                    setActionInterrupted(false);
                  }}
                />
              ) : (
                <CmdbItemView
                  item={item}
                  onValueClick={() => {
                    setEditing(item.id);
                    setAddingNew(false);
                  }}
                />
              )}
            </AccordionDetails>
          </Accordion>
        ))}
        {!loading && items.length === 0 && (
          <Paper elevation={0}>
            <Box sx={{ py: 4, px: 2, textAlign: 'center' }}>
              <Typography variant="h6" fontSize={14} color="textSecondary">
                No data were found
              </Typography>
            </Box>
          </Paper>
        )}
      </Box>
      {paginationState.total !== 0 && 
      <Pagination
        className={classes.pagination}
        count={paginationState.total}
        rowsPerPage={paginationState.perPage}
        onChangeRowsPerPage={() => { }}
        page={paginationState.page}
        onChangePage={changePagination}
      />}
    </Box>
  );
};

export default Cmdb;
