<?php

use auditLogs\constants\AuditLogSettingsNames;

/**
 * @uri /ai-settings
 */
class LLMConfigs extends CfProtectedResource
{
    private CfLLMConfig $cfLLMConfig;
    public function __construct($parameters)
    {
        parent::__construct(parameters: $parameters);
        $this->cfLLMConfig = new CfLLMConfig(\CfSettings::getInstance()->getConnection());
    }
    /**
     * @param $request
     *
     * @rbacName View AI settings
     * @rbacGroup AI settings
     * @rbacAlias ai-settings.get
     * @rbacAllowedByDefault
     *
     * @return Response
     */
    public function get($request)
    {
        $response = new Response($request);
        $configs = $this->cfLLMConfig->list();
        $response->body = json_encode($configs);
        $response->code = Response::OK;
        return $response;
    }

    /**
     * @param $request
     *
     * @rbacName Update AI settings
     * @rbacGroup AI settings
     * @rbacAlias ai-settings.update
     *
     * @return Response
     * @throws Exception
     */
    public function post($request)
    {
        $data = json_decode($request->data, associative: true) ?? [];
        LLMConfigsValidator::validateCreateUpdateRequest($data);

        $config = LLMConfigEntity::fromArray($data);
        $id = $this->cfLLMConfig->create($config);

        AuditLogService::register([
            AuditLogFields::ACTOR => $this->username,
            AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::SETTINGS,
            AuditLogFields::OBJECT_NAME => AuditLogSettingsNames::AI_SETTINGS,
            AuditLogFields::ACTION => AuditLogActions::CREATE,
            AuditLogFields::DETAILS => ['AI settings created.', ['provider' => $config->provider->value, 'model' => $config->model]],
        ]);

        $response = new Response($request);
        $response->code = Response::CREATED;
        $response->body = json_encode(['id' => $id]);

        return $response;
    }
}

/**
 * @uri /ai-settings/:id
 */
class LLMConfigsItem extends CfProtectedResource
{
    private CfLLMConfig $cfLLMConfig;
    public function __construct($parameters)
    {
        parent::__construct($parameters);
        $this->cfLLMConfig = new CfLLMConfig(\CfSettings::getInstance()->getConnection());
    }
    /**
     * @param $request
     *
     * @rbacName View AI settings
     * @rbacGroup AI settings
     * @rbacAlias ai-settings.get
     * @rbacAllowedByDefault
     *
     * @return Response
     */
    public function get($request, $id)
    {
        $response = new Response($request);
        $config = $this->cfLLMConfig->get($id);
        if (!$config) {
            $response->code = Response::NOTFOUND;
            return $response;
        }
        $response->body = json_encode($config);
        $response->code = Response::OK;
        return $response;
    }

    /**
     * @param $request
     *
     * @rbacName Update AI settings
     * @rbacGroup AI settings
     * @rbacAlias ai-settings.update
     *
     * @return Response
     * @throws Exception
     */
    public function patch($request, $id)
    {
        $response = new Response($request);

        $config = $this->cfLLMConfig->get($id);
        if (!$config) {
            $response->code = Response::NOTFOUND;
            return $response;
        }

        $data = json_decode($request->data, associative: true) ?? [];
        LLMConfigsValidator::validateCreateUpdateRequest($data);

        $newConfig = LLMConfigEntity::fromArray($data);

        $this->cfLLMConfig->update($id, $newConfig);

        AuditLogService::register([
            AuditLogFields::ACTOR => $this->username,
            AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::SETTINGS,
            AuditLogFields::OBJECT_NAME => AuditLogSettingsNames::AI_SETTINGS,
            AuditLogFields::ACTION => AuditLogActions::UPDATE,
            AuditLogFields::DETAILS => ['AI settings updated.', ['provider' => $newConfig->provider->value, 'model' => $newConfig->model]],
        ]);

        $response->code = Response::ACCEPTED;
        return $response;
    }

    /**
     * @param $request
     *
     * @rbacName Update AI settings
     * @rbacGroup AI settings
     * @rbacAlias ai-settings.update
     *
     * @return Response
     * @throws Exception
     */
    public function delete($request, $id)
    {
        $response = new Response($request);

        $config = $this->cfLLMConfig->get($id);
        if (!$config) {
            $response->code = Response::NOTFOUND;
            return $response;
        }

        $this->cfLLMConfig->delete($id);

        AuditLogService::register([
            AuditLogFields::ACTOR => $this->username,
            AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::SETTINGS,
            AuditLogFields::OBJECT_NAME => AuditLogSettingsNames::AI_SETTINGS,
            AuditLogFields::ACTION => AuditLogActions::DELETE,
            AuditLogFields::DETAILS => ['AI settings deleted.', ['provider' => $config->provider->value, 'model' => $config->model]],
        ]);

        $response->code = Response::ACCEPTED;
        return $response;
    }
}
