/**
 * Widget to generate vitals graph
 * Depends on Jquery plugin : aware 1.1.1 for lazy loading
 */
(function($) {
    $.widget('ui.vitalSigns', {
        options: {
            contextWidgetElement: 'hclist',
            baseUrl: '',
            interval: 0, // Interval in minutes to refresh the widget
            defaultNumberOfGraphs : 10,
            showVitalSelectionForHost : false,
            disableExtraButtons : false,
            vitalsurl: '/vitals/node_vitals',
            vitalsurlForHost: '/vitals/host_vitals',
            vitalsurlForConfiguration: '/vitals/config_vitals',
            vitalsurlForTopMenu: '/vitals/vital_panel_menu',
            vitalsurlForTopMenuHostView: '/vitals/vital_panel_menu_host',
            vitalsMainLink: '/visual/vital/',
            defaultVital:'loadavg',
            disableScrollOnTop: false
        },


        _create: function() {

            this._context =  {
                includes: [],
                excludes: []
            };

            this.vitalPanel=  $('<div class="graph-container-header-top-menu"></div>' +
                '<div class="loading-div float-left"><span class="refresh-timer-status"></span></div><div class="graph-container-body"></div>' +
                '<div class="graph-container-footer-menu"></div>');

            this.showMoreIcon =  $('<span class="showmore button"  style="display:none">').html('<button class="button">show more...</button>');
            this.errorDiv =  $('<div style="display:none">');
            this.goToTopIcon = $('<div class="pull-right go-to-top"><i class="icon-arrow-up"></i></div>');


            this._startIndex=  0;

            this._cachedData= [];
            this._hostView= false;



            this.selectedVital= this.options.defaultVital;
            this.selectedVitalSortBy= 'last-measured';
            this.selectedVitalOrderBy= 'desc';
            this.selectedVitalTimeFrame= '6';
            this._hostObservable= '', // used for host only observables

            this._timer= null;
            this._statustimer= null;
            this._vitalConfig= null;
            this.graphWidth = 300;
            this.graphHeight = 310;
            this.active = true;
            this.disableScrollOnTop = this.options.disableScrollOnTop;

        },


        _init: function() {
            var $self = this;
            $self.element.append($self.vitalPanel);
            $self._fetchConfig();
            $self.element.find('.loading-div').prepend($self.errorDiv);
            if (!$self.disableScrollOnTop) {
                $('body').append($self.goToTopIcon);
                $self.goToTopIcon.click(function() {
                    $("html, body").animate({
                        scrollTop: 0
                    }, 600);
                    return false;

                });

                $(window).on('scroll', function()
                {
                    if (!$self.active) {
                        $self.goToTopIcon.hide();
                        return false;
                    }

                    if ($(this).scrollTop() === 0) {
                        $self.goToTopIcon.hide();
                    } else {
                        if ($self.active) $self.goToTopIcon.show();
                    }

                });
            }

            $(window).resize(function() {
                $self.getDivsPerRow();
            });
        },

        setContext: function(includes,excludes) {
            var $self = this;
            $self._hostView = false;
            $self.clearCanvas();
            $self._modifyContext(includes, excludes);
            if ($self.active) {
                $self.refresh();
            }
        },

        setHostObservables:function(obs) {
            var $self = this;
            $self._hostObservable = obs;
        },

        setHostViewMode: function () {
            var $self = this;
            $self._hostView = true;
        },

        /**
         * active mode will disable the scroll events and refresh events
         */
        setActiveMode:function(active) {
            this.active = active;
        },

        updateTimerStatus: function(minutes,seconds) {
            var $self = this;
            var el = $self.element.find('.refresh-timer-status');
            var time = minutes*60 + seconds;

            this._statustimer = window.setInterval(
                function() {
                    // if the time is 0 then end the counter
                    if(time == 0) {
                        el.innerHTML = "countdown's over!";
                        clearInterval( this._statustimer);
                        return;
                    }
                    var minutes = Math.floor( time / 60 );
                    if (minutes < 10) minutes = "0" + minutes;
                    var seconds = time % 60;
                    if (seconds < 10) seconds = "0" + seconds;
                    var text = minutes + ':' + seconds;
                    el.html('Refreshing in '+text+' sec');
                    time--;
                },
                1000
                );

        },
        _setTimer: function(value) {
            this.options.interval = value;
            var $self = this;

            if (value == -1) {
                this._resetTimers();
                return;
            }

            if (value > 0) {
                window.clearInterval(this._statustimer);
                window.clearInterval(this._timer);
                // Set a new timer based on the new interval
                this.updateTimerStatus(value,0);
                this._timer = window.setInterval(
                    function() {
                        $.proxy($self.refresh(), $self);
                        $self.updateTimerStatus(value,0);
                    },
                    this.options.interval * 60 * 1000
                    );
            } else {
                // refresh any way
                this.refresh();
            }
        },

        _resetTimers: function() {
            window.clearInterval(this._statustimer);
            window.clearInterval(this._timer);
            this.element.find('.refresh-timer-status').html('');
        },

        _setOption: function(key, value) {
            if (this.options[key] === value) {
                return this; // Do nothing, the value didn't change
            }
            if (key === 'interval') {
                this._setTimer(value);
            }
            return this;
        },

        _modifyContext: function(includes,excludes) {
            var $self = this;
            $self._context.includes = includes;
            $self._context.excludes = excludes;
            $self._resetTimers();
            $self._initializeTopMenu();
        },

        setHostContext: function(hostkey) {

            var $self = this,
            includes = [common.canonify('PK_' + hostkey)],
            excludes = [];
            $self._hostView = true;
            $self.clearCanvas();
            $self._modifyContext(includes, excludes);
            if ($self.active) {
                $self.refresh();
            }
        },

        refresh: function() {
            var $self = this;
            $self.clearCanvas();
            if ($self._hostView === false) {
                $self.refreshForNodeView();
            } else {
                $self.refreshForHostView();
            }
        },

        clearCanvas: function() {
            var $self = this;
            $self.errorDiv.hide();
            $self.element.find('.graph-container-body').empty();
            $self.element.find('.graph-container-footer-menu').empty();
            $self.resetCounter();
        },

        resetCounter: function() {
            var $self = this;
            $self._startIndex = 0;
            $self._cachedData = [];
        },


        _initializeTopMenu: function() {

            var $self = this;

            if (this._hostView == false) {
                var params = {
                    'url': $self.options.baseUrl + $self.options.vitalsurlForTopMenu,
                    'type': 'get',
                    'dataType': 'html',
                    'success': function(data) {
                        $self.element.find('.graph-container-header-top-menu').html(data);

                        $self.element.find('.vitalsSortBySelect').val($self.selectedVitalSortBy);

                        $self.element.find('.vitalsSortBySelectOrder').find('[data-order="'+$self.selectedVitalOrderBy+'"]').addClass("active");

                        $self.element.find('.vitalsTimeFrame').find('[data-hour="'+$self.selectedVitalTimeFrame+'"]').addClass("active");

                        $self.element.find('.selectVital').click(function() {
                            $self.selectedVital = ($(this).attr("data-vital"));
                            $self.clearCanvas();
                            $self.refreshForNodeView();
                            $self.element.find('#selectedVital').text($(this).text());
                        });

                        $self.element.find('.vitalsSortBySelect').change(function() {
                            $self.selectedVitalSortBy = ($(this).val());
                            $self.clearCanvas();
                            $self.refreshForNodeView();
                        });

                        $self.element.find('.vitalsSortBySelectOrder li>a').click(function() {
                            $('.vitalsSortBySelectOrder li>a').removeClass('active');
                            $(this).addClass("active");
                            $self.selectedVitalOrderBy = ($(this).attr("data-order"));
                            $self.clearCanvas();
                            $self.refreshForNodeView();
                        });

                        $self.element.find('.vitalsTimeFrame li>a').click(function() {
                            $self.element.find('.vitalsTimeFrame li>a').removeClass('active');
                            $(this).addClass("active");
                            $self.selectedVitalTimeFrame = ($(this).attr("data-hour"));
                            $self.clearCanvas();
                            $self.refreshForNodeView();
                        });

                        $self.element.find('.refresh-trigger').click(function() {
                            $self._setTimer($(this).attr('rel'));
                        })
                    }
                };
                $self.sendRequest(params);

            } else {
                // build menu for host view only

                var params = {
                    'url': $self.options.baseUrl + $self.options.vitalsurlForTopMenuHostView,
                    'type': 'get',
                    'data':{
                        'obs':$self._hostObservable
                    },
                    'dataType': 'html',
                    'success': function(data) {
                        $self.element.find('.graph-container-header-top-menu').html(data);

                        $self.element.find('.vitalsTimeFrame').find('[data-hour="'+$self.selectedVitalTimeFrame+'"]').addClass("active");

                         $self.element.find('.vitalsTimeFrame li>a').click(function() {
                            $self.element.find('.vitalsTimeFrame li>a').removeClass('active');
                            $(this).addClass("active");
                            $self.selectedVitalTimeFrame = ($(this).attr("data-hour"));
                            $self.clearCanvas();
                            $self.refreshForHostView();
                        });

                        if ($self.options.showVitalSelectionForHost) {

                            $self.element.find('.selectVital').click(function() {
                                $self._hostObservable = ($(this).attr("data-vital"));
                                $self.clearCanvas();
                                $self.refreshForHostView();
                                $self.element.find('#selectedVital').text($(this).attr("data-vital"));
                            });


                        } else  {
                            $self.element.find('.graph-container-header-top-menu .node-menu .vitalsSelect').css('visibility','hidden');
                        }

                        if ($self.options.disableExtraButtons) {
                            $self.element.find('#extrabuttons').hide();
                        }

                        $self.element.find('.refresh-trigger').click(function() {
                            $self._setTimer($(this).attr('rel'));
                        })
                    }
                };
                $self.sendRequest(params);

            }


        },

        _fetchConfig: function() {
            var $self = this;
            var params = {
                'url': $self.options.baseUrl + $self.options.vitalsurlForConfiguration,
                'type': 'get',
                'success': function(data) {
                    $self._vitalConfig = data;
                },
                'error': function() {

                }
            };
            $self.sendRequest(params);
        },

        showEmptyData: function(type) {
            // cannot fetch any data
            var self = this;
            self.element.find('.graph-container-header-top-menu').css('display', 'none');
            var type = type || 'group';
            var message = 'No monitoring data has been collected for this '+ type +' yet. Please check the <span class="italic">report_data_select</span> attribute in your policy, or <a href="'+self.options.baseUrl+'docs/help#monitoring-help">find out how to enable collection of monitoring data</a>.'
            var infoText = $('<div style="margin-top:10px;">').addClass('alert').html(message);
            self.errorDiv.html(infoText);
            self.errorDiv.show();
        },

        refreshForNodeView: function() {
            var self = this;
            var params = {
                'url': self.options.baseUrl + self.options.vitalsurl,
                'type': 'Get',
                'data': {
                    'obs': self.selectedVital,
                    'sort': self.selectedVitalSortBy,
                    'sortOrder': self.selectedVitalOrderBy,
                    'timeFrame': self.selectedVitalTimeFrame
                },
                'success': function(data) {
                    if (data === null || data.length === 0) {
                        self.showEmptyData('group');
                        return;
                    }
                    self._cachedData = data;
                    self.fixHeightOfContainer();
                    self.drawGraphCanvas(self._cachedData);
                    self.getDivsPerRow();
                }
            };
            self.sendRequest(params);
        },

        refreshForHostView: function() {
            var self = this;
            var params = {
                'url': self.options.baseUrl + self.options.vitalsurlForHost,
                'type': 'Get',
                'data': {
                    'obs':self._hostObservable,
                    'timeFrame': self.selectedVitalTimeFrame
                },
                'success': function(data) {
                    if (data === null || data.length === 0 || data.obs.length == 0) {
                        self.showEmptyData('host');
                        return;
                    }
                    self.drawGraphCanvasForHost(data);
                    self.getDivsPerRow();
                }
            };
            self.sendRequest(params);

        },



        plotGraph: function($plotdiv,updateLastValue) {
            var self = this;

            var meta = [];
            meta['obs']=$plotdiv.attr('data-obs');
            meta['hostkey'] = $plotdiv.attr('data-hostkey');
            meta['lastUpdated'] = $plotdiv.attr('data-lastUpdated');
            var lastValue = null;

            var params = {
                'url': self.options.baseUrl + '/vitals/host_vital_performance_data',
                'type': 'GET',
                'errorDiv': $plotdiv,
                'dontShowGlobalSpinner': true,
                'data': {
                    'obs':meta.obs,
                    'timeFrame': self.selectedVitalTimeFrame,
                    'hostkey':meta.hostkey,
                    'lastUpdated':meta.lastUpdated
                },
                'success': function(data) {

                    var length = data != null ? data.length : 0;
                    var lastDataSeries = [];
                    var maxArrayValue = 0; // for scaling the y axis max value
                    // add some internal data to the div
                    $plotdiv.data('hostkey',meta.hostkey);
                    $plotdiv.data('obs',meta.obs);
                    for (var i = 0; i < length; i++) {
                        if (data[i][1] > maxArrayValue) {
                            maxArrayValue = data[i][1];
                        }
                        lastDataSeries.push([data[i][0]*1000, data[i][1]]);
                        if (i == (length -1) ) lastValue = data[i][1];
                    }


                    var ymin = null;
                    var ymax = null;
                    if (('obs' in meta) &&  self._vitalConfig && self._vitalConfig.vitals[meta.obs] !== undefined) {

                        var configYmax = self._vitalConfig.vitals[meta.obs].max ? self._vitalConfig.vitals[meta.obs].max : null;
                        var configYmin = self._vitalConfig.vitals[meta.obs].min ? self._vitalConfig.vitals[meta.obs].min : 0;
                        ymax = configYmax > maxArrayValue ? self._vitalConfig.vitals[meta.obs].max : null;
                        ymin = configYmin;
                    }


                    var options = {
                        grid: {
                            hoverable: true,
                            clickable: true,
                            axisMargin: 15,
                            labelMargin: 15,
                        },
                        yaxis: {
                            min: 0,
                            max: ymax
                        },
                        xaxis: {

                            // timeformat: "%b %d <br/>%H:%M %p",
                            /// mode:'time',
                            ticks:4,
                            max: meta.lastUpdated*1000,
                            tickFormatter: function(val,axis) {
                                if (lastDataSeries.length === 0) return ' <br>';
                                var date = new Date(val);
                                return date.format('H:i');
                            }
                        }
                    };

                    var plot = $.plot($plotdiv,
                        [
                        {
                            label: '',
                            data: lastDataSeries,
                            lines: {
                                show: true
                            },
                            color: 'rgb(255, 153, 0)'
                        }
                        ]
                        , options);

                    var previousPoint = null;
                    $plotdiv.bind('plothover', function(event, pos, item) {
                        $plotdiv.css( 'cursor', 'pointer' );
                        if (item) {
                            if (previousPoint != item.datapoint) {
                                previousPoint = item.datapoint;

                                $('.vital-sign-tooltip').remove();
                                var x = item.datapoint[0],
                                y = item.datapoint[1].toFixed(2);
                                var date = new Date(x);
                                date.format('M dS H:m');
                                var content = y+'<br />'+date;
                                showTooltip(item.pageX, item.pageY, content);
                            }
                        }
                        else {
                            $('.vital-sign-tooltip').remove();
                            previousPoint = null;
                        }
                    });

                    $plotdiv.bind("plotclick", function (event, pos, item) {
                        var hostkey = $(this).data('hostkey');
                        var obs = $(this).data('obs');
                        var eventObj = {
                            'hostkey':hostkey,
                            'obs':obs
                        };
                        self._trigger("graphclicked", event, eventObj);
                    });
                    function showTooltip(x, y, contents) {
                        $('<div class="vital-sign-tooltip">' + contents + '</div>').css({
                            position: 'absolute',
                            display: 'none',
                            top: y + 5,
                            left: x + 5,
                            border: '1px solid #fdd',
                            padding: '10px',
                            color: '#000',
                            'background-color': '#fee',
                            opacity: 0.80
                        }).appendTo('body').fadeIn(100);
                    }

                    if (updateLastValue&&lastValue) {
                        $plotdiv.parent().find('.last-measured-value').html(lastValue);
                    }

                }

            };
            self.sendRequest(params);
        },

        fixHeightOfContainer: function() {
            var $self = this;
            var $parentHeight = $self.element.parent().height();
            if ($parentHeight <=0) $parentHeight = 550;
            var $headerHeight = $self.element.find('.graph-container-header-top-menu').height();
            //$self.element.find('.graph-container-body').height($parentHeight-$headerHeight - 100);


            var $totalWidth =  $self.element.find('.graph-container-body').width();
            var $totalHeight =  $self.element.find('.graph-container-body').height();
            var noOfGraphHorizontal = Math.floor($totalWidth / $self.graphWidth);
            var noOfGraphVertical = Math.ceil($totalHeight/$self.graphHeight);

            var extraWidth = $totalWidth % $self.graphWidth;
            extraWidth -= 25;
            var addWidth = 0;
            if (noOfGraphHorizontal > 0) {
                addWidth = Math.floor(extraWidth / noOfGraphHorizontal);
            }

            if (addWidth>0) {
                $self.graphWidth += addWidth;
            }

            $self.options.defaultNumberOfGraphs = noOfGraphHorizontal * noOfGraphVertical;
        },

        getDivsPerRow: function() {
             var $self = this;
            // ensure rounded corners for appropriate graph container divs
            var numGraphs = $('.graph-container').length;
            if (numGraphs) {

                var $parentWidth = $self.element.parent().width();

                var $divWidth = $('.graph-container').outerWidth(false);
                var num = Math.floor($parentWidth/$divWidth);

                $('.graph-container').removeClass().addClass('graph-container');
                if (num <= 0 || isNaN(num)) return;

                if(numGraphs<num) {
                    $('.graph-container:nth-child(1)').addClass('top-left bottom-left');
                    $('.graph-container:nth-child('+numGraphs+')').addClass('top-right bottom-right');
                } else {
                    $('.graph-container:nth-child('+num+')').addClass('top-right');

                    var remainder = numGraphs % num;

                    var bottomright = numGraphs - remainder;
                    var bottomleft = bottomright + 1;
                    if (bottomleft>numGraphs) {
                        bottomleft = bottomleft - num;
                    }
                    $('.graph-container:nth-child('+bottomleft+')').addClass('bottom-left');
                    $('.graph-container:nth-child('+bottomright+')').addClass('bottom-right');
                }

            }
        },

        drawGraphCanvas: function(data) {
            var self = this;
            self.element.find('.graph-container-header-top-menu').show();
            var slicedData = data;
            $.each(slicedData, function(key, value) {
                var meta = value.meta;
                meta['obs'] = self.selectedVital;

                // create div
                var $url = self.options.baseUrl + self.options.vitalsMainLink + meta.hostkey + '/' + self.selectedVital;
                var $hostVitalLink = $('<a class="graph-title-header" target="_blank">')
                .html(meta.hostname)
                .attr('href', $url)
                .attr('data-hostkey',meta.hostkey)
                .attr('data-obs',self.selectedVital);
                var $hostLabel = $('<div>').addClass('graph-title').append($hostVitalLink);
                var intRegex = /^\d+$/;
                var lastUpdateDate = intRegex.test(meta.lastUpdated) ? common.time.format(common.unixTimeToJavascriptTime(meta.lastUpdated)) : 'unknown';
                var average =  meta.average;
                var $lastUpdated = $('<div class="row-fluid">').html('<span class="span6 muted"> Last updated: </span><span class="span6"> ' + lastUpdateDate+'</span>');
                var $averageDiv = $('<div class="row-fluid">').html('<span class="span6 muted"> Average: </span><span class="span6"> ' + average+'</span>');
                var lastMeasuredValue = (meta.lastValue != null) ? meta.lastValue : 'not available';
                var $lastMeasuredValueDiv = $('<div class="row-fluid">').html('<span class="span6 muted">Last measured value: </span><span class="span6 last-measured-value">' + lastMeasuredValue+'</span>');
                var $graphHeader = $('<div class="graphHeader">').append($hostLabel).append($lastUpdated);
                if (self.selectedVitalSortBy == 'last-measured') {
                    $graphHeader.append($lastMeasuredValueDiv);
                } else if (self.selectedVitalSortBy == 'average') {
                    $graphHeader.append($averageDiv);
                }
                var $graphDiv = $('<div>').addClass('graph-container').attr('id', meta.hostkey + '-container').append($graphHeader);

                $graphDiv.width(self.graphWidth);
                var $plotdiv = $('<div>').attr('id', meta.hostkey).width('280').height('150')
                .attr('data-hostkey',meta.hostkey)
                .attr('data-obs',self.selectedVital)
                .attr('data-lastUpdated',meta.lastUpdated)
                .addClass('plotdiv');

                $graphDiv.append($plotdiv);
                self.element.find('.graph-container-body').append($graphDiv);

                $plotdiv.ready(function() {
                    self.plotGraph($plotdiv);
                });
            });

            self.element.find('.graph-title-header').click(function(e) {
                e.preventDefault();
                var item = $(this);
                var eventObj = {
                    'hostkey':item.attr('data-hostkey'),
                    'obs':item.attr('data-obs')
                };
                self._trigger("graphclicked", e, eventObj);
            })



        },

        drawGraphCanvasForHost: function(data) {

            var self = this;
            if (data.length !== 0) {
                var meta = {
                    'lastUpdated': data.lastUpdated,
                    'hostkey': data.hostkey
                };
                $.each(data.obs, function(key, value) {
                    // create div
                    var $url = self.options.baseUrl + self.options.vitalsMainLink + meta.hostkey + '/' + value.id;
                    var $hostVitalLink = $('<a class="graph-title-header" target="_blank">')
                    .html(value.id + '<br />' + value.description)
                    .attr('href', $url)
                    .attr('data-hostkey',meta.hostkey)
                    .attr('data-obs',value.id);
                    var $hostLabel = $('<div>').addClass('graph-title').append($hostVitalLink);
                    var intRegex = /^\d+$/;
                    var lastUpdateDate = intRegex.test(value.timestamp) ? common.time.format(common.unixTimeToJavascriptTime(value.timestamp)) : 'unknown';
                    var $lastUpdated = $('<div class="graph-last-updated row-fluid">').html('<span class="span6 muted">Last updated:</span><span class="span6"> ' + lastUpdateDate+'</span>');
                    var $unit = $('<div class="graph-unit row-fluid">').html('<span class="span6 muted"> Unit: </span><span class="span6"> ' + value.units+'</span>');
                    var lastMeasuredValue = (false) ? perfdata[perfdata.length - 1][1] : 'not available';
                    var $lastMeasuredValueDiv = $('<div class="row-fluid">').html('<span class="span6 muted">Last measured value: </span><span class="span6 last-measured-value"> ' + lastMeasuredValue+'</span>');
                    var $graphHeader = $('<div class="graphHeader">').append($hostLabel)
                    .append($lastUpdated).append($unit).append($lastMeasuredValueDiv);
                    var $graphDiv = $('<div>').attr('id', value.id + '-container').addClass('graph-container').append($graphHeader);

                    var $plotdiv = $('<div>').attr('id', value.id).width('280').height('150').addClass('plotdiv')
                    .attr('data-hostkey',meta.hostkey)
                    .attr('data-obs',value.id)
                    .attr('data-lastUpdated',meta.lastUpdated);

                    if (self.options.disableExtraButtons) {
                        $plotdiv.addClass('hostInfo');
                    }


                    $graphDiv.append($plotdiv);
                    self.element.find('.graph-container-body').append($graphDiv);
                    meta.obs = value.id;

                    $plotdiv.ready(function() {
                        self.plotGraph($plotdiv,true);
                    });
                });



                self.element.find('.graph-title-header').click(function(e) {
                    e.preventDefault();
                    var item = $(this);
                    var eventObj = {
                        'hostkey':item.attr('data-hostkey'),
                        'obs':item.attr('data-obs')
                    };
                    self._trigger("graphclicked", e, eventObj);
                })

            }
        },


        _getContext: function() {
            var self = this;
            var context = [];
            context['includes'] = encodeURIComponent(self._context.includes);
            context['excludes'] = encodeURIComponent(self._context.excludes);
            return context;
        },

        sendRequest: function(params) {
            var self = this;
            var $errorDiv = params.errorDiv || this.errorDiv;
            var senddata = $.extend(params.data, self._getContext());
            $.ajax({
                beforeSend: function() {
                    if (!params.dontShowGlobalSpinner) {
                        common.globalSpinner.show();
                    }
                },
                type: params.type ? params.type : 'post' ,
                url: params.url,
                data: senddata,
                dataType: params.dataType ? params.dataType : 'json',
                success: function(data) {
                    if ($.isFunction(params.success)) {
                        return $.call(params.success(data));
                    }
                    return data;
                },
                complete: function() {
                    if (!params.dontShowGlobalSpinner) {
                        common.globalSpinner.hide();
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    if ($.isFunction(params.error)) {
                        return $.call(params.error());
                    }
                    self.showAjaxError(jqXHR.status,$errorDiv);
                }
            });
        },
        showAjaxError: function(statusCode,targetDiv,errorMessage) {
            var self = this;
            var $errorDiv = targetDiv || this.errorDiv;
            var errorMessage = errorMessage || null;
            var errorStatusMap = {};
            if (errorMessage)
            {
                errorStatusMap[statusCode] = errorMessage;
            }
            var opt = {
                statusCode:statusCode,
                targetDiv: $errorDiv,
                errorStatusMap: errorStatusMap
            };
            common.showAjaxError(opt);
        },
        destroy: function() {
            this.element.html('');
            $.Widget.prototype.destroy.call(this);
        }

    });

    $.extend($.ui.vitalSigns, {
        instances: []
    });
})(jQuery);
