import autoprefixer from 'autoprefixer';
import fs from 'fs';
import MiniCssExtractPlugin from 'mini-css-extract-plugin';
import { createRequire } from 'module';
import path from 'path';
import TsconfigPathsPlugin from 'tsconfig-paths-webpack-plugin';
import { fileURLToPath } from 'url';
import webpack from 'webpack';
import { merge } from 'webpack-merge';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const require = createRequire(import.meta.url);

const parseEntries = dirName =>
  fs.readdirSync(path.resolve(__dirname, 'src', dirName)).reduce((entries, file) => {
    const name = path.parse(file).name;
    entries[name] = `./src/${dirName}/${file}`;
    return entries;
  }, {});

const injectedComponents = parseEntries('injectedComponents');
const bundledScripts = parseEntries('entries');

const allEntries = {
  ...bundledScripts,
  ...injectedComponents
};

const commonConfig = {
  node: {
    global: true
  },
  entry: allEntries,
  output: {
    path: path.resolve(__dirname, 'dist'),
    filename: '[name].bundle.js',
    publicPath: '/scripts/dist/',
    globalObject: 'window'
  },
  optimization: {
    splitChunks: {
      cacheGroups: {
        vendor: {
          test: /[\\/]node_modules[\\/](react|react-dom|@mui\/material)[\\/]/,
          name: 'vendor',
          chunks: 'all'
        }
      }
    }
  },
  plugins: [
    new webpack.ProvidePlugin({
      process: 'process/browser',
      Buffer: ['buffer', 'Buffer']
    }),
    new webpack.DefinePlugin({
      ENV: 'production',
      XTERM_VERSION: JSON.stringify(require('./package.json').dependencies['@xterm/xterm']),
      XTERM_FIT_VERSION: JSON.stringify(require('./package.json').dependencies['@xterm/addon-fit']),
      XTERM_SEARCH_VERSION: JSON.stringify(require('./package.json').dependencies['@xterm/addon-search'])
    }),
    new MiniCssExtractPlugin({
      filename: '[name].css',
      chunkFilename: '[id].css'
    })
  ],
  resolve: {
    extensions: ['.ts', '.tsx', '.js', '.jsx'],
    alias: {
      '@babel/runtime/helpers/esm': path.resolve(__dirname, 'node_modules/@babel/runtime/helpers/esm'),
      'dayjs/plugin/duration': path.resolve(__dirname, 'node_modules/dayjs/plugin/duration.js'),
      'dayjs/plugin/relativeTime': path.resolve(__dirname, 'node_modules/dayjs/plugin/relativeTime.js'),
      '@mui/material/styles': path.resolve(__dirname, 'node_modules/@mui/material/styles/index.js'),
      '@mui/material/useAutocomplete': path.resolve(__dirname, 'node_modules/@mui/material/useAutocomplete/index.js'),
      '@mui/x-date-pickers/DatePicker': path.resolve(__dirname, 'node_modules/@mui/x-date-pickers/DatePicker/index.js'),
      '@mui/material/ListItemText': path.resolve(__dirname, 'node_modules/@mui/material/ListItemText/index.js'),
      'validator/lib/isUUID': path.resolve(__dirname, 'node_modules/validator/lib/isUUID.js'),
      'jquery-ui': path.resolve(__dirname, 'node_modules/jquery-ui-dist/jquery-ui.min.js'),
      'notify': path.resolve(__dirname, 'notify.js'),
      '@mui/x-date-pickers': path.resolve(__dirname, 'node_modules/@mui/x-date-pickers')
    },
    fallback: {
      assert: require.resolve('assert/'),
      buffer: require.resolve('buffer/'),
      crypto: 'crypto-browserify',
      stream: require.resolve('stream-browserify'),
      util: require.resolve('util/'),
      vm: require.resolve('vm-browserify'),
      'process/browser': require.resolve('process/browser'),
      knockout: false
    },
    plugins: [
      new TsconfigPathsPlugin({
        configFile: 'tsconfig.json',
        extensions: ['.ts', '.tsx', '.js', '.jsx']
      })
    ]
  },
  module: {
    rules: [
      {
        test: /bootstrap-multiselect\.js$/,
        loader: 'esbuild-loader',
        options: {
          loader: 'js',
          define: {
            'this': 'window'
          }
        }
      },
      {
        test: /\.svg$/,
        use: ['@svgr/webpack']
      },
      {
        test: /\.(png|jpe?g|gif)$/i,
        type: 'asset/resource',
        generator: {
          filename: '[name][ext]',
          publicPath: '/scripts/dist/'
        }
      },
      {
        test: /\.module\.scss$/,
        use: [
          'style-loader',
          {
            loader: 'css-loader',
            options: {
              modules: {
                namedExport: false
              }
            }
          },
          'sass-loader'
        ]
      },
      {
        test: /\.m?[jt]sx?$/,
        exclude: [/node_modules/, /\.test\./, /__snapshots__/],
        resolve: { fullySpecified: false },
        loader: 'esbuild-loader',
        options: {
          loader: 'tsx',
          jsx: 'automatic'
        }
      },
      {
        test: /chosen\.css$/,
        use: [
          MiniCssExtractPlugin.loader,
          {
            loader: 'css-loader',
            options: {
              sourceMap: true,
              url: true
            }
          },
          {
            loader: 'postcss-loader',
            options: {
              postcssOptions: { plugins: [autoprefixer({})] },
              sourceMap: true
            }
          },
          {
            loader: 'esbuild-loader',
            options: {
              loader: 'css',
              minify: true
            }
          }
        ]
      },
      {
        test: /\.(less|css)$/,
        exclude: /chosen\.css$/,
        use: [
          MiniCssExtractPlugin.loader,
          {
            loader: 'css-loader',
            options: {
              sourceMap: true,
              url: false
            }
          },
          {
            loader: 'postcss-loader',
            options: {
              postcssOptions: { plugins: [autoprefixer({})] },
              sourceMap: true
            }
          },
          {
            loader: 'esbuild-loader',
            options: {
              loader: 'css',
              minify: true
            }
          },
          'less-loader'
        ]
      },
      {
        test: /\.tsx?$/,
        use: {
          loader: 'ts-loader',
          options: {
            transpileOnly: true // Speeds up builds by skipping type checks
          }
        },
        exclude: /node_modules/
      },
      {
        test: /\.s[ac]ss$/i,
        exclude: /\.module\.scss$/,
        use: ['style-loader', 'css-loader', 'sass-loader']
      }
    ]
  }
};
const devConfig = {
  mode: 'development',
  devtool: 'source-map',
  watch: true
};

const prodConfig = {
  mode: 'production',
  optimization: {
    minimize: true
  }
};

export default (env, argv) => {
  const isProduction = argv.mode === 'production';
  if (!isProduction) {
    console.log('React Components:', Object.keys(injectedComponents));
    console.log('Bundled scripts:', Object.keys(bundledScripts));
  }
  const config = isProduction ? prodConfig : devConfig;
  return merge(commonConfig, config);
};
