// @ts-nocheck
import Api from '@northern.tech/store/api/general-api';
import { SORTING_OPTIONS, headerNames } from '@northern.tech/store/constants';
import { actions, sliceName } from '@northern.tech/store/organizationSlice/index';
import { getAuditlogState } from '@northern.tech/store/organizationSlice/selectors';
import { dateRangeToUnix, deepCompare } from '@northern.tech/utils/helpers';
import { createAsyncThunk } from '@reduxjs/toolkit';

import { auditLogApiUrl } from '../constants';

const prepareAuditlogQuery = ({ startDate, endDate, user: userFilter, type, detail: detailFilter, sort = {} }) => {
  const params = new URLSearchParams();
  const userId = userFilter?.id || userFilter;
  const detail = detailFilter?.id || detailFilter;

  const { start: unixStart, end: endUnix } = dateRangeToUnix(startDate, endDate);
  if (startDate) {
    params.append('created_after', unixStart);
  }

  if (endDate) {
    params.append('created_before', endUnix);
  }

  if (userId) {
    params.append('actor_id', userId);
  }

  if (type) {
    params.append('object_type', type.value);
  }

  if (detail) {
    params.append('object_name', detail);
  }

  const { direction = SORTING_OPTIONS.desc } = sort;
  params.append('sort', direction);

  return params.toString();
};

export const getAuditLogs = createAsyncThunk(`${sliceName}/getAuditLogs`, (selectionState, { dispatch }) => {
  const { page, perPage } = selectionState;
  return (
    Api.get(`${auditLogApiUrl}/logs?page=${page}&per_page=${perPage}&${prepareAuditlogQuery(selectionState)}`)
      .then(({ data, headers }) => {
        const total = Number(headers[headerNames.total]) || data.length;
        return Promise.resolve(dispatch(actions.receiveAuditLogs({ events: data, total })));
      })
      // eslint-disable-next-line no-undef
      .catch(() => notify.error(`There was an error retrieving audit logs`))
  );
});

export const getAuditLog = createAsyncThunk(`${sliceName}/getAuditLog`, (id, { dispatch }) => {
  return (
    Api.get(`${auditLogApiUrl}/log/${id}`)
      .then(({ data }) => {
        return Promise.resolve(dispatch(actions.setAuditLogState({ selectedAuditLog: data })));
      })
      // eslint-disable-next-line no-undef
      .catch(() => notify.error(`There was an error retrieving audit logs`))
  );
});

export const getAuditLogsCsvLink = createAsyncThunk(
  `${sliceName}/getAuditLogsCsvLink`,
  (_, { getState }) => `${auditLogApiUrl}/logs_export?limit=20000&${prepareAuditlogQuery(getAuditlogState(getState()))}`
);

export const setAuditlogsState = createAsyncThunk(`${sliceName}/setAuditlogsState`, (selectionState, { dispatch, getState }) => {
  const { startDate, endDate } = selectionState;
  const { start: unixStart, end: endUnix } = dateRangeToUnix(startDate, endDate);
  if ((startDate && unixStart < 1) || (endDate && endUnix < 1)) {
    return;
  }

  const currentState = getAuditlogState(getState());
  let nextState = {
    ...currentState,
    ...selectionState,
    sort: { ...currentState.sort, ...selectionState.sort }
  };
  let tasks = [];
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  const { isLoading: currentLoading = false, selectedIssue: currentIssue, ...currentRequestState } = currentState;
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  const { isLoading: selectionLoading = false, selectedIssue: selectionIssue, ...selectionRequestState } = nextState;
  if (!deepCompare(currentRequestState, selectionRequestState)) {
    nextState.isLoading = true;
    tasks.push(dispatch(getAuditLogs(nextState)).finally(() => dispatch(actions.setAuditLogState({ isLoading: false }))));
  }
  tasks.push(dispatch(actions.setAuditLogState(nextState)));
  return Promise.all(tasks);
});

export const getAuditLogNameByType = createAsyncThunk(`${sliceName}/getAuditLog`, (type, { dispatch }) => {
  return (
    Api.get(`${auditLogApiUrl}/names/${type}`)
      .then(({ data = [] }) => Promise.resolve(dispatch(actions.setAuditLogState({ detailsOptions: data.filter(item => !!item) }))))
      // eslint-disable-next-line no-undef
      .catch(() => notify.error(`There was an error retrieving audit logs`))
  );
});
